/**
 * Copyright (c) 2002 The JA-SIG Collaborative.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by the JA-SIG Collaborative
 *    (http://www.jasig.org/)."
 *
 * THIS SOFTWARE IS PROVIDED BY THE JA-SIG COLLABORATIVE "AS IS" AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE JA-SIG COLLABORATIVE OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

package org.jasig.portal.channels.bookmarks;

import org.apache.xerces.dom.DocumentImpl;
import org.apache.xerces.parsers.DOMParser;
import org.apache.xml.serialize.OutputFormat;
import org.apache.xml.serialize.XMLSerializer;
import org.jasig.portal.RDBMServices;
import org.jasig.portal.channels.BaseChannel;
import org.jasig.portal.security.IPerson;
import org.jasig.portal.services.LogService;
import org.jasig.portal.utils.DTDResolver;
import org.jasig.portal.utils.XSLT;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;
import org.xml.sax.ContentHandler;
import org.xml.sax.InputSource;

import java.io.*;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.Enumeration;
import java.util.Hashtable;

/**
 * Bookmarks channel
 * @author Peter Kharchenko
 * @author Steven Toth
 * @author Bernie Durfee
 * @author Mark Mazelin, MarkMazelin@cedarville.edu
 * @author <a href="smb1@cornell.edu?subject=CBookmarks">Steve Barrett</a>
 * @version $Revision: 1.6 $
 * <hr>
 *
 * The ability to specify link targets has been sorely needed for quite some time.
 * The easiest approach is to be able to specify the target for all links within
 * the bookmarks channel.  The best approach would be to allow the user specify
 * the target per link.  In the interest of time, I have implemented the easiest
 * approach.
 * <br><br>
 * The ability to specify target links works as follows:
 * <br><br>
 * <ol>
 *  <li>If you do nothing, the bookmark links function as they always have.
 *  <li>If you edit the channel definition to include a parameter named
 *   "target.window", you may specify the links target (_blank for new windows
 *    and something else (non-blank) for named windows).
 *  <li>If you would like to give the user the ability to set the target value themselves you can:
 *   <ol>
 *    <li>Allow them to edit the "target.window" channel definition parameter (this has issues)
 *    <li>Define the bookmarks channel in the UP_CHAN_TYPE table and point to the
 *     CBookmarks.cpd file.  This will list the Bookmarks channel as a "type" of
 *     channel in the channel publishing channel which may not be what you want.
 *     It will, however, present users with canned options to choose from for
 *     setting the links target.  These options are selectable from a pulldown
 *     select box.  This is a far less than elegant solution but defining radio
 *     button within a .cpd appears to be impossible for flawed.
 *   </ol>
 * </ol>
 * <hr>
 */
public class CBookmarks extends BaseChannel {

  /** Target window parameter key value = "target.window". @author smb1 @since 2003.06.18 */
 /** protected static final String TARGETWINDOW = "target.window"; */
	protected static final String TARGETWINDOW = "_blank";

  /** Target window parameter value = "". @author smb1 @since 2003.06.18 */
  protected static final String TARGETWINDOWNONE = "";

  // A DOM document where all the bookmark information will be contained
  protected DocumentImpl m_bookmarksXML;
  // Location of the stylesheet list file
  private static final String sslLocation = "CBookmarks.ssl";
  // Define some constants to keep the state of the channel
  private final int VIEWMODE = 0;
  private final int EDITMODE = 1;
  private final int EDITCHOOSENODEMODE = 2;
  private final int EDITNODEMODE = 3;
  private final int ADDNODEMODE = 4;
  private final int MOVENODEMODE = 5;
  private final int DELETENODEMODE = 6;
  // Start out in view mode by default
  private int m_currentState = 0;
  // Keep track of the node that the user is currently working with
  private String m_activeNodeID = null;
  private String m_activeNodeType = null;
  private static final boolean printXML = false;

  private DocumentImpl getBookmarkXML () {
    Connection connection = null;
    // Return the cached bookmarks if they've already been read in
    if (m_bookmarksXML != null) {
      return  (m_bookmarksXML);
    }
    try {
		 String inputXML = null;
       // Create a new parser for the incoming bookmarks document
		 DOMParser domParser = new DOMParser();
		 // Get a connection to the database
		 connection = getConnection();
		 // Get the current user's ID
		 int userid = staticData.getPerson().getID();
		 // Attempt to retrieve the user's bookmark's
		 String query = "SELECT BOOKMARK_XML, PORTAL_USER_ID FROM UPC_BOOKMARKS WHERE PORTAL_USER_ID=" + userid;


		 //replaced statement by prepared statement - Luis Mendes
		 PreparedStatement ps = connection.prepareStatement(query);

		 try {
        // Get the result set
			 ResultSet rs = ps.executeQuery();
			 try {
				 if (rs.next()) {
					 // If a result came back then use that for the XML...
					 BufferedReader br = new BufferedReader(
 						 rs.getCharacterStream(1));
					  StringBuffer tmpBuff = new StringBuffer();
					  String tmp = null;
					  while((tmp = br.readLine()) != null){
						  tmpBuff.append(tmp);
					  }
					  inputXML = tmpBuff.toString();
					  br.close();
				 }
			 } finally {
          rs.close();
        }
      } finally {
			ps.close();
			//stmt.close();
      }
      if (inputXML == null || inputXML.length() == 0) {
        // ...or else use the bookmarks from the default user
        inputXML = getDefaultBookmarks();
      }
      // Turn validation on for the DOM parser to make sure it reads the DTD
      domParser.setFeature("http://xml.org/sax/features/validation", true);
      DTDResolver dtdResolver = new DTDResolver("xbel-1.0.dtd");
      domParser.setEntityResolver(dtdResolver);
      //domParser.setEntityResolver(new org.jasig.portal.utils.DTDResolver("xbel-1.0.dtd"));
      // Parse the XML document containing the user's bookmarks
      if (printXML) {
         LogService.instance().log(LogService.DEBUG, "CBookmarks::transformXML() : XML incoming :\n\n" + inputXML + "\n\n");
      }
      domParser.parse(new InputSource(new StringReader(inputXML)));
      // Cache the bookmarks DOM locally
      m_bookmarksXML = (DocumentImpl)domParser.getDocument();
    } catch (Exception e) {
      LogService.instance().log(LogService.ERROR, e);
    } finally {
      // Release the database connection
      if (connection != null) {
        releaseConnection(connection);
      }
    }
    // Return what is cached
    return  (m_bookmarksXML);
  }


  private String getDefaultBookmarks () {
    Connection connection = null;
    String inputXML = null;
    int defaultuid = 2;
    ResultSet rs = null;

    try {
      // Get a connection to the database
      connection = getConnection();
      // Get the bookmarks for the 'default' user
      IPerson person = this.staticData.getPerson();
      String defaultUser = (String)person.getAttribute("uPortalTemplateUserName");
      if (defaultUser == null || defaultUser.equals("")) {
        defaultUser = "system";
      }
      String query = "SELECT USER_ID FROM UP_USER WHERE USER_NAME = '" + defaultUser + "'";
      Statement stmt = connection.createStatement();
      try {
         rs = stmt.executeQuery(query);
         if (rs.next()) {
            defaultuid = rs.getInt("USER_ID");
         } else {
            LogService.instance().log(LogService.WARN, "CBookmarks.getDefaultBookmarks(): Could not get userid  for 'default' user");
         }
      } finally {
          rs.close();
      }

      query = "SELECT BOOKMARK_XML, PORTAL_USER_ID FROM UPC_BOOKMARKS WHERE PORTAL_USER_ID = '" + defaultuid + "'";

      try {
        // Try to get the 'default' bookmarks from the database
        rs = stmt.executeQuery(query);
        try {
          if (rs.next()) {
            // Use the 'default' user's bookmarks...
          inputXML = rs.getString("BOOKMARK_XML");
          }
          else {
            // Generate the XML here as a last resort

            StringBuffer strbuf = new StringBuffer();
            strbuf.append("<?xml version=\"1.0\"?>\n");
            strbuf.append("<!DOCTYPE xbel PUBLIC \"+//IDN python.org//DTD XML Bookmark Exchange Language 1.0//EN//XML\" \"http://www.python.org/topics/xml/dtds/xbel-1.0.dtd\">\n");
            strbuf.append("<xbel>\n");
            strbuf.append("  <title>Default Bookmarks</title>\n");
            strbuf.append("  <info>\n");
            strbuf.append("    <metadata owner=\"" + staticData.getPerson().getID() + "\"/>\n");
            strbuf.append("  </info>\n");
            strbuf.append("</xbel>");
            inputXML = strbuf.toString();

            LogService.instance().log(LogService.WARN, "CBookmarks.getDefaultBookmarks(): Could not find bookmarks for 'default' user");
          }
        } finally {
          rs.close();
        }
        // Now add a row to the database for the user
        String insert = "INSERT INTO UPC_BOOKMARKS (PORTAL_USER_ID, BOOKMARK_XML) VALUES (" + staticData.getPerson().getID()
            + ",'" + inputXML + "')";
        //LogService.instance().log(LogService.DEBUG, insert);
        stmt.executeUpdate(insert);
      } finally {
        stmt.close();
      }
    } catch (Exception e) {
      LogService.instance().log(LogService.ERROR, e);
    } finally {
      if (connection != null) {
        releaseConnection(connection);
      }
      if (inputXML == null) {
        // ...or else just start with an empty set of bookmarks
        LogService.instance().log(LogService.ERROR, "CBookmarks.getDefaultBookmarks() - Could not retrieve default bookmark xml, using blank xml.");
        inputXML = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><xbel></xbel>";
      }
    }
    return  (inputXML);
  }

  protected void saveXML () {
    if (m_bookmarksXML == null) {
      return;
    }
    Connection connection = getConnection();
    try {
      StringWriter stringWriter = new StringWriter();
      // Serialize the DOM tree to a string
      XMLSerializer xmlSerializer = new XMLSerializer(stringWriter, new OutputFormat(m_bookmarksXML));
      xmlSerializer.serialize(m_bookmarksXML);
      // Get a connection to the database


		//replace the previous statement - Luis Mendes - felipeen@udel.edu
		String update = "UPDATE UPC_BOOKMARKS SET BOOKMARK_XML = ?" +
			" WHERE PORTAL_USER_ID = "	+ staticData.getPerson().getID();
		PreparedStatement	ps = connection.prepareStatement(update);

      try {

  			byte[] buffer = stringWriter.toString().getBytes();
 			StringReader reader=new StringReader(stringWriter.toString());
 			ps.setCharacterStream(1, reader, buffer.length);
			ps.executeUpdate();
			ps.close();

      } finally {
        ps.close();
      }
    } catch (Exception e) {
      LogService.instance().log(LogService.ERROR, e);
    } finally {
      releaseConnection(connection);
    }
  }

  /**
   *  Render the user's layout based on the current state of the channel
   */
  public void renderXML (ContentHandler out) {
    // Retrieve the command passed in by the user
    String command = runtimeData.getParameter("command");
    if (command != null) {
      if (command.equals("fold")) {
        // Get the ID of the specified folder to close
        String folderID = runtimeData.getParameter("ID");
        if (folderID != null) {
          Element folderElement = getBookmarkXML().getElementById(folderID);
          if (folderElement != null && folderElement.getNodeName().equals("folder")) {
            folderElement.setAttribute("folded", "yes");
          }
        }
      }
      else if (command.equals("unfold")) {
        // Get the ID of the specified folder to open
        String folderID = runtimeData.getParameter("ID");
        if (folderID != null) {
          Element folderElement = getBookmarkXML().getElementById(folderID);
          if (folderElement != null && folderElement.getNodeName().equals("folder")) {
            folderElement.setAttribute("folded", "no");
          }
        }
      }
      else if (command.equals("View")) {
        // Switch to view mode
        m_currentState = VIEWMODE;
        m_activeNodeID = null;
      }
      // 30-DEC-2002-MBM: added "save" command (also needed to add link and icon to view_mode_html.xsl)
      else if (command.equals("Save")) {
        // Save the user's XML
        saveXML();
        // Switch to view mode
        m_currentState = VIEWMODE;
        m_activeNodeID = null;
      }
      else if (command.equals("Edit")) {
        // Switch to edit mode
        m_currentState = EDITMODE;
        m_activeNodeID = null;
      }
      // 3-DEB-2003-MBM: added "editbookmark" command
      else if (command.equals("EditBookmark")) {
        if ((m_currentState != EDITNODEMODE) && (m_currentState != EDITCHOOSENODEMODE)) {
          m_currentState = EDITCHOOSENODEMODE;
          m_activeNodeID = null;
          m_activeNodeType = "bookmark";
        } else {
          String submitButton = runtimeData.getParameter("SubmitButton");
          if (submitButton != null && submitButton.equals("Cancel")) {
            // The user pressed the cancel button so return to view mode
            m_activeNodeID = null;
            m_activeNodeType = null;
            m_currentState = VIEWMODE;
          }
          else if (submitButton != null && submitButton.equals("Edit Bookmark")) {
            // Node has been chosen
            // Check for the incoming parameters
            String bookmarkID = runtimeData.getParameter("BookmarkRadioButton");
            // set activeNodeID here so we can use it later
            m_activeNodeID = bookmarkID;
            m_currentState = EDITNODEMODE;
          }
          else if (submitButton != null && submitButton.equals("Update")) {
            // Node has been edited
            // Check for the incoming parameters
            String bookmarkID = runtimeData.getParameter("BookmarkID");
            String bookmarkTitle = runtimeData.getParameter("BookmarkTitle");
            String bookmarkURL = runtimeData.getParameter("BookmarkURL");
            String bookmarkDesc = runtimeData.getParameter("BookmarkDescription");
            if (bookmarkTitle == null || bookmarkTitle.length() < 1) {}
            else if (bookmarkURL == null || bookmarkURL.length() < 1) {}
            else if (bookmarkID == null || bookmarkID.length() < 1) {}
            else {
              // Build the bookmark XML DOM
              Element bookmarkElement = m_bookmarksXML.getElementById(bookmarkID);
              if (bookmarkElement != null && bookmarkElement.getNodeName().equals(m_activeNodeType)) {
                bookmarkElement.setAttribute("href", bookmarkURL);
                setElementChildValue(bookmarkElement, "title", bookmarkTitle);
                setElementChildValue(bookmarkElement, "desc", bookmarkDesc);
                // Save the user's XML
                saveXML();
              }
              // Return to view mode
              m_activeNodeID = null;
              m_activeNodeType = null;
              m_currentState = VIEWMODE;
            }
          }
        }
      }
      // 5-DEB-2003-MBM: added "editfolder" command
      else if (command.equals("EditFolder")) {
        if ((m_currentState != EDITNODEMODE) && (m_currentState != EDITCHOOSENODEMODE)) {
          m_currentState = EDITCHOOSENODEMODE;
          m_activeNodeID = null;
          m_activeNodeType = "folder";
        } else {
          String submitButton = runtimeData.getParameter("SubmitButton");
          if (submitButton != null && submitButton.equals("Cancel")) {
            // The user pressed the cancel button so return to view mode
            m_activeNodeID = null;
            m_activeNodeType = null;
            m_currentState = VIEWMODE;
          }
          else if (submitButton != null && submitButton.equals("Edit Folder")) {
            // Node has been chosen
            // Check for the incoming parameters
            String folderID = runtimeData.getParameter("FolderRadioButton");
            // set activeNodeID here so we can use it later
            m_activeNodeID = folderID;
            m_currentState = EDITNODEMODE;
          }
          else if (submitButton != null && submitButton.equals("Update")) {
            // Node has been edited
            // Check for the incoming parameters
            String folderID = runtimeData.getParameter("FolderID");
            String folderTitle = runtimeData.getParameter("FolderTitle");
            if (folderTitle == null || folderTitle.length() < 1) {}
            else if (folderID == null || folderID.length() < 1) {}
            else {
              // Build the folder XML DOM
              Element folderElement = m_bookmarksXML.getElementById(folderID);
              if (folderElement != null && folderElement.getNodeName().equals(m_activeNodeType)) {
                setElementChildValue(folderElement, "title", folderTitle);
                // Save the user's XML
                saveXML();
              }
              // Return to view mode
              m_activeNodeID = null;
              m_activeNodeType = null;
              m_currentState = VIEWMODE;
            }
          }
        }
      }
      else if (command.equals("AddBookmark")) {
        if (m_currentState != ADDNODEMODE) {
          // Switch to add bookmark mode
          m_currentState = ADDNODEMODE;
          m_activeNodeID = null;
          m_activeNodeType = "bookmark";
        }
        else {
          String submitButton = runtimeData.getParameter("SubmitButton");
          if (submitButton != null && submitButton.equals("Cancel")) {
            // The user pressed the cancel button so return to view mode
            m_activeNodeID = null;
            m_activeNodeType = null;
            m_currentState = VIEWMODE;
          }
          else if (submitButton != null && submitButton.equals("Add")) {
            // Check for the incoming parameters
            String bookmarkTitle = runtimeData.getParameter("BookmarkTitle");
            String bookmarkURL = runtimeData.getParameter("BookmarkURL");
            String bookmarkDesc = runtimeData.getParameter("BookmarkDescription");
            String folderID = runtimeData.getParameter("FolderRadioButton");
            if (bookmarkTitle == null || bookmarkTitle.length() < 1) {}
            else if (bookmarkURL == null || bookmarkURL.length() < 1) {}
            else if (folderID == null || folderID.length() < 1) {}
            else {
              Element folderElement;
              if (folderID.equals("RootLevel")) {
                folderElement = (Element)m_bookmarksXML.getElementsByTagName("xbel").item(0);
              }
              else {
                folderElement = m_bookmarksXML.getElementById(folderID);
              }
              if (folderElement == null) {}
              else {
                // Build the bookmark XML DOM
                Element bookmarkElement = m_bookmarksXML.createElement("bookmark");
                bookmarkElement.setAttribute("href", bookmarkURL);
                bookmarkElement.setAttribute("id", createUniqueID());
                // Create the title element
                Element titleElement = m_bookmarksXML.createElement("title");
                titleElement.appendChild(m_bookmarksXML.createTextNode(bookmarkTitle));
                bookmarkElement.appendChild(titleElement);
                // Create the desc element
                Element descElement = m_bookmarksXML.createElement("desc");
                descElement.appendChild(m_bookmarksXML.createTextNode(bookmarkDesc));
                bookmarkElement.appendChild(descElement);
                folderElement.appendChild(bookmarkElement);
                // Notify the DOM of the new ID
                m_bookmarksXML.putIdentifier(bookmarkElement.getAttribute("id"), bookmarkElement);
                // The user pressed the cancel button so return to view mode
                m_activeNodeID = null;
                m_activeNodeType = null;
                // Return to view mode
                m_currentState = VIEWMODE;
                // Save the user's XML
                saveXML();
              }
            }
          }
        }
      }
      else if (command.equals("AddFolder")) {
        if (m_currentState != ADDNODEMODE) {
          // Switch to add bookmark mode
          m_currentState = ADDNODEMODE;
          m_activeNodeID = null;
          m_activeNodeType = "folder";
        }
        else {
          String submitButton = runtimeData.getParameter("SubmitButton");
          if (submitButton != null && submitButton.equals("Cancel")) {
            // The user pressed the cancel button so return to view mode
            m_activeNodeID = null;
            m_activeNodeType = null;
            m_currentState = VIEWMODE;
          }
          else if (submitButton != null && submitButton.equals("Add")) {
            // Check for the incoming parameters
            String folderTitle = runtimeData.getParameter("FolderTitle");
            String folderID = runtimeData.getParameter("FolderRadioButton");
            if (folderTitle == null || folderTitle.length() < 1) {}
            else if (folderID == null || folderID.length() < 1) {}
            else {
              Element folderElement;
              if (folderID.equals("RootLevel")) {
                folderElement = (Element)m_bookmarksXML.getElementsByTagName("xbel").item(0);
              }
              else {
                folderElement = m_bookmarksXML.getElementById(folderID);
              }
              if (folderElement == null) {}
              else {
                // Build the new folder XML node
                Element newFolderElement = m_bookmarksXML.createElement("folder");
                newFolderElement.setAttribute("id", createUniqueID());
                // Create the title element
                Element titleElement = m_bookmarksXML.createElement("title");
                titleElement.appendChild(m_bookmarksXML.createTextNode(folderTitle));
                newFolderElement.appendChild(titleElement);
                folderElement.appendChild(newFolderElement);
                // Notify the DOM of the new ID
                m_bookmarksXML.putIdentifier(newFolderElement.getAttribute("id"), newFolderElement);
                // The user pressed the cancel button so return to view mode
                m_activeNodeID = null;
                m_activeNodeType = null;
                // Return to view mode
                m_currentState = VIEWMODE;
                // Save the user's XML
                saveXML();
              }
            }
          }
        }
      }
      else if (command.equals("MoveBookmark")) {
       
       		if (m_currentState !=MOVENODEMODE) {
        	m_currentState = MOVENODEMODE;
        	m_activeNodeID = null;
        	m_activeNodeType = "bookmark";	
			}
       		else {
       	     	String submitButton = runtimeData.getParameter("SubmitButton");
          		if (submitButton != null) {
            		if (submitButton.equals("Cancel")) {
              		m_currentState = VIEWMODE;
              		m_activeNodeType = null;
            		}else if (submitButton.equals("Move")) {
            			Element destFolder;
            		
            			String bookmarkToMove = runtimeData.getParameter("BookmarkToMove");	
            			
            		    String position = runtimeData.getParameter("Position");	
            		    
            		    
            		    if (position!=null && bookmarkToMove!=null){   	
            		    	
            			Element bookmarkElement = m_bookmarksXML.getElementById(bookmarkToMove); 
            			//dplacement en derniere position d'un repertoire 
            			if (position.substring(position.length()-1 ,position.length()).equals("f")) {
            				destFolder= m_bookmarksXML.getElementById(position.substring(0 ,position.length()-1));
            				destFolder.appendChild(bookmarkElement);
            			}else {
            				           			     			
            			//bookmark  la place duquel on veut mettre le bookmark dplac           		 
            			String testFolder = position.substring(0,position.length()-1);  
            			Element bookDestFolder = m_bookmarksXML.getElementById(testFolder);
            			destFolder =(Element) bookDestFolder.getParentNode();
            			if (destFolder.getNodeName().equals("xbel") ){
            				destFolder= (Element)m_bookmarksXML.getElementsByTagName("xbel").item(0);
            			}	            			
            			if (position.substring(position.length()-1 ,position.length()).equals("e")){
            				destFolder.appendChild(bookmarkElement);
            			}else{
    					destFolder.insertBefore(bookmarkElement,bookDestFolder);
            					} 
            							
        				}	      					
             			saveXML();	
            		}
            		  
             			m_currentState = VIEWMODE;
             			m_activeNodeType = null;
            	
            	}// fin cas "move"	
            }// fin test bouton
        }// fin test du mode
     } //fin move
     
     else if (command.equals("MoveFolder")) {
      
       		if (m_currentState !=MOVENODEMODE) {
        	m_currentState = MOVENODEMODE;
        	m_activeNodeID = null;
        	m_activeNodeType ="folder";
			}
       		else {
       	     	String submitButton = runtimeData.getParameter("SubmitButton");
          		if (submitButton != null) {
            		if (submitButton.equals("Cancel")) {
              		m_currentState = VIEWMODE;
              		m_activeNodeType = null;
            		}else if (submitButton.equals("Move")) {
            			Element destFolder;	
            			String folderToMove = runtimeData.getParameter("FolderToMove");	
            		   	String position = runtimeData.getParameter("Position");	
            		   	if (position!=null && folderToMove!=null){
            		   	Element folderElement = m_bookmarksXML.getElementById(folderToMove); 
    					
    					
    					String testFolder = position.substring(0,position.length()-1); 
    					
    					if (!testFolder.equals(folderToMove)){ 
            			Element folderDestFolder = m_bookmarksXML.getElementById(testFolder);
    						
            			Element parent = (Element) folderDestFolder.getParentNode();
            			String parentId=parent.getAttribute("id");
            			boolean testParente = parentId.equals(folderToMove) ;
            			boolean stop=(false ||  parent.getNodeName().equals("xbel"));
            			while (!testParente && !stop)
            			{
            				parent = (Element) parent.getParentNode();
            				parentId=parent.getAttribute("id");
            				if(parentId.equals("")){
            				
            					stop=true;
            				}else {
            			 		testParente = parentId.equals(folderToMove);
            			 	
            		 		}
            		    } 
            			
            			if (stop) {
            			
            			 	if (position.substring(position.length()-1 ,position.length()).equals("f")) {
            		 			destFolder= m_bookmarksXML.getElementById(position.substring(0 ,position.length()-1));
            					destFolder.appendChild(folderElement);
            				} else {
            				   		destFolder =(Element) folderDestFolder.getParentNode();
            						if (destFolder.getNodeName().equals("xbel") ){
            							destFolder= (Element)m_bookmarksXML.getElementsByTagName("xbel").item(0);
            						}	            			
            						if (position.substring(position.length()-1 ,position.length()).equals("e")){
            							destFolder.appendChild(folderElement);
            						}else{
    									destFolder.insertBefore(folderElement,folderDestFolder);
            						} 
            				}
            			}else {
            				
            				}
            		 
            		 	
            			saveXML();	
            		
            		}
            	}
             			m_currentState = VIEWMODE;
             			m_activeNodeType = null;
             		
         
            	
            	}// fin cas "move"	
            }// fin test bouton
        }// fin test du mode
        
               
     } //fin move
     
      else if (command.equals("DeleteBookmark")) {
        if (m_currentState != DELETENODEMODE) {
          m_currentState = DELETENODEMODE;
          m_activeNodeType = "bookmark";
        }
        else {
          String submitButton = runtimeData.getParameter("SubmitButton");
          if (submitButton != null) {
            if (submitButton.equals("Cancel")) {
              m_currentState = VIEWMODE;
              m_activeNodeType = null;
            }
            else if (submitButton.equals("Delete")) {
              // Run through the passed in parameters and delete the bookmarks
              Enumeration e = runtimeData.keys();
              while (e.hasMoreElements()) {
                String key = (String)e.nextElement();
                if (key.startsWith("BookmarkCheckbox#")) {
                  String bookmarkID = key.substring(17);
                  Element bookmarkElement = m_bookmarksXML.getElementById(bookmarkID);
                  if (bookmarkElement != null && bookmarkElement.getNodeName().equals("bookmark")) {
                    bookmarkElement.getParentNode().removeChild(bookmarkElement);
                  }
                }
              }
              saveXML();
              m_currentState = VIEWMODE;
              m_activeNodeType = null;
            }
            else if (submitButton.equals("ConfirmDelete")) {}
          }
        }
      }
      else if (command.equals("DeleteFolder")) {
        if (m_currentState != DELETENODEMODE) {
          m_currentState = DELETENODEMODE;
          m_activeNodeType = "folder";
        }
        else {
          String submitButton = runtimeData.getParameter("SubmitButton");
          if (submitButton != null) {
            if (submitButton.equals("Cancel")) {
              m_currentState = VIEWMODE;
              m_activeNodeType = null;
            }
            else if (submitButton.equals("Delete")) {
              // Run through the passed in parameters and delete the bookmarks
              Enumeration e = runtimeData.keys();
              while (e.hasMoreElements()) {
                String key = (String)e.nextElement();
                if (key.startsWith("FolderCheckbox#")) {
                  // The ID should come after the FolderCheckbox# part
                  String bookmarkID = key.substring(15);
                  // Find the folder in the DOM tree
                  Element folderElement = m_bookmarksXML.getElementById(bookmarkID);
                  // Remove the folder from the DOM tree
                  if (folderElement != null && folderElement.getNodeName().equals("folder")) {
                    folderElement.getParentNode().removeChild(folderElement);
                  }
                }
              }
              saveXML();
              m_currentState = VIEWMODE;
              m_activeNodeType = null;
            }
            else if (submitButton.equals("ConfirmDelete")) {}
          }
        }
      }
    }
    try {
      // Render content based on the current state of the channel
      switch (m_currentState) {
        case VIEWMODE:
          renderViewModeXML(out);
          break;
        case EDITCHOOSENODEMODE:
          renderEditChooseNodeXML(out);
          break;
        case EDITNODEMODE:
          renderEditNodeXML(out);
          break;
        case ADDNODEMODE:
          renderAddNodeXML(out);
          break;
        case MOVENODEMODE:
          renderMoveNodeXML(out);
          break;
        case DELETENODEMODE:
          renderDeleteNodeXML(out);
          break;
      }
    } catch (Exception e) {
      LogService.instance().log(LogService.ERROR, e);
    }
  }


  private void renderViewModeXML (ContentHandler out) throws Exception {
    transformXML(out, "view_mode", getBookmarkXML());
  }

  /**
   * renderEditChooseNodeXML [03-FEB-2003 Mark Mazelin, MarkMazelin@cedarville.edu]
   * Used to display the bookmark tree to select the node for editing bookmarks or folders.
   * @param out
   * @throws Exception
   */
  private void renderEditChooseNodeXML (ContentHandler out) throws Exception {
    Hashtable parameters = new Hashtable(1);
    if (m_activeNodeType == null) {
      LogService.instance().log(LogService.ERROR, "CBookmarks.renderEditChooseNodeXML: No active node type has been set");
      renderViewModeXML(out);
    }
    else if (m_activeNodeType.equals("bookmark")) {
      parameters.put("EditMode", "EditBookmark");
      transformXML(out, "edit_choose_node", getBookmarkXML(), parameters);
    }
    else if (m_activeNodeType.equals("folder")) {
      parameters.put("EditMode", "EditFolder");
      transformXML(out, "edit_choose_node", getBookmarkXML(), parameters);
    }
    else {
      LogService.instance().log(LogService.ERROR, "CBookmarks.renderEditChooseNodeXML: Unknown active node type - " + m_activeNodeType);
      renderViewModeXML(out);
    }
  }

  /**
   * renderEditNodeXML [03-FEB-2003 Mark Mazelin, MarkMazelin@cedarville.edu]
   * Used to display edit boxes for editing bookmarks or folders.
   * @param out the ContentHandler
   * @throws Exception
   */
  private void renderEditNodeXML (ContentHandler out) throws Exception {
    String bookmarkTitle = "";
    String bookmarkURL = "";
    String bookmarkDesc = "";
    String folderTitle = "";
    Hashtable parameters = new Hashtable(4);
    if (m_activeNodeType == null) {
      LogService.instance().log(LogService.ERROR, "CBookmarks.renderEditNodeXML: No active node type has been set");
      renderViewModeXML(out);
    }
    else if (m_activeNodeType.equals("bookmark")) {
      // lookup activeNodeID in the XML structure to get title, etc.
      Element bookmarkElement = m_bookmarksXML.getElementById(m_activeNodeID);
      if (bookmarkElement != null && bookmarkElement.getNodeName().equals(m_activeNodeType)) {
        bookmarkURL = bookmarkElement.getAttribute("href");
        bookmarkTitle = getElementChildValue(bookmarkElement, "title");
        bookmarkDesc = getElementChildValue(bookmarkElement, "desc");
      }
      parameters.put("EditMode", "EditBookmark");
      parameters.put("BookmarkID", m_activeNodeID);
      parameters.put("BookmarkURL", bookmarkURL);
      parameters.put("BookmarkTitle", bookmarkTitle);
      parameters.put("BookmarkDescription", bookmarkDesc);
      transformXML(out, "edit_node", getBookmarkXML(), parameters);
    }
    else if (m_activeNodeType.equals("folder")) {
      // lookup activeNodeID in the XML structure to get title, etc.
      Element folderElement = m_bookmarksXML.getElementById(m_activeNodeID);
      if (folderElement != null && folderElement.getNodeName().equals(m_activeNodeType)) {
        folderTitle = getElementChildValue(folderElement, "title");
      }
      parameters.put("EditMode", "EditFolder");
      parameters.put("FolderID", m_activeNodeID);
      parameters.put("FolderTitle", folderTitle);
      transformXML(out, "edit_node", getBookmarkXML(), parameters);
    }
    else {
      LogService.instance().log(LogService.ERROR, "CBookmarks.renderEditNodeXML: Unknown active node type - " + m_activeNodeType);
      renderViewModeXML(out);
    }
  }

  /**
   * getElementChildValue [04-FEB-2003 Mark Mazelin, MarkMazelin@cedarville.edu]
   * Note that this will only get the value from the _first_ matching child tag, even if there are multiple ones
   * that match.
   * @param parentElement The bookmark or folder XML element that is the parent of the child nodes to retrieve.
   * @param childTagName The name of child node tag to retrieve the value.
   * @return returns the Value of the child node or a blank string if the child node is not found.
   */
  private String getElementChildValue (Element parentElement, String childTagName) {
    String returnValue = "";
    NodeList nodeList = parentElement.getElementsByTagName(childTagName);
    if (nodeList.getLength() > 0) {
      // grab the first element
      Element element = (Element)nodeList.item(0);
      if (element.hasChildNodes()) {
        // get the first child (should only have one)
        Text childText = (Text)element.getFirstChild();
        String nodeValue = childText.getNodeValue();
        if (nodeValue != null) {
          returnValue = nodeValue;
        }
      }
    }
    LogService.instance().log(LogService.DEBUG, "CBookmarks.getElementChildValue: returnValue = "+returnValue);
    return returnValue;
  }

  /**
   * setElementChildValue [04-FEB-2003 Mark Mazelin, MarkMazelin@cedarville.edu]
   * Note that this will only set the value on the _first_ matching child tag, even if there are multiple ones
   * that match.
   * @param parentElement The bookmark or folder XML element that is the parent of the child nodes to retrieve.
   * @param childTagName The name of child node tag to retrieve the value.
   * @param newValue The new node value.
   */
  private void setElementChildValue (Element parentElement, String childTagName, String newValue) {
    if ((newValue == null) || (newValue.length() < 1)) { return; }
    NodeList nodeList = parentElement.getElementsByTagName(childTagName);
    if (nodeList.getLength() > 0) {
      // grab the first element
      Element element = (Element)nodeList.item(0);
      if (element.hasChildNodes()) {
        // get the first child (should only have one)
        Text childText = (Text)element.getFirstChild();
        // set the new value
        childText.setNodeValue(newValue);
      }
    }
  }

  private void renderAddNodeXML (ContentHandler out) throws Exception {
    Hashtable parameters = new Hashtable(1);
    if (m_activeNodeType == null) {
      LogService.instance().log(LogService.ERROR, "CBookmarks.renderAddNodeXML: No active node type has been set");
      renderViewModeXML(out);
    }
    else if (m_activeNodeType.equals("bookmark")) {
      parameters.put("EditMode", "AddBookmark");
      transformXML(out, "add_node", getBookmarkXML(), parameters);
    }
    else if (m_activeNodeType.equals("folder")) {
      parameters.put("EditMode", "AddFolder");
      transformXML(out, "add_node", getBookmarkXML(), parameters);
    }
    else {
      LogService.instance().log(LogService.ERROR, "CBookmarks.renderAddNodeXML: Unknown active node type - " + m_activeNodeType);
      renderViewModeXML(out);
    }
  }

 private void renderMoveNodeXML (ContentHandler out) throws Exception {
     
    Hashtable parameters = new Hashtable(2);
     
   if (m_activeNodeType.equals("bookmark")) {
     parameters.put("MoveMode", "MoveBookmark");
    }else if (m_activeNodeType.equals("folder")){
    
    	parameters.put("MoveMode", "MoveFolder");
    }else {
             
    } 
      	   transformXML(out, "move_node", getBookmarkXML(), parameters);
  }



/**private void renderMoveNodeXML (ContentHandler out) throws Exception {
    Hashtable parameters = new Hashtable(2);
  //  parameters.put("NodeType", m_activeNodeType);
    parameters.put("TreeMode", "MoveNode");
    transformXML(out, "move_node", getBookmarkXML(), parameters);
  }
**/







  private void renderDeleteNodeXML (ContentHandler out) throws Exception {
    Hashtable parameters = new Hashtable(1);
    if (m_activeNodeType == null) {
      LogService.instance().log(LogService.ERROR, "CBookmarks.renderDeleteNodeXML: No active node type has been set");
      renderViewModeXML(out);
    }
    else if (m_activeNodeType.equals("bookmark")) {
      parameters.put("EditMode", "DeleteBookmark");
      transformXML(out, "delete_node", getBookmarkXML(), parameters);
    }
    else if (m_activeNodeType.equals("folder")) {
      parameters.put("EditMode", "DeleteFolder");
      transformXML(out, "delete_node", getBookmarkXML(), parameters);
    }
    else {
      LogService.instance().log(LogService.ERROR, "CBookmarks.renderDeleteNodeXML: Unknown active node type - " + m_activeNodeType);
      renderViewModeXML(out);
    }
  }


  private void transformXML (ContentHandler out, String stylesheetName, DocumentImpl inputXML) throws Exception {
    transformXML(out, stylesheetName, inputXML, null);
  }


  private void transformXML (ContentHandler out, String stylesheetName, DocumentImpl inputXML, Hashtable parameters) throws Exception {
    // Create the parameters hashtable if it is null
    if (parameters == null) {
      parameters = new Hashtable(1);
    }

    // Add the baseActionURL to the stylesheet parameters
    parameters.put("baseActionURL", runtimeData.getBaseActionURL());

    // Add target window information - smb1@cornell.edu
    if( m_currentState == VIEWMODE )
      setTargetWindowParameter( parameters );

    // Use the XSLT utility to perform the transformation
    XSLT xslt = new XSLT(this);
    xslt.setXML(inputXML);
    xslt.setXSL(sslLocation, stylesheetName, runtimeData.getBrowserInfo());
    xslt.setTarget(out);
    xslt.setStylesheetParameters(parameters);
    xslt.transform();
  }

  /**
   * Set the value for the link target window.
   * @author smb1@cornell.edu
   * @since 2003.06.18
   * @return
   */
  private void setTargetWindowParameter( Hashtable parameters ) {

    if( !this.staticData.containsKey( TARGETWINDOW ))
      parameters.put( TARGETWINDOW, TARGETWINDOWNONE );
     else
      parameters.put( TARGETWINDOW, this.staticData.getParameter( TARGETWINDOW ));

  }// setTargetWindowParameter

  private String createUniqueID () {
  	String uniqueID="";
  	String temp ="";
  	temp=temp+System.currentTimeMillis();
  	   uniqueID = "n" + temp;
    while (m_bookmarksXML.getElementById(uniqueID) != null) {
      temp=temp+System.currentTimeMillis();
  	  uniqueID = "n" + temp;
    }
    return  (uniqueID);
  }


  private static String makeUrlSafe (String url) {
    // Return if the url is correctly formed
    if (url.toLowerCase().startsWith("http://")) {
      return  (url);
    }
    // Make sure the URL is well formed
    if (url.toLowerCase().startsWith("http:/")) {
      url = url.substring(0, 6) + "/" + url.substring(7);
      return  (url);
    }
    // If it's a mail link then be sure mailto: is on the front
    if (url.indexOf('@') != -1) {
      if (!url.toLowerCase().startsWith("mailto:")) {
        url = "mailto:" + url;
      }
      return  (url);
    }
    // Make sure http:// is on the front
    url = "http://" + url;
    return  (url);
  }



  private Connection getConnection () {
    try {
      RDBMServices rdbmServices = new RDBMServices();
      return  (rdbmServices.getConnection());
    } catch (Exception e) {
      LogService.instance().log(LogService.ERROR, e);
      return  (null);
    }
  }


  private void releaseConnection (Connection connection) {
    try {
      RDBMServices rdbmServices = new RDBMServices();
      rdbmServices.releaseConnection(connection);
    } catch (Exception e) {
      LogService.instance().log(LogService.ERROR, e);
    }
  }
}



