package edu.yale.its.tp.portal.security;

import java.io.IOException;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jasig.portal.ResourceMissingException;
import org.jasig.portal.security.*;
import org.jasig.portal.utils.ResourceLoader;

/**
 * <p>
 * The factory class for the Yale security context.
 * </p>
 * 
 * @author Shawn Bayern
 * @author andrew.petro@yale.edu $Revision: 1.2 $ $Date: 2004/06/16 14:36:29 $
 */

public class YaleCasContextFactory implements ISecurityContextFactory {
    private static final Log log = LogFactory
            .getLog(YaleCasContextFactory.class);

    /**
     * The https: URL to which CAS is to deliver proxy granting tickets. Static
     * to avoid going back to security properties repeatedly.
     */
    private static String casProxyCallbackUrl;

    /**
     * The URL to which service tickets will authenticate users. Static to avoid
     * going back to security properties repeatedly.
     */
    private static String portalServiceUrl;

    /**
     * The URL at which these service tickets can be validated. Static to avoid
     * going back to security properties repeatedly.
     */
    private static String casValidateUrl;

    /**
     * Retrieve my properties from security.properties
     * 
     * @throws IOException
     * @throws ResourceMissingException
     */
    private void initializeFromSecurityProperties()
            throws ResourceMissingException, IOException {
        //We retrieve the tokens representing the credential and principal
        // parameters from the security properties file.
        Properties props = ResourceLoader.getResourceAsProperties(
                YaleCasContextFactory.class, "/properties/security.properties");

        YaleCasContextFactory.casProxyCallbackUrl = props
                .getProperty("org.jasig.portal.security.provider.YaleCasContext.CasProxyCallbackUrl");
        log.debug("CasProxyCallbackUrl is ["
                + YaleCasContextFactory.casProxyCallbackUrl + "]");
        YaleCasContextFactory.portalServiceUrl = props
                .getProperty("org.jasig.portal.security.provider.YaleCasContext.PortalServiceUrl");
        log.debug("PortalServiceUrl is ["
                + YaleCasContextFactory.portalServiceUrl + "]");
        YaleCasContextFactory.casValidateUrl = props
                .getProperty("org.jasig.portal.security.provider.YaleCasContext.CasValidateUrl");
        log.debug("CasValidateUrl is [" + YaleCasContextFactory.casValidateUrl
                + "]");
    }

    public ISecurityContext getSecurityContext() {
        try {
            if (YaleCasContextFactory.casValidateUrl == null
                    || YaleCasContextFactory.portalServiceUrl == null) {
                initializeFromSecurityProperties();
            }
            return new YaleCasContext(YaleCasContextFactory.portalServiceUrl,
                    YaleCasContextFactory.casValidateUrl,
                    YaleCasContextFactory.casProxyCallbackUrl);
        } catch (Throwable t) {
            log.error(t);
            return new BrokenSecurityContext();
        }
    }

    /**
     * Static setter method for poor-man's dependency injection. Calling this
     * class's static setter methods is an alternative to using
     * security.properties. You could call this from a context listener. You can
     * safely just ignore this method.
     * 
     * @param casProxyCallbackUrlArg -
     *                the https: URL to which you would like CAS deliver proxy
     *                granting tickets.
     */
    public static void setCasProxyCallbackUrl(String casProxyCallbackUrlArg) {
        if (casProxyCallbackUrlArg != null
                || casProxyCallbackUrlArg.toUpperCase().startsWith("HTTPS://"))
            YaleCasContextFactory.casProxyCallbackUrl = casProxyCallbackUrlArg;
    }

    /**
     * Static setter method for poor-man's dependency injection. Calling this
     * class's static setter methods is an alternative to using
     * security.properties. You could call this from a context listener. You can
     * safely just ignore this method.
     * 
     * @param portalServiceUrlArg -
     *                the URL to which service tickets will authenticate portal
     *                users
     */
    public static void setPortalServiceUrl(String portalServiceUrlArg) {
        if (portalServiceUrlArg == null)
            throw new IllegalArgumentException(
                    "Cannot set the portal service URL to null.");
        YaleCasContextFactory.portalServiceUrl = portalServiceUrlArg;
    }

    /**
     * Static setter method for poor-man's dependency injection. Calling this
     * class's static setter methods is an alternative to using
     * security.properties. You could call this from a context listener. You can
     * safely just ignore this method.
     * 
     * @param casValidateUrlArg -
     *                the https: URL at which CAS will validate the tickets
     */
    public static void setCasValidateUrl(String casValidateUrlArg) {
        if (casValidateUrlArg == null)
            throw new IllegalArgumentException(
                    "Cannot set the cas validate URL to null.");
        YaleCasContextFactory.casValidateUrl = casValidateUrlArg;
    }

}