package edu.yale.its.tp.portal.security;

import java.util.Enumeration;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jasig.portal.ChannelRuntimeData;
import org.jasig.portal.ChannelStaticData;
import org.jasig.portal.security.IPerson;
import org.jasig.portal.security.ISecurityContext;
import org.jasig.portal.security.LocalConnectionContext;

/**
 * Connection context that uses a security context implementing
 * IYaleSecurityContext to obtain a proxy tickets for use in descriptors
 * provided by this connection context.
 * 
 * @author unattributed
 * @author andrew.petro@yale.edu
 * @author susan.bramhall@yale.edu
 */
public class CasConnectionContext extends LocalConnectionContext {
    private static final Log log = LogFactory
            .getLog(CasConnectionContext.class);

    private ChannelStaticData staticData = null;

    private IPerson person = null;

    private IYaleCasContext yaleSecurityContext = null;

    public void init(ChannelStaticData sd) {
        this.staticData = sd;
        this.person = sd.getPerson();

        ISecurityContext ic = this.person.getSecurityContext();
        if (ic instanceof IYaleCasContext && ic.isAuthenticated())
            this.yaleSecurityContext = (IYaleCasContext) ic;

        // loop through subcontexts to find implementations of
        // IYaleSecurityContext
        Enumeration en = ic.getSubContexts();
        while (en.hasMoreElements()) {
            ISecurityContext sctx = (ISecurityContext) en.nextElement();
            if (sctx instanceof IYaleCasContext && sctx.isAuthenticated())
                this.yaleSecurityContext = (IYaleCasContext) sctx;
        }

        if (this.yaleSecurityContext == null)
            log.error("Unable to find authenticated IYaleCasContext");
    }

    public String getDescriptor(String descriptor, ChannelRuntimeData rd) {
        if (log.isTraceEnabled()) {
            log.trace("getDescriptor(" + descriptor + ", " + rd + ")");
        }
        if (rd.getHttpRequestMethod().equals("GET")) {
            // get proxy service ticket for the service if needed
            String proxyTicket = null;
            if (this.yaleSecurityContext != null)
                try {
                    proxyTicket = this.yaleSecurityContext
                            .getCasServiceToken(descriptor);
                } catch (CASProxyTicketAcquisitionException casex) {
                    log.error(
                            "getDescriptor() - Error retreiving proxy ticket.",
                            casex);
                }

            // if the descriptor was null then at least return
            // the pst as a unique string
            if (descriptor == null)
                descriptor = proxyTicket;

            // append ticket parameter and value to query string
            if (descriptor.indexOf("?") != -1) {
                descriptor = descriptor + "&ticket=" + proxyTicket;
            } else {
                descriptor = descriptor + "?ticket=" + proxyTicket;
            }
        }

        if (log.isTraceEnabled()) {
            log.trace("returning from getDescriptor() with [" + descriptor
                    + "]");
        }
        return descriptor;

    }

    /**
     * Returns url with proxy service ticket appended. Looks for static
     * parameter upc_cas_service_uri and uses that for service. If not
     * specified, uses the passed uri
     * 
     * @param descriptor
     *                The original descriptor.
     * @return descriptor with CAS proxy ticket parameter appended.
     */
    public String getDescriptor(String descriptor) {
        if (log.isTraceEnabled()) {
            log.trace("entering getDescriptor(" + descriptor + ")");
        }
        String proxyTicket = null;
        if (this.yaleSecurityContext != null)
            try {
                // if no specified parameter for service, use target descriptor
                String casUri = this.staticData
                        .getParameter("upc_cas_service_uri");
                if (casUri != null)
                    proxyTicket = this.yaleSecurityContext.getCasServiceToken(casUri);
                else
                    proxyTicket = this.yaleSecurityContext
                            .getCasServiceToken(descriptor);
            } catch (CASProxyTicketAcquisitionException casex) {
                log
                        .error(
                                "CasConnectionContext::getDescriptor() - Error retreiving proxy ticket.",
                                casex);
            }
        // append ticket parameter and value to query string
        if (descriptor.indexOf("?") != -1)
            descriptor = descriptor + "&ticket=" + proxyTicket;
        else
            descriptor = descriptor + "?ticket=" + proxyTicket;
        
        if (log.isTraceEnabled()){
            log.trace("returning from getDescriptor() with [" + descriptor + "]");
        }
        return descriptor;
    }

    public String getPostData(ChannelRuntimeData rd) {
        // get proxy service ticket for the service if needed
        String proxyTicket = null;
        if (this.yaleSecurityContext != null)
            try {
                String xmlUri = rd.getParameter("cw_xml");
                if (xmlUri == null)
                    xmlUri = this.staticData.getParameter("cw_xml");
                proxyTicket = this.yaleSecurityContext.getCasServiceToken(xmlUri);
            } catch (CASProxyTicketAcquisitionException casex) {
                log.error("sendLocalData() - Error retreiving proxy ticket.",
                        casex);
            }
        return ("ticket=" + proxyTicket);
    }

    public void sendLocalData(Object conParam, ChannelRuntimeData rd) {
        // CAS does not have anything to do here. Post data is handled by the
        // getPostData method instead of being sent separately in this method.
        return;
    }

    public String toString() {
        StringBuffer sb = new StringBuffer();
        sb.append(this.getClass().getName());
        sb.append(" staticData:");
        sb.append(this.staticData);
        sb.append(" person:");
        sb.append(this.person);
        sb.append(" yaleSecurityContext:");
        sb.append(this.yaleSecurityContext);
        return sb.toString();
    }
}