package edu.yale.its.tp.portal.security;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jasig.portal.security.*;
import org.jasig.portal.security.provider.ChainingSecurityContext;

import edu.yale.its.tp.cas.client.ServiceTicketValidator;
import edu.yale.its.tp.cas.proxy.ProxyTicketReceptor;
import java.io.IOException;

/**
 * <p>
 * A SecurityContext using Yale's Central Authentication Server.
 * </p>
 * An IYaleCasContext implementation in keeping with the traditional approach of
 * performing actual authentication inside this security context. This security
 * context is suitable for cases where your login URL (the "portalServiceUrl")
 * is a constant.
 * 
 * @author Susan Bramhall
 * @author andrew.petro@yale.edu 
 * @version $Revision: 1.7 $ $Date: 2004/07/14 17:09:27 $
 */
public class YaleCasContext extends ChainingSecurityContext implements
        IYaleCasContext {

    private static final Log log = LogFactory.getLog(YaleCasContext.class);

    /**
     * The URL of the uPortal Login servlet to which service tickets will
     * authenticate users.
     */
    private final String portalServiceUrl;

    /**
     * The https: URL at which CAS offers its ticket validation service.
     */
    private final String casValidateUrl;

    /**
     * The https: URL at which CAS is to call back the uPortal with Proxy
     * Granting Tickets.
     */
    private final String casProxyCallbackUrl;

    /**
     * The pgtIou which keys to the Proxy Granting Ticket associated with this
     * authenticated security context, if any.
     */
    private String pgtIou = null;

    /**
     * Instantiate a YaleCasContext given a constant URL to which CAS service
     * tickets will authenticate users, a URL at which to validate those
     * tickets, and a callback URL at which to ask CAS to deliver Proxy Granting
     * Tickets.
     * 
     * @param portalServiceUrl -
     *                the constant URL to which service tickets authenticate users
     * @param casValidateUrl -
     *                the https: URL at which CAS offers its ticket validation
     *                service
     * @param casProxyCallbackUrl -
     *                the https: URL to which CAS should send proxy granting
     *                tickets.
     */
    public YaleCasContext(String portalServiceUrl, String casValidateUrl,
            String casProxyCallbackUrl) {
        super();
        log.trace("entering YaleCasContext(" + portalServiceUrl + ","
                + casValidateUrl + "," + casProxyCallbackUrl + ")");

        if (portalServiceUrl == null) {
            throw new IllegalArgumentException(
                    "Cannot instantiate a YaleCasContext with a null portalServiceUrl.");
        }
        if (casValidateUrl == null) {
            throw new IllegalArgumentException(
                    "Cannot instantiate a YaleCasContext with a null casValidateUrl.");
        }
        this.casProxyCallbackUrl = casProxyCallbackUrl;
        this.portalServiceUrl = portalServiceUrl;
        this.casValidateUrl = casValidateUrl;

    }

    /**
     * Instantiate a YaleCasContext given a constant URL to which CAS service
     * tickets will authenticate users, a URL at which to validate those
     * tickets.
     * 
     * @param portalServiceUrl -
     *                the constant URL to which service tickets authenticate users
     * @param casValidateUrl -
     *                the https: URL at which CAS offers its ticket validation
     *                service
     */
    public YaleCasContext(String portalServiceUrl, String casValidateUrl) {
        this(portalServiceUrl, casValidateUrl, null);
    }

    public int getAuthType() {
        return IYaleCasContext.YALE_CAS_AUTHTYPE;
    }

    public synchronized void authenticate() throws PortalSecurityException {
        if (log.isTraceEnabled()) {
            log.trace("entering authenticate()");
        }
        String serviceTicket = new String(
                this.myOpaqueCredentials.credentialstring);

        this.isauth = false;
        try {
            ServiceTicketValidator sv = new ServiceTicketValidator();

            sv.setCasValidateUrl(this.casValidateUrl);
            if (this.casProxyCallbackUrl != null) {
                sv.setProxyCallbackUrl(this.casProxyCallbackUrl);
            }
            sv.setService(this.portalServiceUrl);
            sv.setServiceTicket(serviceTicket);
            log.debug("authenticate(): Validating ServiceTicket: ["
                    + serviceTicket + "]");
            sv.validate();
            log
                    .debug("authenticate(): got response:[" + sv.getResponse()
                            + "]");

            if (sv.isAuthenticationSuccesful()) {
                this.myPrincipal.setUID(sv.getUser());

                // We keep the pgtIOU around as the key to reference the Proxy
                // granting ticket
                // that will be used to obtain future Proxy Tickets.
                // A channel has
                // access to this securityContext and can request a Proxy ticket
                // via
                // the public getCasServiceToken method
                this.pgtIou = sv.getPgtIou();
                this.isauth = true;
                log.debug("CASContext authenticated ["
                        + this.myPrincipal.getUID() + "]");
            }
            //else {
            //  throw new PortalSecurityException("error code: " +
            // sv.getErrorCode()+"\n error message: "+sv.getErrorMessage());
            // }
        } catch (Throwable t) {
            log.error(t);
            throw new PortalSecurityException("Error in CAS Authentication: "
                    + t.getMessage());
            /*
             * if PortalSecurityException supported it, it would be nice to
             * chain here: throw new PortalSecurityException("Error in CAS
             * Authentication:", t);
             */
        }
        this.myAdditionalDescriptor = null; //no additional
                                                      // descriptor from CAS
        super.authenticate();
        if (log.isTraceEnabled()) {
            log.trace("returning from authenticate()");
        }
        return;
    }

    public String getCasServiceToken(String target)
            throws CASProxyTicketAcquisitionException {
        if (log.isTraceEnabled()) {
            log.trace("entering getCasServiceToken(" + target
                    + "), previously cached pgtIou=[" + this.pgtIou + "]");
        }
        String proxyTicket;
        try {
            proxyTicket = ProxyTicketReceptor.getProxyTicket(this.pgtIou,
                    target);
        } catch (IOException e) {
            throw new CASProxyTicketAcquisitionException(target, this.pgtIou, e);
        }
        if (log.isTraceEnabled()) {
            log.trace("returning from getCasServiceToken() with return value ["
                    + proxyTicket + "]");
        }
        return proxyTicket;
    }

    public String toString() {
        StringBuffer sb = new StringBuffer();
        sb.append(this.getClass().getName());
        sb.append(" portalServiceUrl=[").append(this.portalServiceUrl).append(
                "]");
        sb.append(" casValidateUrl=[").append(this.casValidateUrl).append("]");
        sb.append(" casProxyCallbackUrl=[").append(this.casProxyCallbackUrl)
                .append("]");
        sb.append(" pgtIou=[").append(this.pgtIou).append("]");
        return sb.toString();
    }
}