package org.esupportail.portlet.spring.dao;

import java.util.List;
import java.util.Map;

import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.BasicAttributes;

import net.sf.ldaptemplate.AttributesMapper;
import net.sf.ldaptemplate.LdapTemplate;
import net.sf.ldaptemplate.support.DistinguishedName;

import org.esupportail.portlet.spring.beans.Entry;

public class DataServiceLdap implements DataService {

    private String context;
    private Map<String, Object> staticAttributes;
    private LdapTemplate ldapTemplate;

    /**
     * Positionne le contexte LDAP
     * @param context
     */	
    public void setContext(String context) {
	this.context = context;
    }

    /**
     * Positionne les attributs statiques
     * @param staticAttributes
     */
    public void setStaticAttributes(Map<String, Object> staticAttributes) {
	this.staticAttributes = staticAttributes;
    }

    /**
     * Positionne le service LdapTemplate
     * @param ldapTemplate
     */
    public void setLdapTemplate(LdapTemplate ldapTemplate) {
	this.ldapTemplate = ldapTemplate;
    }

    /**
     * Retourne la liste des adresses
     * @return
     */
    @SuppressWarnings("unchecked")
    public List<Entry> getAddresses() {
	DistinguishedName dn = new DistinguishedName(context);
	return (List<Entry>)ldapTemplate.search(dn, "uid=*", new EntryAttributesMapper());
    }

    /**
     * Retourne une adresse
     * @param id
     * @return
     */
    public Entry getAddress(String id) {
	DistinguishedName dn = new DistinguishedName(context);
	return (Entry)ldapTemplate.search(dn, "uid=" + id, new EntryAttributesMapper());
    }

    /**
     * Ajoute une adresse
     * @param entry
     */
    public void addAddress(Entry entry) {
	DistinguishedName dn = new DistinguishedName(context);
	dn.add("uid", entry.getId());
	ldapTemplate.bind(dn, null, buildAttributes(entry));
    }

    /**
     * Met  jour une adresse
     * @param entry
     */
    public void updateAddress(Entry entry) {
	DistinguishedName dn = new DistinguishedName(context);
	dn.add("uid", entry.getId());
	ldapTemplate.rebind(dn, null, buildAttributes(entry));
    }

    /**
     * Supprime une adresse
     * @param entry
     */
    public void deleteAddress(Entry entry) {
	DistinguishedName dn = new DistinguishedName(context);
	dn.add("uid", entry.getId());
	ldapTemplate.unbind(dn);
    }

    /**
     * Mapping Objet Java / Attributs LDAP
     * @param entry
     * @return
     */
    private Attributes buildAttributes(Entry entry) {
	Attributes attrs = new BasicAttributes(true);
	// Attributs statiques
	for(String key : staticAttributes.keySet()) {
	    Object o = staticAttributes.get(key);
	    if(o instanceof List) {
		List values = (List)o;
		Attribute a = new BasicAttribute(key);
		for(Object value : values) {
		    a.add(value);
		}
		attrs.put(a);
	    }
	    else {
		attrs.put(key, o);
	    }
	}
	attrs.put("uid", entry.getId());
	attrs.put("sn", entry.getName());
	attrs.put("givenName", entry.getFirstName());
	attrs.put("mail", entry.getMail());
	return attrs;
    }

    private class EntryAttributesMapper implements AttributesMapper {

	/**
	 * Mapping Attributs LDAP / Objet Java
	 * @param attrs
	 * @return
	 */
	public Object mapFromAttributes(Attributes attrs) throws NamingException {
	    Entry entry = new Entry();
	    entry.setId(attrs.get("uid").get().toString());
	    entry.setName(attrs.get("sn").get().toString());
	    entry.setFirstName(attrs.get("givenName").get().toString());
	    entry.setMail(attrs.get("mail").get().toString());
	    return entry;
	}
    }
}
