/*
ESUP-portail is a french academic project developed under the GPL (General Public License) augmented according to the following :
A binary or source file developped by ESUP-portail can be used or compiled with products under Apache license.
The official english text of the GPL can be found here : http://www.gnu.org/licenses/gpl.html .
A non official french translation can be found here : http://www.linux-France.org/article/these/gpl.html .
The different kinds of licenses governing the products developed by the Apache foundation can be found here : http://www.apache.org/licenses .
It follows that you can as well as use download contents as well modify and redistribute them provided you respect the GPL terms.
Downloading and using such contents do not provide any guaranty.
Be sure that you have well understood the terms of the license before using the contents it covers.
The ESUP-portail distribution includes the following distributions :
    * UPortal :
      software developed by JA-SIG (Java Architecture - Special Interest Group)
      You can find the license page here : http://mis105.udel.edu/ja-sig/uportal/license.html
    * CAS :
      SSO solution developed by Yale University
      You can find the project page here : http://www.yale.edu/tp/auth
    * Cocoon :
      XML framework distributed by the Apache foundation under Apache license;
      Please find the full text here : http://cocoon.apache.org/2.1/license.html
    * Mod_dav:
      A DAV module for Apache web server
      You can find the project page here : http://www.webdav.org/mod_dav
    * IMP :
      webmail from Horde application framework
      You can find the project page here : http://www.horde.org
    * . To be completed
*/
package org.esupportail.portal.channels.CAnnuaire.LDAP;
    
import java.util.Hashtable;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.AttributeModificationException;
import javax.naming.directory.DirContext;
import javax.naming.directory.InitialDirContext;
import javax.naming.directory.Attributes; 
import javax.naming.directory.SearchControls;
import javax.naming.*;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.esupportail.portal.channels.CAnnuaire.LDAP.Exceptions.*;

public class LDAPHandler {

    private static final Log log = LogFactory.getLog(LDAPHandler.class);
    private String[] attributes = null;
	private String[] attributesValues = null;
	private int maxEntries = 0;
	private String base = null;
	private String filter = null;
	private int scope = 0;
	private String url = null;
	private String bindDn = null;
	private String bindPass = null;
	private DirContext  ctx = null;
	private Hashtable env = null;

	/**
	 * Constructeur
	 */
	public LDAPHandler() {
	}

	/**
	 * Dfinit les attributs  rechercher ou a crire
	 * @param listAttr
	 */
	public void setAttributes(String[] listAttr) {
		attributes = listAttr;
	}
	
	/**
	 * Dfinit les valeurs des attributs  enregistrer
	 * @param listValues
	 */
	public void setAttributesValues(String[] listValues) {
		attributesValues = listValues;
	}

	/**
	 * Dfinit le nombre maximun de rponse que peut renvoyer la mthode de recherche
	 * 0 indique un nombre illimit de rponses
	 * @param max
	 */
	public void setMaxEntries(int max) {
		maxEntries = max;
	}
 
	/**
	 * Dfinit la base de recherche
	 * @param b
	 */
	public void setBase(String b) {
		base = b;
	}

	/**
	 * Dfinit le filtre de recherche
	 * @param f
	 */
	public void setFilter(String f) {
		filter = f;
	}

	/**
	 * Dfinit la profondeur de recherche (sub ou one-level)
	 * @param s
	 */
	public void setScope(int s) {
		scope = s;
	}

	/**
	 * Dfinit l'url de connection
	 * @param u
	 */
	public void setUrl(String u) {
		url = u;
	}
	
	/**
	 * Dfinit le DN de connection
	 * @param u
	 */
	public void setBindDn(String u) {
		bindDn = u;
	}
	
	/**
	 * Dfinit le mo de connection 
	 * @param u
	 */
	public void setBindPass(String u) {
		bindPass = u;
	}

	/**
	 * Lance la recherche
	 * @return L'numration des rsultats
	 * @throws LDAPException (SearchLDAPException)
	 */
	public NamingEnumeration search() throws LDAPException {
		NamingEnumeration answer = null;

		if (authenticate()){
			
			/*Criteres de recherche*/
			SearchControls sc = new SearchControls();
			sc.setReturningAttributes(attributes);
			sc.setCountLimit(maxEntries);
			sc.setSearchScope(scope);
			
			/*Lance la recherche*/	
			try {
				answer = ctx.search(base, filter, sc);
			} catch (NamingException e) {
			    log.error("LDAPHandler:search() : le filtre "+filter+" n'est pas correcte. ",e);
				throw new SearchLDAPException();
			}
			}
		
		return (answer);
	}
	
	
	/**
	 * Ferme la connection 
	 */
	public void close() {
		if (ctx!=null){
			try {
				ctx.close();
				ctx = null;
			} catch (NamingException e) {
			    log.error("LDAPHandler:close() : " ,e);
			}
		}
	}
	
		
	/**
	 * Ouvre la connection. Si le BindDN est renseign, c'est une connection non anonyme.
	 * @return True si la connection est ouverte.
	 * @throws LDAPException (AuthLDAPException)
	 */
	public boolean authenticate() throws LDAPException{
		String dnLog = new String ();
		dnLog = "anonymous";
		
		env = new Hashtable();
		env.put(
			InitialDirContext.INITIAL_CONTEXT_FACTORY,
			"com.sun.jndi.ldap.LdapCtxFactory");
		env.put(Context.PROVIDER_URL, url);
		
		if ((bindDn != null) && (!bindDn.equals("")) ){
			env.put(InitialDirContext.SECURITY_AUTHENTICATION, "simple");
			env.put(InitialDirContext.SECURITY_PRINCIPAL,      bindDn);
			env.put(InitialDirContext.SECURITY_CREDENTIALS,    bindPass);
			dnLog = bindDn;
		}

		ctx = null;
		
		try {
			ctx = new InitialDirContext(env);
		} catch (NamingException e1) {
		    log.error("LDAPHandler:authenticate() : " + dnLog +" authentification failed");
			throw new AuthLDAPException("L'authentification de "+dnLog+" a chou.");
		}
		
		if (log.isDebugEnabled()){
		    log.debug("LDAPHandler:authenticate() : " + dnLog +" authenticated");
		}
		return true;
	}
	
	

	/**
	 * Ecrit dans LDAP l'objet dont le DN est en paramtre.
	 * L'criture ne concerne que les attributs dfinis.
	 * @param shortDn
	 * @throws LDAPException (SearchLDAPException,WriteLDAPException)
	 */
	public void write(String shortDn) throws LDAPException{
		if (authenticate()){
			boolean update = false;
			Attributes a = null;
			
			try {
				a = ctx.getAttributes(shortDn,attributes);
			} catch (NamingException e1) {
			    log.error("LDAPHandler:write() : NamingException : ",e1);
				throw new SearchLDAPException("erreur dans l'acquisition d'un attribut.");
			}
			
			/*Si l'entre existe et a retourn au moins un des attributs demands*/
			if (a!=null){
			    
			    
				log.info("LDAPHandler:write() : Preparation de modification de "+bindDn+" ");
				/*Parcours de la liste des attributs demands*/			    
				for (int i=0;i<attributes.length;i++){
				    /*
				     * si l'attribut courant n'est pas renseign dans l'entre slectionne
				     * on l'ajoute
				     * */
				    if( (a.get(attributes[i])==null) 
				         && (attributesValues[i]!=null) 
				         && (!attributesValues[i].equals("")) 
				         )  {  
				    	if (log.isDebugEnabled()){
						    log.debug("LDAPHandler:write() : attribut : "+attributes[i]);
						}
						a.put(attributes[i],attributesValues[i]);     
						log.info("LDAPHandler:write() : * ajout de "+attributes[i]+" ");      
						update = true;
					}
				    
				    /*
				     * Si l'attribut courant existe, on le met  jour si sa valeur n'est pas 
				     * la nouvelle valeur
				     * */
					else if ((attributesValues[i]!=null) 
					         && (a.get(attributes[i])!=null) 
					         && (!a.get(attributes[i]).contains(attributesValues[i]))
					         && (!attributesValues[i].equals("")) 
					         )  {  
						if (log.isDebugEnabled()){
						    log.debug("LDAPHandler:write() : attribut : "+attributes[i]);
						}                
						a.get(attributes[i]).remove(0);
						a.get(attributes[i]).add(0,attributesValues[i]);
						log.info("LDAPHandler:write() : * modification de "+attributes[i]+" ");
						update = true;
					}
					
					/*
					 * Si l'attribut existe mais que la valeur est vide
					 * on supprime l'attribut
					 */
					else if ( (a.get(attributes[i])!=null)
					          && ( (attributesValues[i]==null) 
					               || (attributesValues[i].equals("")) 
					         	 )
					          )  { 
						
							a.get(attributes[i]).remove(0);
							log.info("LDAPHandler:write() : * suppression de "+attributes[i]+" ");
							update = true;
					}
				}
				
				if (update){		
					try {

						log.info("LDAPHandler:write() : tentative d'enregistrement ...");
						
						ctx.modifyAttributes(shortDn,DirContext.REPLACE_ATTRIBUTE,a);

						log.info("LDAPHandler:write() : enregistrement effectue.");
		            }
		            catch(NoPermissionException e) {
		                log.error("LDAPHandler:write() : modification de "+a+" => NoPermissionException : ",e);
						throw new WriteLDAPException("Vous n'avez pas les droits pour modifier l'un des attributs.");
		            } 
		            catch (AttributeModificationException e) {
		                log.error("LDAPHandler:write() : modification de "+a+" => AttributeModificationException : ",e);
						throw new WriteLDAPException("Une erreur est survene lors de la modification de l'un des champs.");
					}  
					catch (OperationNotSupportedException e) {
		                log.error("LDAPHandler:write() : modification de "+a+" => OperationNotSupportedException : ",e);
						throw new WriteLDAPException("Une erreur est survene lors de la modification de l'un des champs.");
					} 
					catch (PartialResultException e) {
		                log.error("LDAPHandler:write() : modification de "+a+" => PartialResultException : ",e);
						throw new WriteLDAPException("Une erreur est survene lors de la modification de l'un des champs.");
					} 
					catch (NamingException e) {
		                log.error("LDAPHandler:write() : modification de "+a+" => NamingException : ",e);
						throw new WriteLDAPException("Une erreur est survene lors de la modification de l'un des champs.");
					} 
				}
				else{
					log.info("LDAPHandler:write() : rien a modifier");
				}
			}	
			else{
			    log.error("LDAPHandler:write() : impossible de trouver l'entree LDAP");
				throw new SearchLDAPException("Erreur dans l'acquisition d'un attribut.");
			}
		}
	}
}
