package org.esupportail.portal.channels.CIntranet.data;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.LinkedList;
import java.util.Vector;

import org.apache.log4j.Priority;
import org.esupportail.portal.channels.CIntranet.beans.Document;
import org.esupportail.portal.channels.CIntranet.beans.Folder;
import org.esupportail.portal.channels.CIntranet.beans.Intranet;
import org.esupportail.portal.channels.CIntranet.beans.Mime;
import org.esupportail.portal.channels.CIntranet.beans.Path;
import org.esupportail.portal.channels.CIntranet.beans.Population;
import org.esupportail.portal.channels.CIntranet.beans.SubFolder;
import org.esupportail.portal.channels.CIntranet.cache.Cache;
import org.esupportail.portal.channels.CIntranet.cache.CircularCache;
import org.esupportail.portal.channels.CIntranet.config.Config;
import org.esupportail.portal.utils.database.Query;
import org.jasig.portal.RDBMServices;
import org.jasig.portal.services.LogService;

/**
 * DataReadManager<br>
 * <br>
 * Implmentation d'un IDataReadManager utilisant des instructions<br>
 * SQL standard<br>
 * L'utilisation d'instructions spcifiques  un SGBD doit tre<br>
 * ralise dans une autre implmentation<br>
 * <br>
 * (c)Copyright <a href="http://www.esup-portail.org">ESup-Portail 2004</a><br>
 * @author <a href="mailto:mathieu.larchet@univ-nancy2.fr">Mathieu Larchet</a>
 * @version 1.3
 * 
 */

public class DataReadManager implements IDataReadManager {

	private static Cache FOLDER_PATH = new CircularCache(32);
	private static Cache INTRANET = new CircularCache(8);
	private Priority nivlog = Priority.DEBUG;
	private int requests = 0;
	private Query query = null;

	/**
	 * Constructeur
	 */
	public DataReadManager() {
	}

	/************************************************************
	 *	Intranets												*
	 ************************************************************/
	
	/**
	 * Rcupration d'un intranet
	 * @param id L'identifiant d'intranet
	 * @return L'intranet correspondant
	 * @throws DataException
	 */
	public Intranet getIntranet(String id) throws DataException {
		LogService.log(nivlog, "DataReadManager::getIntranet()");
		
		// Recherche dans le cache
		Intranet cachedintranet = (Intranet)INTRANET.get(id);
		if(cachedintranet != null) {
			LogService.log(nivlog, "DataReadManager::getIntranet() : Trouv dans le cache");
			return cachedintranet;
		}
		
		Intranet res = new Intranet();
		String sql = "SELECT ID_DOS, INT_INTRANET.ID_INT, NOM_INT, ID_RES FROM INT_INTRANET, INT_DOSSIER WHERE INT_INTRANET.ID_INT = '" + id + "' AND INT_DOSSIER.ID_INT = '" + id + "' AND INT_DOSSIER.ID_PAR = 0";
		Query query = getQuery();
		try {
			query.setSql(sql);
			query.select();
			ResultSet result = query.getRs();
			if(!result.next()) {
				LogService.log(nivlog, "DataReadManager::getIntranet() : Aucun intranet ayant pour ID = " + id);
				close();
				throw new DataException("Impossible d'accder  cet intranet");
			}
			res.setId(id);
			res.setName(result.getString("NOM_INT"));
			res.setRessource(result.getString("ID_RES"));
			res.setRoot(result.getString("ID_DOS"));
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataReadManager::getIntranet() : " + e);
			LogService.log(Priority.ERROR, "DataReadManager::getIntranet() : SQL : " + sql);
			close();
			throw new DataException("Impossible d'accder  cet intranet");
		}
		close();
		
		// Mise en cache
		INTRANET.put(id, res);
		return res;
	}

	/**
	 * Rcupration de la liste de tous les intranets
	 * @return La liste de tous les intranets
	 * @throws DataException
	 */
	public Collection getIntranets() throws DataException {
		LogService.log(nivlog, "DataReadManager::getIntranets()");
		Collection res = new Vector();
		
		String sql = "SELECT INT_INTRANET.*, INT_DOSSIER.ID_DOS FROM INT_INTRANET, INT_DOSSIER WHERE INT_DOSSIER.ID_PAR = 0 AND INT_DOSSIER.ID_INT = INT_INTRANET.ID_INT ORDER BY NOM_INT ASC";
		Query query = getQuery();
		try {
			query.setSql(sql);
			query.select();
			ResultSet result = query.getRs();
			while(result.next()) {
				Intranet tmp = new Intranet();
				tmp.setId(result.getString("ID_INT"));
				tmp.setName(result.getString("NOM_INT"));
				tmp.setRessource(result.getString("ID_RES"));
				tmp.setRoot(result.getString("ID_DOS"));
				res.add(tmp);
			}
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataReadManager::getIntranets() : " + e);
			LogService.log(Priority.ERROR, "DataReadManager::getIntranets() : SQL : " + sql);
			close();
			throw new DataException("Impossible d'accder aux intranets");
		}
		close();
		return res;
	}

	
	/************************************************************
	 *	Populations												*
	 ************************************************************/

	/**
	 * Rcupration d'une population
	 * @param id L'identifiant d'intranet
	 * @return La population de l'intranet
	 * @throws DataException
	 */
	public Population getPopulation(String id) throws DataException {
		LogService.log(nivlog, "DataReadManager::getPopulation()");
		Population res = new Population();
		Collection groups = new Vector();
		Collection users = new Vector();
		
		StringBuffer sql1 = new StringBuffer();
		sql1.append("SELECT ID_POP FROM INT_POPULATION WHERE ");
		sql1.append("ID_INT = '" + id + "' AND TYPE_POP = 'U'");
		
		StringBuffer sql2 = new StringBuffer();
		sql2.append("SELECT ID_POP FROM INT_POPULATION WHERE ");
		sql2.append("ID_INT = '" + id + "' AND TYPE_POP = 'G'");
		
		Query query = getQuery();
		
		try {
			query.setSql(sql1.toString());
			query.select();
			ResultSet result = query.getRs();
			while(result.next()) {
				users.add(result.getString("ID_POP"));
			}
			res.setUsers(users);
			query.setSql(sql2.toString());
			query.select();
			result = query.getRs();
			while(result.next()) {
				groups.add(result.getString("ID_POP"));
			}
			res.setGroups(groups);
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataReadManager::getPopulation() : " + e);
			close();
			throw new DataException("Erreur lors de la lecture de la population");
		}
		close();
		return res;
	}
	
	
	/************************************************************
	 *	Super Utilisateurs										*
	 ************************************************************/

	/**
	 * Rcupration des super-utilisateurs
	 * @return La liste des super-utilisateurs
	 * @throws DataException
	 */
	public Collection getSuperUsers() throws DataException {
		LogService.log(nivlog, "DataReadManager::getSuperUsers()");
		Collection res = new Vector();
		
		String sql = "SELECT LOGIN FROM INT_SUPERUSER";
		
		Query query = getQuery();
		try {
			query.setSql(sql);
			query.select();
			ResultSet result = query.getRs();
			while(result.next()) {
				res.add(result.getString("LOGIN"));
			}
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataReadManager::getSuperUsers() : " + e);
			LogService.log(Priority.ERROR, "DataReadManager::getIntranet() : SQL : " + sql);
			close();
			throw new DataException("Erreur lors de la lecture des super utilisateurs");
		}
		close();
		return res;
	}

	
	/************************************************************
	 *	Dossiers												*
	 ************************************************************/
	
	/**
	 * Rcupration d'un dossier
	 * @param id L'identifiant du dossier
	 * @param sub Si true alors on rcupre galement les sous-dossiers<br>
	 * et les documents
	 * @return Le dossier
	 * @throws DataException
	 */
	public Folder getFolder(String id, boolean sub) throws DataException {
		LogService.log(nivlog, "DataReadManager::getFolder()");
		Folder res = new Folder();
		
		String sql = "SELECT * FROM INT_DOSSIER WHERE ID_DOS = '" + id + "'";
		Query query = getQuery();
		try {
			query.setSql(sql);
			query.select();
			ResultSet result = query.getRs();
			if(!result.next()) {
				LogService.log(nivlog, "DataReadManager::getFolder() : Aucun dossier ayant pour ID = " + id);
				close();
				throw new DataException("Impossible d'accder  ce dossier");
			}
			res.setId(id);
			res.setIntranet(result.getString("ID_INT"));
			res.setName(result.getString("NOM_DOS"));
			res.setParent(result.getString("ID_PAR"));
			String val = result.getString("VAL_DOS");
			if(val.equals("1")) {
				res.setValidation(true);
			}
			else {
				res.setValidation(false);
			}
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataReadManager::getFolder() : " + e);
			LogService.log(Priority.ERROR, "DataReadManager::getIntranet() : SQL : " + sql);
			close();
			throw new DataException("Impossible d'accder  ce dossier");
		}
		close();
		
		if(sub) {
			// Rcupration chemin
			res.setPath(getPath(id));
			// Rcupration sous dossiers
			res.setSubFolders(getSubFolders(id));
			// Rcupration des documents
			res.setDocuments(getDocuments(id));
		}
		return res;
	}
	
	/**
	 * Rcupration d'un sous-dossier
	 * @param id L'identifiant du sous-dossier
	 * @return Le sous-dossier
	 * @throws DataException
	 */
	public SubFolder getSubFolder(String id) throws DataException {
		LogService.log(nivlog, "DataReadManager::getSubFolder()");
		SubFolder res = new SubFolder();
		
		String sql = "SELECT * FROM INT_DOSSIER WHERE ID_DOS = '" + id + "'";
		Query query = getQuery();
		try {
			query.setSql(sql);
			query.select();
			ResultSet result = query.getRs();
			if(!result.next()) {
				LogService.log(nivlog, "DataReadManager::getSubFolder() : Aucun dossier ayant pour ID = " + id);
				close();
				throw new DataException("Impossible d'accder  ce dossier");
			}
			res.setId(id);
			res.setIntranet(result.getString("ID_INT"));
			res.setName(result.getString("NOM_DOS"));
			res.setParent(result.getString("ID_PAR"));
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataReadManager::getSubFolder() : " + e);
			LogService.log(Priority.ERROR, "DataReadManager::getIntranet() : SQL : " + sql);
			close();
			throw new DataException("Impossible d'accder  ce dossier");
		}
		close();
		return res;
	}

	/**
	 * Rcupration des sous-dossiers
	 * @param id L'identifiant du dossier parent
	 * @return La liste des sous-dossiers
	 * @throws DataException
	 */
	public Collection getSubFolders(String id) throws DataException {
		LogService.log(nivlog, "DataReadManager::getSubFolders()");
		Collection res = new Vector();
		
		String sql = "SELECT * FROM INT_DOSSIER WHERE ID_PAR = '" + id + "' ORDER BY NOM_DOS";
		
		Query query = getQuery();
		try {
			query.setSql(sql);
			query.select();
			ResultSet result = query.getRs();
			while(result.next()) {
				SubFolder tmp = new SubFolder();
				tmp.setId(result.getString("ID_DOS"));
				tmp.setIntranet(result.getString("ID_INT"));
				tmp.setName(result.getString("NOM_DOS"));
				tmp.setParent(result.getString("ID_PAR"));
				res.add(tmp);
			}
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataReadManager::getSubFolders() : " + e);
			LogService.log(Priority.ERROR, "DataReadManager::getIntranet() : SQL : " + sql);
			close();
			throw new DataException("Erreur lors de la lecture des sous dossiers");
		}
		close();
		return res;
	}

	
	/************************************************************
	 *	Documents												*
	 ************************************************************/
	
	/**
	 * Rcupration d'un document
	 * @param intranet L'identifiant de l'intranet auquel appartient le document
	 * @param id L'identifiant du dossier auquel appartient le document
	 * @return Le document
	 * @throws DataException
	 */
	public Document getDocument(String intranet, String id) throws DataException {
		LogService.log(nivlog, "DataReadManager::getDocument()");
		Document res = new Document();
		
		String sql = "SELECT * FROM INT_DOCUMENT WHERE ID_DOC = '" + id + "' AND ID_INT = '" + intranet + "'";
		
		Query query = getQuery();
		try {
			query.setSql(sql);
			query.select();
			ResultSet result = query.getRs();
			if(!result.next()) {
				LogService.log(nivlog, "DataReadManager::getDocument() : Aucun document ayant pour ID = " + id);
				close();
				throw new DataException("Impossible d'accder  ce document");
			}
			res.setId(id);
			res.setIntranet(intranet);
			res.setFolder(result.getString("ID_DOS"));
			if(result.getString("TYPE_DOC").equals("F")) {
				res.setType(Document.FILE);
			}
			else {
				res.setType(Document.LINK);
			}
			res.setName(result.getString("NOM_DOC"));
			res.setFile(result.getString("FILE_DOC"));
			res.setDescription(result.getString("DES_DOC"));
			res.setAuthor(result.getString("AUT_DOC"));
			res.setModificationDate(result.getString("DATE_DOC"));
			
			if(result.getString("LOCK_DOC").equals("1")) {
				res.setLocked(true);
			}
			else {
				res.setLocked(false);
			}
			if(result.getString("NEW_DOC").equals("1")) {
				res.setNew(true);
			}
			else {
				res.setNew(false);
			}
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataReadManager::getDocument() : " + e);
			LogService.log(Priority.ERROR, "DataReadManager::getDocument() : SQL : " + sql);
			close();
			throw new DataException("Erreur lors de la lecture du document");
		}
		close();
		return res;
	}

	/**
	 * Rcupration des documents d'un dossier
	 * @param id L'identiant du dossier
	 * @return La liste des documents
	 * @throws DataException
	 */
	public Collection getDocuments(String id) throws DataException {
		LogService.log(nivlog, "DataReadManager::getDocuments()");
		Collection res = new Vector();
		
		String sql = "SELECT * FROM INT_DOCUMENT WHERE ID_DOS = '" + id + "' ORDER BY NOM_DOC";
		
		Query query = getQuery();
		try {
			query.setSql(sql);
			query.select();
			ResultSet result = query.getRs();
			while(result.next()) {
				Document tmp = new Document();
				tmp.setId(result.getString("ID_DOC"));
				tmp.setIntranet(result.getString("ID_INT"));
				tmp.setFolder(id);
				if(result.getString("TYPE_DOC").equals("F")) {
					tmp.setType(Document.FILE);
				}
				else {
					tmp.setType(Document.LINK);
				}
				tmp.setName(result.getString("NOM_DOC"));
				tmp.setFile(result.getString("FILE_DOC"));
				tmp.setDescription(result.getString("DES_DOC"));
				tmp.setAuthor(result.getString("AUT_DOC"));
				tmp.setModificationDate(result.getString("DATE_DOC"));
				if(result.getString("LOCK_DOC").equals("1")) {
					tmp.setLocked(true);
				}
				else {
					tmp.setLocked(false);
				}
				if(result.getString("NEW_DOC").equals("1")) {
					tmp.setNew(true);
				}
				else {
					tmp.setNew(false);
				}
				res.add(tmp);
			}
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataReadManager::getDocuments() : " + e);
			LogService.log(Priority.ERROR, "DataReadManager::getIntranet() : SQL : " + sql);
			close();
			throw new DataException("Erreur lors de la lecture des documents");
		}
		close();
		return res;
	}

	
	/************************************************************
	 *	Recherche												* 
	 ************************************************************/
	
	/**
	 * Recherche de documents par nom / description
	 * @param intranet L'intranet concern
	 * @param name Tout ou partie du nom du document
	 * @param description Tout ou partie de la description du document
	 * @param valid La validit des documents recherchs
	 * @return La liste des documents correspondants au(x) critre(s) de recherche
	 */
	public Collection findByName(String intranet, String name, String description, boolean valid) throws DataException {
	    LogService.log(nivlog, "DataReadManager::findbyName()");
	    
	    Vector res = new Vector();
	    StringBuffer sql = new StringBuffer();
	    sql.append("SELECT INT_DOCUMENT.* FROM INT_DOCUMENT, INT_DOSSIER ");
	    sql.append("WHERE INT_DOCUMENT.ID_DOS = INT_DOSSIER.ID_DOS ");
	    sql.append("AND INT_DOSSIER.ID_INT = '" + intranet + "' ");
	    sql.append("AND NOM_DOC LIKE '%" + RDBMServices.sqlEscape(name) + "%' ");
	    sql.append("AND DES_DOC LIKE '%" + RDBMServices.sqlEscape(description) + "%' ");
	    if(valid) {
	        sql.append("AND ID_DOC LIKE 'v%'");
	    }
	    else {
	        sql.append("AND (NEW_DOC = '1' OR ID_DOC LIKE 'v%'");
	    }
	    
	    Query query = getQuery();
	    try {
	        query.setSql(sql.toString());
	        query.select();
	        ResultSet result = query.getRs();
	        while(result.next()) {
	            Document tmp = new Document();
	            tmp.setId(result.getString("ID_DOC"));
	            tmp.setIntranet(intranet);
	            tmp.setFolder(result.getString("ID_DOS"));
				if(result.getString("TYPE_DOC").equals("F")) {
					tmp.setType(Document.FILE);
				}
				else {
					tmp.setType(Document.LINK);
				}
				tmp.setName(result.getString("NOM_DOC"));
				tmp.setFile(result.getString("FILE_DOC"));
				tmp.setDescription(result.getString("DES_DOC"));
				tmp.setAuthor(result.getString("AUT_DOC"));
				tmp.setModificationDate(result.getString("DATE_DOC"));
				
				if(result.getString("LOCK_DOC").equals("1")) {
					tmp.setLocked(true);
				}
				else {
					tmp.setLocked(false);
				}
				if(result.getString("NEW_DOC").equals("1")) {
					tmp.setNew(true);
				}
				else {
					tmp.setNew(false);
				}
				res.add(tmp);
	        }
	    }
	    catch(SQLException e) {
	        LogService.log(Priority.ERROR, "DataReadManager::findByName() : " + e);
			LogService.log(Priority.ERROR, "DataReadManager::findByName() : SQL : " + sql);
			close();
			throw new DataException("Erreur lors de la recherche de documents");
	    }
	    close();
	    return res;
	}
	
	/**
	 * Recherche de tous les documents en attente de validation
	 * @param intranet L'intranet concern
	 * @return La liste des documents en attente de validation
	 * @throws DataException
	 */
	public Collection findInvalid(String intranet, String name, String description) throws DataException {
	    LogService.log(nivlog, "DataReadManager::findInvalid()");
	    
	    Vector res = new Vector();
	    StringBuffer sql = new StringBuffer();
	    sql.append("SELECT INT_DOCUMENT.* FROM INT_DOCUMENT, INT_DOSSIER ");
	    sql.append("WHERE INT_DOCUMENT.ID_DOS = INT_DOSSIER.ID_DOS ");
	    sql.append("AND INT_DOSSIER.ID_INT = '" + intranet + "' ");sql.append("AND NOM_DOC LIKE '%" + RDBMServices.sqlEscape(name) + "%' ");
	    sql.append("AND DES_DOC LIKE '%" + RDBMServices.sqlEscape(description) + "%' ");
	    sql.append("AND (NEW_DOC = '1' OR LOCK_DOC = '1'");
	    
	    Query query = getQuery();
	    try {
	        query.setSql(sql.toString());
	        query.select();
	        ResultSet result = query.getRs();
	        while(result.next()) {
	            Document tmp = new Document();
	            tmp.setId(result.getString("ID_DOC"));
	            tmp.setIntranet(intranet);
	            tmp.setFolder(result.getString("ID_DOS"));
				if(result.getString("TYPE_DOC").equals("F")) {
					tmp.setType(Document.FILE);
				}
				else {
					tmp.setType(Document.LINK);
				}
				tmp.setName(result.getString("NOM_DOC"));
				tmp.setFile(result.getString("FILE_DOC"));
				tmp.setDescription(result.getString("DES_DOC"));
				tmp.setAuthor(result.getString("AUT_DOC"));
				tmp.setModificationDate(result.getString("DATE_DOC"));
				
				if(result.getString("LOCK_DOC").equals("1")) {
					tmp.setLocked(true);
				}
				else {
					tmp.setLocked(false);
				}
				if(result.getString("NEW_DOC").equals("1")) {
					tmp.setNew(true);
				}
				else {
					tmp.setNew(false);
				}
				res.add(tmp);
	        }
	    }
	    catch(SQLException e) {
	        LogService.log(Priority.ERROR, "DataReadManager::findInvalid() : " + e);
			LogService.log(Priority.ERROR, "DataReadManager::findInvalid() : SQL : " + sql);
			close();
			throw new DataException("Erreur lors de la recherche de documents");
	    }
	    close();
	    return res;
	}
	
	/**
	 * Recherche des documents dposs par un utilisateur en attente de validation
	 * @param intranet L'intranet concern
	 * @param login L'identifiant de l'utilisateur
	 * @return La liste des documents de l'utilisateur en attente de validation
	 * @throws DataException
	 */
	public Collection findMyInvalid(String intranet, String name, String description, String login) throws DataException {
	    LogService.log(nivlog, "DataReadManager::findMyInvalid()");
	    
	    Vector res = new Vector();
	    StringBuffer sql = new StringBuffer();
	    sql.append("SELECT INT_DOCUMENT.* FROM INT_DOCUMENT, INT_DOSSIER ");
	    sql.append("WHERE INT_DOCUMENT.ID_DOS = INT_DOSSIER.ID_DOS ");
	    sql.append("AND INT_DOSSIER.ID_INT = '" + intranet + "' ");
	    sql.append("AND NOM_DOC LIKE '%" + RDBMServices.sqlEscape(name) + "%' ");
	    sql.append("AND DES_DOC LIKE '%" + RDBMServices.sqlEscape(description) + "%' ");
	    sql.append("AND ID_DOC LIKE 'i%' AND AUT_DOC = '" + login + "'");

	    Query query = getQuery();
	    try {
	        query.setSql(sql.toString());
	        query.select();
	        ResultSet result = query.getRs();
	        while(result.next()) {
	            Document tmp = new Document();
	            tmp.setId(result.getString("ID_DOC"));
	            tmp.setIntranet(intranet);
	            tmp.setFolder(result.getString("ID_DOS"));
				if(result.getString("TYPE_DOC").equals("F")) {
					tmp.setType(Document.FILE);
				}
				else {
					tmp.setType(Document.LINK);
				}
				tmp.setName(result.getString("NOM_DOC"));
				tmp.setFile(result.getString("FILE_DOC"));
				tmp.setDescription(result.getString("DES_DOC"));
				tmp.setAuthor(result.getString("AUT_DOC"));
				tmp.setModificationDate(result.getString("DATE_DOC"));
				
				if(result.getString("LOCK_DOC").equals("1")) {
					tmp.setLocked(true);
				}
				else {
					tmp.setLocked(false);
				}
				if(result.getString("NEW_DOC").equals("1")) {
					tmp.setNew(true);
				}
				else {
					tmp.setNew(false);
				}
				res.add(tmp);
	        }
	    }
	    catch(SQLException e) {
	        LogService.log(Priority.ERROR, "DataReadManager::findMyInvalid() : " + e);
			LogService.log(Priority.ERROR, "DataReadManager::findMyInvalid() : SQL : " + sql);
			close();
			throw new DataException("Erreur lors de la recherche de documents");
	    }
	    close();
	    return res;
	}
	
	
	/************************************************************
	 *	Types MIME												*
	 ************************************************************/
	
	/**
	 * Rcupration des types MIME valides
	 * @return La liste des types MIME
	 * @throws DataException
	 */
	public Collection getMIMEs() throws DataException {
		LogService.log(nivlog, "DataReadManager::getMIMEs()");
		Collection res = new Vector();
		
		String sql = "SELECT * FROM INT_MIME ORDER BY EXT_MIME ASC";
		
		Query query = getQuery();
		try {
			query.setSql(sql);
			query.select();
			ResultSet result = query.getRs();
			while(result.next()) {
				Mime tmp = new Mime();
				tmp.setExtension(result.getString("EXT_MIME"));
				tmp.setType(result.getString("TYPE_MIME"));
				res.add(tmp);
			}
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataReadManager::getMIMEs() : " + e);
			LogService.log(Priority.ERROR, "DataReadManager::getIntranet() : SQL : " + sql);
			close();
			throw new DataException("Erreur lors de la lecture des types MIME");
		}
		close();
		return res;
	}

	
	/************************************************************
	 *	Valideurs												*
	 ************************************************************/
	
	/**
	 * Rcupration des valideurs d'un dossier
	 * @param id L'identifiant du dossier
	 * @return La liste des valideurs
	 * @throws DataException
	 */
	public Collection getValidators(String id) throws DataException {
		LogService.log(nivlog, "DataReadManager::getValidators()");
		Collection res = new Vector();
		
		String sql = "SELECT LOGIN FROM INT_VALIDEUR WHERE ID_DOS = '" + id + "'";
		
		Query query = getQuery();
		try {
			query.setSql(sql);
			query.select();
			ResultSet result = query.getRs();
			while(result.next()) {
				res.add(result.getString("LOGIN"));
			}
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataReadManager::getValidators() : " + e);
			LogService.log(Priority.ERROR, "DataReadManager::getIntranet() : SQL : " + sql);
			close();
			throw new DataException("Erreur lors de la lecture des valideurs");
		}
		close();
		return res;
	}

	
	/************************************************************
	 *	Publicateurs											*
	 ************************************************************/
	
	/**
	 * Rcupration des publicateurs d'un dossier
	 * @param id L'identifiant du dossier
	 * @return La liste des publicateurs
	 * @throws DataException
	 */
	public Collection getUsersPublicators(String id) throws DataException {
		LogService.log(nivlog, "DataReadManager::getUsersPublicators()");
		Collection res = new Vector();
		
		String sql = "SELECT ID_PUB FROM INT_PUBLICATEUR WHERE TYPE_PUB = 'U' AND ID_DOS = '" + id + "'";
		
		Query query = getQuery();
		try {
			query.setSql(sql);
			query.select();
			ResultSet result = query.getRs();
			while(result.next()) {
				res.add(result.getString("ID_PUB"));
			}
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataReadManager::getUsersPublicators() : " + e);
			LogService.log(Priority.ERROR, "DataReadManager::getIntranet() : SQL : " + sql);
			close();
			throw new DataException("Erreur lors de la lecture des publicateurs");
		}
		close();
		return res;
	}

	/**
	 * Rcupration des groupes de publicateurs d'un dossier
	 * @param id L'identifiant du dossier
	 * @return La liste des groupes de publicateurs
	 * @throws DataException
	 */
	public Collection getGroupsPublicators(String id) throws DataException {
		LogService.log(nivlog, "DataReadManager::getGroupsPublicators()");
		Collection res = new Vector();
		
		String sql = "SELECT ID_PUB FROM INT_PUBLICATEUR WHERE TYPE_PUB = 'G' AND ID_DOS = '" + id + "'";
		
		Query query = getQuery();
		try {
			query.setSql(sql);
			query.select();
			ResultSet result = query.getRs();
			while(result.next()) {
				res.add(result.getString("ID_PUB"));
			}
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataReadManager::getGroupsPublicators() : " + e);
			LogService.log(Priority.ERROR, "DataReadManager::getIntranet() : SQL : " + sql);
			close();
			throw new DataException("Erreur lors de la lecture des groupes de publicateurs");
		}
		close();
		return res;
	}

	
	/************************************************************
	 *	Divers													*
	 ************************************************************/
	
	/**
	 * Rcupration du chemin d'un dossier
	 * @param id L'identifiant du dossier
	 * @return Le chemin du dossier
	 */
	public Path getPath(String id) throws DataException {
		LogService.log(nivlog, "DataReadManager::getPath()");
		Path cachedpath = (Path)FOLDER_PATH.get(id);
		if(cachedpath != null) {
			LogService.log(nivlog, "DataReadManager::getPath() : Trouv dans le cache");
			return cachedpath;
		}
		
		LinkedList path = new LinkedList();
		String currentId = id;
		SubFolder tmp = null;
		do {
			tmp = getSubFolder(currentId);
			currentId = tmp.getParent();
			path.addFirst(tmp);
		}
		while(!currentId.equals("0"));
		Path res = new Path(path);
		FOLDER_PATH.put(id, res);
		return res;
	}
	
	
	/************************************************************
	 *	Fonctionnement interne									*
	 ************************************************************/
	
	/**
	 * Retourne la connexion  la base de donnes
	 * @return La connexion
	 */
	private Query getQuery() {
		LogService.log(nivlog, "DataReadManager::getQuery()");
		if (requests == 0) {
			query = Config.getInstance().getConnexionDefault();
		}
		requests++;
		return query;
	}

	/**
	 * Ferme la connexion courante
	 */
	private void close() {
		requests--;
		if(requests == 0) {
			LogService.log(nivlog, "DataReadManager::close() : CLOSE");
			query.close();
			query = null;
		}
	}

	/**
	 * Vrifie lors de la destruction de l'objet que la connexion est bien ferme
	 */
	protected void finalize() throws Throwable {
		super.finalize();
		if(query != null) {
			LogService.log(Priority.FATAL, "DataReadManager::finalize() : Des connexions n'ont pas t fermes");
		}
	}
}