package org.esupportail.portal.channels.CIntranet.data;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Iterator;
import java.util.Vector;

import org.apache.log4j.Priority;
import org.esupportail.portal.channels.CIntranet.beans.Document;
import org.esupportail.portal.channels.CIntranet.beans.Folder;
import org.esupportail.portal.channels.CIntranet.beans.Intranet;
import org.esupportail.portal.channels.CIntranet.beans.Mime;
import org.esupportail.portal.channels.CIntranet.beans.Population;
import org.esupportail.portal.channels.CIntranet.beans.SubFolder;
import org.esupportail.portal.channels.CIntranet.config.Config;
import org.esupportail.portal.utils.database.Query;
import org.jasig.portal.RDBMServices;
import org.jasig.portal.services.LogService;

/**
 * DataWriteManager<br>
 * <br>
 * Implmentation d'un IDataWriteManager utilisant des instructions<br>
 * SQL standard<br>
 * L'utilisation d'instructions spcifiques  un SGBD doit tre<br>
 * ralise dans une autre implmentation<br>
 * <br>
 * (c)Copyright <a href="http://www.esup-portail.org">ESup-Portail 2004</a><br>
 * @author <a href="mailto:mathieu.larchet@univ-nancy2.fr">Mathieu Larchet</a>
 * @version 1.3
 * 
 */

public class DataWriteManager implements IDataWriteManager {

	private Priority nivlog = Priority.INFO;
	private int transactions = 0;
	private int requests = 0;
	private Query query = null;

	/**
	 * Constructeur
	 */
	public DataWriteManager() {
	}

	/************************************************************
	 *	Intranets												*
	 ************************************************************/
	
	/**
	 * Cration d'un intranet
	 * @param intranet L'intranet  crer
	 */
	public void createIntranet(Intranet intranet) throws DataException {
		LogService.log(nivlog, "DataWriteManager::createIntranet()");
		
		Query query = getQuery();
		
		// Vrification existance intranet
		String sql1 = "SELECT * FROM INT_INTRANET WHERE ID_INT = '" + intranet.getId() + "'";
		try {
		    query.setSql(sql1);
		    query.select();
		    ResultSet result = query.getRs();
		    if(result.next()) {
				LogService.log(Priority.ERROR, "DataWriteManager::createIntranet() Intranet existant " + intranet.getId());
				close();
				throw new DataException("Un intranet possde dj cet identifiant");
			}
		}
		catch(SQLException e) {
		    LogService.log(Priority.ERROR, "DataWriteManager::createIntranet() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::createIntranet() : SQL : " + sql1);
			close();
			throw new DataException("Erreur lors de la cration de l'intranet");
		}

		// Cration du dossier racine
		Folder folder = new Folder();
		folder.setIntranet(intranet.getId());
		folder.setName(intranet.getName());
		folder.setParent("0");
		folder.setValidation(true);
		
		// Dbut de transaction
		begin();
		try {
			createFolder(folder);
		}
		catch(DataException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::createIntranet() : " + e);
			rollback();
			throw new DataException("Erreur lors de la cration du dossier racine");
		}
		
		// Cration intranet
		StringBuffer sql = new StringBuffer();
		sql.append("INSERT INTO INT_INTRANET(ID_INT, NOM_INT, DOC_INT, ID_RES) VALUES(");
		sql.append("'" + intranet.getId() + "', ");
		sql.append("'" + RDBMServices.sqlEscape(intranet.getName()) + "', ");
		sql.append("1, ");
		sql.append("'" + intranet.getRessource() + "')");		
		
		try {
			query.setSql(sql.toString());
			query.insert();
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::createIntranet() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::getIntranet() : SQL : " + sql);
			rollback();
			throw new DataException("Erreur lors de la cration de l'intranet");
		}
	}

	/**
	 * Suppression d'un intranet
	 * @param intranet L'intranet  supprimer
	 * @throws DataException
	 */
	public void removeIntranet(Intranet intranet) throws DataException {
		LogService.log(nivlog, "DataWriteManager::removeIntranet()");
		
		// Rcupration de tous les dossiers
		Iterator i = getFolderChildren(intranet.getRoot()).iterator();
		
		Query query = getQuery();
		String sql = new String();
		begin();
		try {
			// Suppression de tous les documents, valideurs, publicateurs
			while(i.hasNext()) {
				SubFolder tmp = (SubFolder)i.next();
				sql = "DELETE FROM INT_DOCUMENT WHERE ID_DOS = " + tmp.getId();
				query.setSql(sql);
				query.delete();
				
				removeAllValidators(tmp.getId());
				removeAllPublicators(tmp.getId());
			}
			
			// Suppression des documents de la racine et des valideurs, publicateurs
			sql = "DELETE FROM INT_DOCUMENT WHERE ID_DOS = " + intranet.getRoot();
			query.setSql(sql);
			query.delete();
			
			removeAllValidators(intranet.getRoot());
			removeAllPublicators(intranet.getRoot());
			
			// Suppression de tous les dossiers
			sql = "DELETE FROM INT_DOSSIER WHERE ID_INT = '" + intranet.getId() + "'";
			query.setSql(sql);
			query.delete();
			
			// Suppression de la population
			removePopulation(intranet.getId());
			
			// Suppression de l'intranet
			sql = "DELETE FROM INT_INTRANET WHERE ID_INT = '" + intranet.getId() + "'";
			query.setSql(sql);
			query.delete();
		}
		catch(DataException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::removeIntranet() : " + e);
			rollback();
			throw new DataException("Impossible de supprimer cet intranet");
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::removeIntranet() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::removeIntranet() : SQL : " + sql);
			rollback();
			throw new DataException("Impossible de supprimer cet intranet");
		}
	}

	
	/************************************************************
	 *	Populations												*
	 ************************************************************/

	/**
	 * Ajout d'une population
	 * @param population La population  ajouter
	 */
	public void addPopulation(Population population) throws DataException {
		LogService.log(nivlog, "DataWriteManager::getPopulation()");
		
		StringBuffer sql = new StringBuffer();
		sql.append("INSERT INTO INT_POPULATION(ID_INT, TYPE_POP, ID_POP) VALUES(");
		sql.append("'" + population.getIntranet() + "', ");
		sql.append("'%T', ");
		sql.append("'%I')");
		
		Query query = getQuery();
		begin();
		try {
			Iterator i = population.getUsers().iterator();
			String req = sql.toString().replaceFirst("%T", "U");
			while(i.hasNext()) {
				String tmp = (String)i.next();
				String reqtmp = req.replaceFirst("%I", tmp);
				query.setSql(reqtmp);
				query.insert();
			}
			
			i = population.getGroups().iterator();
			req = sql.toString().replaceFirst("%T", "G");
			while(i.hasNext()) {
				String tmp = (String)i.next();
				String reqtmp = req.replaceFirst("%I", tmp);
				query.setSql(reqtmp);
				query.insert();
			}
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::addPopulation() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::getIntranet() : SQL : " + sql);
			rollback();
			throw new DataException("Erreur lors de l'ajout de la population");
		}
		commit();
	}

	/**
	 * Suppression d'une population
	 * @param intranet L'intranet dont on veut supprimer la population
	 * @throws DataException
	 */
	public void removePopulation(String intranet) throws DataException {
		LogService.log(nivlog, "DataWriteManager::removePopulation()");
		
		StringBuffer sql = new StringBuffer();
		sql.append("DELETE FROM INT_POPULATION WHERE ");
		sql.append("ID_INT = '" + intranet + "'");
		
		Query query = getQuery();
		try {
			query.setSql(sql.toString());
			query.delete();
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::removePopulation() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::getIntranet() : SQL : " + sql);
			close();
			throw new DataException("Erreur lors de la suppression de la population");
		}
		close();
	}

	
	/************************************************************
	 *	Super Utilisateurs										*
	 ************************************************************/

	/**
	 * Ajout d'un super-utilisateur
	 * @param login L'identifiant de l'utilisateur
	 */
	public void addSuperUser(String login) throws DataException {
		LogService.log(nivlog, "DataWriteManager::addSuperUser()");
		
		StringBuffer sql = new StringBuffer();
		sql.append("INSERT INTO INT_SUPERUSER(LOGIN) VALUES(");
		sql.append("'" + login + "')");
		
		Query query = getQuery();
		try {
			query.setSql(sql.toString());
			query.insert();
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::addSuperUser() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::getIntranet() : SQL : " + sql);
			close();
			throw new DataException("Erreur lors de l'ajout du super utilisateur");
		}
		close();
	}

	/**
	 * Suppression d'un super-utilisateur
	 * @param login L'identifiant de l'utilisateur
	 * @throws DataException
	 */
	public void removeSuperUser(String login) throws DataException {
		LogService.log(nivlog, "DataWriteManager::removeSuperUser()");
		
		StringBuffer sql = new StringBuffer();
		sql.append("DELETE FROM INT_SUPERUSER WHERE ");
		sql.append("LOGIN = '" + login + "'");
		
		Query query = getQuery();
		try {
			query.setSql(sql.toString());
			query.delete();
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::removeSuperUser() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::getIntranet() : SQL : " + sql);
			close();
			throw new DataException("Erreur lors de la suppression du super utilisateur");
		}
		close();
	}

	
	/************************************************************
	 *	Dossiers												*
	 ************************************************************/
	
	/**
	 * Cration d'un dossier
	 * @param folder Le dossier  crer
	 * @return L'identifiant du dossier cr
	 */
	public String createFolder(Folder folder) throws DataException {
		LogService.log(nivlog, "DataWriteManager::createFolder()");
		
		// Vrification existance dossier
		String sql = "SELECT * FROM INT_DOSSIER WHERE ID_PAR = " + folder.getParent() + " AND NOM_DOS = '" + folder.getName() + "'";
		Query query = getQuery();
		try {
			query.setSql(sql);
			query.select();
			ResultSet result = query.getRs();
			if(result.next()) {
				LogService.log(Priority.ERROR, "DataWriteManager::createFolder() Dossier existant " + folder.getName());
				close();
				throw new DataException("Un dossier portant ce nom existe dj");
			}
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::createFolder() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::createFolder() : SQL : " + sql);
			close();
			throw new DataException("Erreur lors de la cration du dossier");
		}
		close();
		
		// Rcupration identifiant
		int id = getNextKey("INT_DOSSIER");
		if(id == 0) {
			LogService.log(Priority.ERROR, "DataWriteManager::createFolder() Erreur rcupration de la cl");
			throw new DataException("Erreur lors de la cration du dossier");
		}
		folder.setId(new Integer(id).toString());
		StringBuffer sql2 = new StringBuffer();
		sql2.append("INSERT INTO INT_DOSSIER(ID_DOS, ID_INT, ID_PAR, NOM_DOS, VAL_DOS) VALUES(");
		sql2.append(id + ", ");
		sql2.append("'" + folder.getIntranet() + "', ");
		sql2.append(folder.getParent() + ", ");
		sql2.append("'" + RDBMServices.sqlEscape(folder.getName()) + "', ");
		if(folder.isValidation()) {
			sql2.append("'1')");
		}
		else {
			sql2.append("'0')");
		}
		query = getQuery();
		begin();
		try {
			query.setSql(sql2.toString());
			query.insert();
			
			// Propagation valideurs
			if(!folder.isRoot()) {
			    // Rcupration IDataReadManager
			    IDataReadManager data = DataFactory.makeRead();
			    
				Iterator i = data.getValidators(folder.getParent()).iterator();
				while(i.hasNext()) {
					String tmp = (String)i.next();
					addValidator(folder.getId(), tmp, false, false);
				}
			
				// Propagation publicateurs
				i = data.getUsersPublicators(folder.getParent()).iterator();
				while(i.hasNext()) {
					String tmp = (String)i.next();
					addPublicator(folder.getId(), tmp, false);
				}
				
				// Propagation groupes de publicateurs
				i = data.getGroupsPublicators(folder.getParent()).iterator();
				while(i.hasNext()) {
					String tmp = (String)i.next();
					addPublicators(folder.getId(), tmp, false);
				}
			}
			
		}
		catch(DataException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::createFolder() : " + e);
			rollback();
			throw new DataException("Erreur lors de la cration du dossier");
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::createFolder() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::createFolder() : SQL : " + sql2);
			rollback();
			throw new DataException("Erreur lors de la cration du dossier");
		}
		commit();
		return new Integer(id).toString();
	}

	/**
	 * Suppression d'un dossier
	 * @param id L'identifiant du dossier
	 * @throws DataException
	 */
	public void removeFolder(String id) throws DataException {
		LogService.log(nivlog, "DataWriteManager::removeFolder()");
		
		String sql = "SELECT * FROM INT_DOSSIER WHERE ID_PAR = " + id;
		Query query = getQuery();
		try {
			// Test prsence sous dossiers
			query.setSql(sql);
			query.select();
			ResultSet result = query.getRs();
			if(result.next()) {
				LogService.log(nivlog, "DataWriteManager::removeFolder() : Dossier non vide");
				close();
				throw new DataException("Suppression d'un dossier non vide");
			}
			
			// Test prsence documents
			sql = "SELECT * FROM INT_DOCUMENT WHERE ID_DOS = " + id;
			query.setSql(sql);
			query.select();
			result = query.getRs();
			if(result.next()) {
				LogService.log(nivlog, "DataWriteManager::removeFolder() : Dossier non vide");
				close();
				throw new DataException("Suppression d'un dossier non vide");
			}
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::removeFolder() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::removeFolder() : SQL : " + sql);
			close();
			throw new DataException("Erreur lors de la suppression du dossier");
		}
		
		begin();
		try {
			// Suppression dossier
			sql = "DELETE FROM INT_DOSSIER WHERE ID_DOS = " + id;
			query.setSql(sql);
			query.delete();
			
			// Suppression valideurs
			removeAllValidators(id);
			removeAllPublicators(id);
		}
		catch(DataException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::removeFolder() : " + e);
			rollback();
			throw new DataException("Erreur lors de la suppression du dossier");
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::removeFolder() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::removeFolder() : SQL : " + sql);
			rollback();
			throw new DataException("Erreur lors de la suppression du dossier");
		}
		commit();
	}

	/**
	 * Mise  jour d'un dossier
	 * @param folder L'identifiant du dossier
	 * @throws DataException
	 */
	public void updateFolder(Folder folder) throws DataException {
		LogService.log(nivlog, "DataWriteManager::updateFolder()");
		
		// Vrification du nom
		String sql = "SELECT * FROM INT_DOSSIER WHERE ID_PAR = " + folder.getParent() + " AND NOM_DOS = '" + RDBMServices.sqlEscape(folder.getName()) + "' AND ID_DOS != " + folder.getId();
		Query query = getQuery();
		try {
			query.setSql(sql);
			query.select();
			ResultSet result = query.getRs();
			if(result.next()) {
				LogService.log(Priority.ERROR, "DataWriteManager::updateFolder() : Un dossier portant ce nom existe dj");
				query.close();
				throw new DataException("Un dossier portant ce nom existe dj");
			}
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::updateFolder() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::updateFolder() : SQL : " + sql);
			query.close();
			throw new DataException("Erreur lors de la modification du dossier");
		}
		
		if(folder.isValidation()) {
			sql = "UPDATE INT_DOSSIER SET NOM_DOS = '" + RDBMServices.sqlEscape(folder.getName()) + "', VAL_DOS = '1' WHERE ID_DOS = " + folder.getId();
		}
		else {
			sql = "UPDATE INT_DOSSIER SET NOM_DOS = '" + RDBMServices.sqlEscape(folder.getName()) + "', VAL_DOS = '0' WHERE ID_DOS = " + folder.getId();
		}
		
		begin();
		try {
			query.setSql(sql);
			query.update();
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::updateFolder() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::updateFolder() : SQL : " + sql);
			rollback();
			throw new DataException("Erreur lors de la mise  jour du dossier");
		}
		commit();
	}

	
	/************************************************************
	 *	Documents												*
	 ************************************************************/
	
	/**
	 * Cration d'un document
	 * @param doc Le document  crer
	 * @param valid Si le document est valide true, false sinon
	 */
	public void createDocument(Document doc, boolean valid) throws DataException {
		LogService.log(nivlog, "DataWriteManager::createDocument()");
					
		// Vrification existance fichier
		String sql1 = "SELECT * FROM INT_DOCUMENT WHERE NOM_DOC = '" + RDBMServices.sqlEscape(doc.getName()) + "' AND ID_DOS = '" + doc.getFolder() + "'";
		Query query = getQuery();
		try {
			query.setSql(sql1);
			query.select();
			ResultSet result = query.getRs();
			if(result.next()) {
				LogService.log(Priority.ERROR, "Le document existe dj");
				close();
				throw new DataException("Un document portant ce nom existe dj");
			}
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::createDocument() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::createDocument() : SQL : " + sql1);
			close();
			throw new DataException("Erreur lors de la cration du document");
		}

		// Rcupration identifiant
		int id = getNextDocument(doc.getIntranet());
		if(id == 0) {
			LogService.log(Priority.ERROR, "DataWriteManager::createDocument() Erreur rcupration de la cl");
			throw new DataException("Erreur lors de la cration du document");
		}
		
		if(valid) {
			doc.setId("v" + id);
		}
		else {
			doc.setId("i" + id);
		}
		
		StringBuffer sql = new StringBuffer();
		sql.append("INSERT INTO INT_DOCUMENT(ID_DOC, ID_INT, ID_DOS, NOM_DOC, DES_DOC, TYPE_DOC, FILE_DOC, AUT_DOC, DATE_DOC, LOCK_DOC, EXT_DOC, NEW_DOC) VALUES(");
		sql.append("'" + doc.getId() + "', ");
		sql.append("'" + doc.getIntranet() + "', ");
		sql.append(doc.getFolder() + ", ");
		sql.append("'" + RDBMServices.sqlEscape(doc.getName()) + "', ");
		sql.append("'" + RDBMServices.sqlEscape(doc.getDescription()) + "', ");
		if(doc.getType() == Document.FILE) {
			sql.append("'F', ");
		}
		else {
			sql.append("'L', ");
		}
		sql.append("'" + RDBMServices.sqlEscape(doc.getFile()) + "', ");
		sql.append("'" + doc.getAuthor() + "', ");
		sql.append("'" + doc.getModificationDate() + "', ");
		if(doc.isLocked()) {
			sql.append("'1', ");
		}
		else {
			sql.append("'0', ");
		}
		sql.append("'" + doc.getExtension() + "', ");
		if(doc.isNew()) {
			sql.append("'1')");
		}
		else {
			sql.append("'0')");
		}
		
		begin();
		try {
			query.setSql(sql.toString());
			query.insert();
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::createDocument() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::createDocument() : SQL : " + sql);
			rollback();
			throw new DataException("Erreur lors de la cration du document");
		}
	}

	/**
	 * Suppression d'un document
	 * @param doc Le document  supprimer
	 * @throws DataException
	 */
	public void removeDocument(Document doc) throws DataException {
		LogService.log(nivlog, "DataWriteManager::removeDocument()");
		
		String sql = null;
		Query query = getQuery();
		begin();
		
		// Test de la prsence d'un document ayant le mme ID mais tant valide
		if(!doc.isValid()) {
			Document tmp = null;
			String id = "v" + doc.getId().substring(1);
			sql = "SELECT * FROM INT_DOCUMENT WHERE ID_DOC = '" + id + "' AND ID_INT = '" + doc.getIntranet() + "'";
			try {
				query.setSql(sql);
				query.select();
				ResultSet result = query.getRs();
				if(result.next()) {
				    sql = "UPDATE INT_DOCUMENT SET LOCK_DOC = '0' WHERE ID_DOC = '" + id + "' AND ID_INT = '" + doc.getIntranet() + "'";
				    query.setSql(sql);
					query.update();
				}
			}
			catch(SQLException e) {
			    LogService.log(Priority.ERROR, "DataWriteManager::removeDocument() : " + e);
				LogService.log(Priority.ERROR, "DataWriteManager::removeDocument() : SQL : " + sql);
				rollback();
				throw new DataException("Erreur lors de la la suppression du document");
			}
		}
		sql = "DELETE FROM INT_DOCUMENT WHERE ID_DOC = '" + doc.getId() + "' AND ID_INT = '" + doc.getIntranet() + "'";
		
		try {
			query.setSql(sql);
			query.delete();
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::removeDocument() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::removeDocument() : SQL : " + sql);
			rollback();
			throw new DataException("Erreur lors de la la suppression du document");
		}
	}

	/**
	 * Mise  jour d'un document
	 * @param doc Le document  mettre  jour
	 * @throws DataException
	 */
	public void updateDocument(Document doc) throws DataException {
		LogService.log(nivlog, "DataWriteManager::updateDocument()");
		
		if(doc.isValid()) {
			StringBuffer sql = new StringBuffer();
			sql.append("UPDATE INT_DOCUMENT SET ");
			sql.append("NOM_DOC = '" + RDBMServices.sqlEscape(doc.getName()) + "', ");
			sql.append("DES_DOC = '" + RDBMServices.sqlEscape(doc.getDescription()) + "', ");
			sql.append("FILE_DOC = '" + RDBMServices.sqlEscape(doc.getFile()) + "', ");
			sql.append("AUT_DOC = '" + doc.getAuthor() + "', ");
			sql.append("DATE_DOC = '" + doc.getModificationDate() + "', ");
			if(doc.getType() == Document.FILE) {
				sql.append("EXT_DOC = '" + doc.getExtension() + "', ");
			}
			if(doc.isLocked()) {
				sql.append("LOCK_DOC = '1' ");
			}
			else {
				sql.append("LOCK_DOC = '0' ");
			}
			sql.append("WHERE ID_DOC = '" + doc.getId() + "' AND ID_INT = '" + doc.getIntranet() + "'");	
		
			Query query = getQuery();
			try {
				query.setSql(sql.toString());
				query.update();
			}
			catch(SQLException e) {
				LogService.log(Priority.ERROR, "DataWriteManager::updateDocument() : " + e);
				LogService.log(Priority.ERROR, "DataWriteManager::updateDocument() : SQL : " + sql);
				close();
				throw new DataException("Erreur lors de la mise  jour du document");
			}
			close();
		}
		else {
			String sql = "UPDATE INT_DOCUMENT SET LOCK_DOC = '1' WHERE ID_DOC = '" + "v" + doc.getId().substring(1) + "' AND ID_INT = '" + doc.getIntranet() + "'";
			Query query = getQuery();
			begin();
			try {
				query.setSql(sql);
				query.update();
			}
			catch(SQLException e) {
				LogService.log("DataWriteManager::updateDocument() : " + e);
				LogService.log("DataWriteManager::updateDocument() : SQL : " + sql);
				rollback();
				throw new DataException("Erreur lors de la mise  jour du document");
			}
			StringBuffer sql2 = new StringBuffer();
			sql2.append("INSERT INTO INT_DOCUMENT(ID_DOC, ID_INT, ID_DOS, NOM_DOC, DES_DOC, TYPE_DOC, FILE_DOC, AUT_DOC, DATE_DOC, LOCK_DOC, EXT_DOC, NEW_DOC) VALUES(");
			sql2.append("'" + doc.getId() + "', ");
			sql2.append("'" + doc.getIntranet() + "', ");
			sql2.append(doc.getFolder() + ", ");
			sql2.append("'" + RDBMServices.sqlEscape(doc.getName()) + "', ");
			sql2.append("'" + RDBMServices.sqlEscape(doc.getDescription()) + "', ");
			if(doc.getType() == Document.FILE) {
				sql2.append("'F', ");
			}
			else {
				sql2.append("'L', ");
			}
			sql2.append("'" + RDBMServices.sqlEscape(doc.getFile()) + "', ");
			sql2.append("'" + doc.getAuthor() + "', ");
			sql2.append("'" + doc.getModificationDate() + "', ");
			sql2.append("'0', ");
			sql2.append("'" + doc.getExtension() + "', ");
			sql2.append("'0')");
			try {
				query.setSql(sql2.toString());
				query.insert();
			}
			catch(SQLException e) {
				LogService.log("DataWriteManager::updateDocument() : " + e);
				LogService.log("DataWriteManager::updateDocument() : SQL : " + sql2.toString());
				rollback();
				throw new DataException("Erreur lors de la mise  jour du document");
			}
		}
	}

	/**
	 * Validation d'un document
	 * @param doc Le document  valider
	 * @throws DataException
	 */
	public void validDocument(Document doc) throws DataException {
		LogService.log(nivlog, "DataWriteManager::validDocument()");
		String sql = "SELECT * FROM INT_DOCUMENT WHERE ID_DOC = '" + doc.getId() + "' AND ID_INT = '" + doc.getIntranet() + "'";
		Query query = getQuery();
		boolean valid = false;
		
		try {
			query.setSql(sql);
			query.select();
			ResultSet result = query.getRs();
			if(result.next()) {
				valid = true;
			}
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::validDocument() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::validDocument() : SQL : " + sql);
			close();
			throw new DataException("Erreur lors de la validation du document");
		}
		
		begin();
		if(valid) {
			sql = "DELETE FROM INT_DOCUMENT WHERE ID_DOC = '" + doc.getId ()+ "' AND ID_INT = '" + doc.getIntranet() + "'";
			try {
				query.setSql(sql);
				query.delete();
			}
			catch(SQLException e) {
				LogService.log(Priority.ERROR, "DataWriteManager::validDocument() : " + e);
				LogService.log(Priority.ERROR, "DataWriteManager::validDocument() : SQL : " + sql);
				rollback();
				throw new DataException("Erreur lors de la validation du document");
			}
		}
		
		sql = "UPDATE INT_DOCUMENT SET ID_DOC = '" + doc.getId() + "', NEW_DOC = '0' WHERE ID_DOC = '" + "i" + doc.getId().substring(1) + "' AND ID_INT = '" + doc.getIntranet() + "'";
		try {
			query.setSql(sql);
			int result = query.update();
			if(result == 0) {
				rollback();
				throw new DataException("Erreur lors de la validation du document");				
			}
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::validDocument() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::validDocument() : SQL : " + sql);
			rollback();
			throw new DataException("Erreur lors de la validation du document");
		}
	}

	
	/************************************************************
	 *	Types MIME												*
	 ************************************************************/
	
	/**
	 * Cration d'un type MIME
	 * @param mime Le type MIME  crer
	 */
	public void createMIME(Mime mime) throws DataException {
		LogService.log(nivlog, "DataWriteManager::createMIME()");
		
		StringBuffer sql = new StringBuffer();
		sql.append("INSERT INTO INT_MIME(EXT_MIME, TYPE_MIME) VALUES(");
		sql.append("'" + mime.getExtension() + "', ");
		sql.append("'" + mime.getType() + "')");
		
		Query query = getQuery();
		try {
			query.setSql(sql.toString());
			query.insert();
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::createMIME() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::getIntranet() : SQL : " + sql);
			close();
			throw new DataException("Erreur lors de la cration du type MIME");
		}
		close();
	}

	/**
	 * Suppression d'un type MIME
	 * @param extension L'extension associe au type MIME
	 * @throws DataException
	 */
	public void removeMIME(String extension) throws DataException {
		LogService.log(nivlog, "DataWriteManager::removeMIME()");
		
		String sql = "DELETE FROM INT_MIME WHERE EXT_MIME = '" + extension + "'";
		
		Query query = getQuery();
		try {
			query.setSql(sql);
			query.delete();
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::removeMIME() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::getIntranet() : SQL : " + sql);
			close();
			throw new DataException("Erreur lors de la suppression du type MIME");
		}
		close();
	}

	
	/************************************************************
	 *	Valideurs												*
	 ************************************************************/
	
	/**
	 * Ajout d'un valideur
	 * @param id L'identifiant du dossier
	 * @param login L'identifiant de l'utilisateur
	 * @param notification Si l'utilisateur souhaite tre notifi true, false sinon
	 * @param propagation Si ce valideur doit tre propag aux sous-dossiers true, false sinon
	 */
	public void addValidator(String id, String login, boolean notification, boolean propagation) throws DataException {
		LogService.log(nivlog, "DataWriteManager::addValidator()");
		
		StringBuffer sql = new StringBuffer();
		sql.append("INSERT INTO INT_VALIDEUR(ID_DOS, LOGIN, NOT_VAL) VALUES(");
		sql.append("'" + id + "', ");
		sql.append("'" + login + "', ");
		if(notification) {
			sql.append("'1')");
		}
		else {
			sql.append("'0')");
		}
		
		Query query = getQuery();
		if(propagation) {
			begin();
		}
		try {
			query.setSql(sql.toString());
			query.insert();
			
			if(propagation) {
				Iterator i = getFolderChildren(id).iterator();
				while(i.hasNext()) {
					SubFolder tmp = (SubFolder)i.next();
					addValidator(tmp.getId(), login, notification, false);
				}
			}
		}
		catch(DataException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::addValidator() : " + e);
			if(propagation) {
				rollback();
			}
			else {
				close();
			}
			throw new DataException("Erreur lors de l'ajout du valideur");
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::addValidator() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::getIntranet() : SQL : " + sql);
			if(propagation) {
				rollback();
			}
			else {
				close();
			}
			throw new DataException("Erreur lors de l'ajout du valideur");
		}
		if(propagation) {
			commit();
		}
		else {
			close();
		}
	}

	/**
	 * Suppression d'un valideur
	 * @param id L'identifiant du dossier
	 * @param login L'identifiant de l'utilisateur
	 * @param propagation Si le valideur doit tre supprim des sous-dossiers true, false sinon
	 * @throws DataException
	 */
	public void removeValidator(String id, String login, boolean propagation) throws DataException {
		LogService.log(nivlog, "DataWriteManager::removeValidator()");
		
		StringBuffer sql = new StringBuffer();
		sql.append("DELETE FROM INT_VALIDEUR WHERE ");
		sql.append("ID_DOS = '" + id + "' AND ");
		sql.append("LOGIN = '" + login + "'");
		
		Query query = getQuery();
		if(propagation) {
			begin();
		}
		try {
			query.setSql(sql.toString());
			query.delete();
			
			if(propagation) {
				Iterator i = getFolderChildren(id).iterator();
				while(i.hasNext()) {
					SubFolder tmp = (SubFolder)i.next();
					removeValidator(tmp.getId(), login, false);
				}
			}
		}
		catch(DataException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::removeValidator() : " + e);
			if(propagation) {
				rollback();
			}
			else {
				close();
			}
			throw new DataException("Erreur lors de la suppression du valideur");
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::removeValidator() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::getIntranet() : SQL : " + sql);
			if(propagation) {
				rollback();
			}
			else {
				close();
			}
			throw new DataException("Erreur lors de la suppression du valideur");
		}
		if(propagation) {
			commit();
		}
		else {
			close();
		}
	}
	
	/**
	 * Suppression de tous les valideurs d'un dossier
	 * @param id L'identifiant du dossier
	 * @throws DataException
	 */
	public void removeAllValidators(String id) throws DataException {
		LogService.log(nivlog, "DataWriteManager::removeAllValidators()");
		
		StringBuffer sql = new StringBuffer();
		sql.append("DELETE FROM INT_VALIDEUR WHERE ");
		sql.append("ID_DOS = '" + id + "'");
		
		Query query = getQuery();
		try {
			query.setSql(sql.toString());
			query.delete();
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::removeAllValidators() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::getIntranet() : SQL : " + sql);
			close();
			throw new DataException("Erreur lors de la suppression des valideurs");
		}
		close();
	}

	
	/************************************************************
	 *	Publicateurs											*
	 ************************************************************/
	
	/**
	 * Ajout d'un publicateur
	 * @param id L'identifiant du dossier
	 * @param login L'identifiant de l'utilisateur
	 */
	public void addPublicator(String id, String login, boolean propagation) throws DataException {
		LogService.log(nivlog, "DataWriteManager::addPublicator()");
		
		StringBuffer sql = new StringBuffer();
		sql.append("INSERT INTO INT_PUBLICATEUR(ID_DOS, TYPE_PUB, ID_PUB) VALUES(");
		sql.append("'" + id + "', ");
		sql.append("'U', ");
		sql.append("'" + login + "');");
		
		Query query = getQuery();
		if(propagation) {
			begin();
		}
		try {
			query.setSql(sql.toString());
			query.insert();
			
			if(propagation) {
				Iterator i = getFolderChildren(id).iterator();
				while(i.hasNext()) {
					SubFolder tmp = (SubFolder)i.next();
					addPublicator(tmp.getId(), login, false);
				}
			}
		}
		catch(DataException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::addPublicator() : " + e);
			if(propagation) {
				rollback();
			}
			else {
				close();
			}
			throw new DataException("Erreur lors de l'ajout du publicateur");
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::addPublicator() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::getIntranet() : SQL : " + sql);
			if(propagation) {
				rollback();
			}
			else {
				close();
			}
			throw new DataException("Erreur lors de l'ajout du publicateur");
		}
		if(propagation) {
			commit();
		}
		else {
			close();
		}
	}

	/**
	 * Ajout d'un groupe de publicateurs
	 * @param id L'identifiant du dossier
	 * @param group L'identifiant du groupe
	 * @throws DataException
	 */
	public void addPublicators(String id, String group, boolean propagation) throws DataException {
		LogService.log(nivlog, "DataWriteManager::addPublicators()");
		
		StringBuffer sql = new StringBuffer();
		sql.append("INSERT INTO INT_PUBLICATEUR(ID_DOS, TYPE_PUB, ID_PUB) VALUES(");
		sql.append("'" + id + "', ");
		sql.append("'G', ");
		sql.append("'" + group + "')");
		
		Query query = getQuery();
		if(propagation) {
			begin();
		}
		try {
			query.setSql(sql.toString());
			query.insert();
			
			if(propagation) {
				Iterator i = getFolderChildren(id).iterator();
				while(i.hasNext()) {
					SubFolder tmp = (SubFolder)i.next();
					addPublicators(tmp.getId(), group, false);
				}
			}
		}
		catch(DataException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::addPublicators() : " + e);
			if(propagation) {
				rollback();
			}
			else {
				close();
			}
			throw new DataException("Erreur lors de l'ajout du groupe de publicateurs");
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::addPublicators() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::getIntranet() : SQL : " + sql);
			if(propagation) {
				rollback();
			}
			else {
				close();
			}
			throw new DataException("Erreur lors de l'ajout du grope de publicateurs");
		}
		if(propagation) {
			commit();
		}
		else {
			close();
		}
	}

	/**
	 * Suppression d'un publicateur
	 * @param id L'identifiant du dossier
	 * @param login L'identifiant de l'utilisateur
	 * @throws DataException
	 */
	public void removePublicator(String id, String login, boolean propagation) throws DataException {
		LogService.log(nivlog, "DataWriteManager::removePublicator()");
		
		StringBuffer sql = new StringBuffer();
		sql.append("DELETE FROM INT_PUBLICATEUR WHERE ");
		sql.append("ID_DOS = '" + id + "' AND ");
		sql.append("TYPE_PUB = 'U' AND ");
		sql.append("ID_PUB = '" + login + "'");
		
		Query query = getQuery();
		if(propagation) {
			begin();
		}
		try {
			query.setSql(sql.toString());
			query.delete();
			
			if(propagation) {
				Iterator i = getFolderChildren(id).iterator();
				while(i.hasNext()) {
					SubFolder tmp = (SubFolder)i.next();
					removePublicator(tmp.getId(), login, false);
				}
			}
		}
		catch(DataException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::removePublicator() : " + e);
			if(propagation) {
				rollback();
			}
			else {
				close();
			}
			throw new DataException("Erreur lors de la suppression du publicateur");
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::removePublicator() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::getIntranet() : SQL : " + sql);
			if(propagation) {
				rollback();
			}
			else {
				close();
			}
			throw new DataException("Erreur lors de la suppression du publicateur");
		}
		if(propagation) {
			commit();
		}
		else {
			close();
		}
	}

	/**
	 * Suppression d'un groupe de publicateurs
	 * @param id L'identifiant du dossier
	 * @param group L'identifiant du groupe
	 * @throws DataException
	 */
	public void removePublicators(String id, String group, boolean propagation) throws DataException {
		LogService.log(nivlog, "DataWriteManager::removePublicators()");
		
		StringBuffer sql = new StringBuffer();
		sql.append("DELETE FROM INT_PUBLICATEUR WHERE ");
		sql.append("ID_DOS = '" + id + "' AND ");
		sql.append("TYPE_PUB = 'G' AND ");
		sql.append("ID_PUB = '" + group + "'");
		
		Query query = getQuery();
		if(propagation) {
			begin();
		}
		try {
			query.setSql(sql.toString());
			query.delete();
			
			if(propagation) {
				Iterator i = getFolderChildren(id).iterator();
				while(i.hasNext()) {
					SubFolder tmp = (SubFolder)i.next();
					removePublicators(tmp.getId(), group, false);
				}
			}
		}
		catch(DataException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::removePublicators() : " + e);
			if(propagation) {
				rollback();
			}
			else {
				close();
			}
			throw new DataException("Erreur lors de la suppression du groupe de publicateurs");
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::removePublicators() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::getIntranet() : SQL : " + sql);
			if(propagation) {
				rollback();
			}
			else {
				close();
			}
			throw new DataException("Erreur lors de la suppression du groupe de publicateurs");
		}
		if(propagation) {
			commit();
		}
		else {
			close();
		}
	}
	
	/**
	 * Suppression de tous les publicateurs d'un dossier
	 * @param id L'identifiant du dossier
	 * @throws DataException
	 */
	public void removeAllPublicators(String id) throws DataException {
		LogService.log(nivlog, "DataWriteManager::removeAllPublicators()");
		
		StringBuffer sql = new StringBuffer();
		sql.append("DELETE FROM INT_PUBLICATEUR WHERE ");
		sql.append("ID_DOS = '" + id + "'");
		
		Query query = getQuery();
		try {
			query.setSql(sql.toString());
			query.delete();
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::removeAllPublicators() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::getIntranet() : SQL : " + sql);
			close();
			throw new DataException("Erreur lors de la suppression des publicateurs");
		}
		close();
	}

	
	/************************************************************
	 *	Fonctionnement interne									*
	 ************************************************************/
	
	/**
	 * Rcupration de tous les sous-dossiers d'un dossier<br>
	 * (rcursif)
	 * @param id L'identifiant du dossier racine
	 * @return La liste des dossiers fils
	 * @throws DataException
	 */
	private Collection getFolderChildren(String id) throws DataException {
		Collection res = new Vector();
		Collection children = new Vector();
		
		// Rcupration IDataReadManager
		IDataReadManager data = DataFactory.makeRead();
		res = data.getSubFolders(id);
		
		Iterator i = res.iterator();
		while(i.hasNext()) {
			SubFolder tmp = (SubFolder)i.next();
			children.addAll(getFolderChildren(tmp.getId()));
		}
		res.addAll(children);
		return res;
	}
	
	/**
	 * Retourne la cl suivante d'une table
	 * @param table La table dont on souhaite connatre la prochaine cl
	 * @return La cl
	 */
	private int getNextKey(String table) {
		LogService.log(nivlog, "DataWriteManager::getNextKey() : " + table);
		int res = 0;
		
		String sql = "SELECT CLE FROM INT_CLE WHERE NOM_TABLE = '" + table + "'";
		Query query = getQuery();
		begin();
		try {
			query.setSql(sql);
			query.select();
			ResultSet result = query.getRs();
			if(!result.next()) {
				rollback();
				return res;
			}
			res = result.getInt("CLE");
			sql = "UPDATE INT_CLE SET CLE = " + (res+1) + " WHERE NOM_TABLE = '" + table + "'";
			query.setSql(sql);
			query.update();
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::getNextKey() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::getIntranet() : SQL : " + sql);
			rollback();
			return 0;
		}
		commit();
		return res;
	}
	
	/**
	 * Retourne le prochain identifiant de document
	 * @param intranet L'identifiant de l'intranet auquel appartient le document
	 * @return L'identifiant
	 */
	private int getNextDocument(String intranet) {
		LogService.log(nivlog, "DataWriteManager::getNextDocument()");
		int res = 0;
		
		String sql = "SELECT DOC_INT FROM INT_INTRANET WHERE ID_INT = '" + intranet + "'";
		Query query = getQuery();
		begin();
		try {
			query.setSql(sql);
			query.select();
			ResultSet result = query.getRs();
			if(!result.next()) {
				rollback();
				return res;
			}
			res = result.getInt("DOC_INT");
			sql = "UPDATE INT_INTRANET SET DOC_INT = " + (res+1) + " WHERE ID_INT = '" + intranet + "'";
			query.setSql(sql);
			query.update();
		}
		catch(SQLException e) {
			LogService.log(Priority.ERROR, "DataWriteManager::getNextDocument() : " + e);
			LogService.log(Priority.ERROR, "DataWriteManager::getNextDocument() : SQL : " + sql);
			rollback();
			return 0;
		}
		commit();
		return res;
	}
	
	/**
	 * Dbute une transaction
	 */
	private void begin() {
		if(transactions == 0) {
		    query.setAutoCommit(false);
		    query.setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);
		}
		transactions++;
		LogService.log(nivlog, "DataWriteManager::begin() transactions = " + transactions);
	}

	/**
	 * Retourne la connexion  la base de donnes
	 * @return La connexion
	 */
	private Query getQuery() {
		LogService.log(nivlog, "DataWriteManager::getQuery()");
		if (requests == 0) {
			query = Config.getInstance().getConnexionDefault();
		}
		requests++;
		return query;
	}

	/**
	 * Ferme la connexion courante
	 */
	private void close() {
		requests--;
		if(requests == 0) {
			LogService.log(nivlog, "DataWriteManager::close() : CLOSE");
			query.close();
			query = null;
		}
	}

	/**
	 * Vrifie lors de la destruction de l'objet que la connexion est bien ferme
	 */
	protected void finalize() throws Throwable {
		super.finalize();
		if(query != null) {
			LogService.log(Priority.FATAL, "DataWriteManager::finalize() : Des connexions n'ont pas t fermes");
		}
	}

	
	/************************************************************
	 *	Transactions											*
	 ************************************************************/
	
	/**
	 * Valide les modifications
	 */
	public void commit() {
		transactions--;
		if(transactions == 0) {
			LogService.log(nivlog, "DataWriteManager::commit() : COMMIT");
			query.commit();
			close();
		}
		else {
		    close();
		}
		if(transactions < 0) {
		    LogService.log(Priority.FATAL, "DataWriteManager::commit() : transactions < 0");
		    transactions = 0;
		}
		LogService.log(nivlog, "DataWriteManager::commit() : transactions = " + transactions);
	}
	
	/**
	 * Annule les modifications
	 */
	public void rollback() {
		transactions--;
		if(transactions == 0) {
			LogService.log(nivlog, "DataWriteManager::rollback() : ROLLBACK");
			query.rollback();
			close();
		}
		else {
		    close();
		}
		if(transactions < 0) {
		    LogService.log(Priority.FATAL, "DataWriteManager::commit() : transactions < 0");
			transactions = 0;
		}
		LogService.log(nivlog, "DataWriteManager::rollback() : transactions = " + transactions);
	}
}