package org.esupportail.portal.channels.CIntranet.data;

import java.util.Collection;

import org.esupportail.portal.channels.CIntranet.beans.Document;
import org.esupportail.portal.channels.CIntranet.beans.Folder;
import org.esupportail.portal.channels.CIntranet.beans.Intranet;
import org.esupportail.portal.channels.CIntranet.beans.Mime;
import org.esupportail.portal.channels.CIntranet.beans.Population;


/**
 * IDataAccess<br>
 * <br>
 * Cette interface dcrit les mthodes d'accs aux donnes<br>
 * <br>
 * (c)Copyright <a href="http://www.esup-portail.org">ESup-Portail 2004</a><br>
 * @author <a href="mailto:mathieu.larchet@univ-nancy2.fr">Mathieu Larchet</a>
 * @version 2.0
 * 
 */
public interface IDataAccess {
    
    /**
	 * Rcupration d'un document
	 * @param intranet L'identifiant de l'intranet auquel appartient le document
	 * @param id L'identifiant du dossier auquel appartient le document
	 * @return Le document
	 * @throws DataException
	 */
	public Document getDocument(String intranet, String id) throws DataException;
	
	/**
	 * Cration d'un document
	 * @param doc Le document  crer
	 * @param valid Si le document est valide true, false sinon
	 */
	public void createDocument(Document doc, boolean valid) throws DataException;
	
	/**
	 * Suppression d'un document
	 * @param doc Le document  supprimer
	 * @throws DataException
	 */
	public void removeDocument(Document doc) throws DataException;
	
	/**
	 * Mise  jour d'un document
	 * @param doc Le document  mettre  jour
	 * @throws DataException
	 */
	public void updateDocument(Document doc) throws DataException;
	
	/**
	 * Validation d'un document
	 * @param doc Le document  valider
	 * @throws DataException
	 */
	public void validDocument(Document doc) throws DataException;
	
	/**
	 * Rcupration d'un dossier
	 * @param id L'identifiant du dossier
	 * @param sub Si true alors on rcupre galement les sous-dossiers<br>
	 * et les documents
	 * @return Le dossier
	 * @throws DataException
	 */
	public Folder getFolder(String id, boolean sub) throws DataException;
	
	/**
	 * Cration d'un dossier
	 * @param folder Le dossier  crer
	 * @return L'identifiant du dossier cr
	 */
	public void createFolder(Folder folder) throws DataException;
	
	/**
	 * Suppression d'un dossier
	 * @param id L'identifiant du dossier
	 * @throws DataException
	 */
	public void removeFolder(String id) throws DataException;
	
	/**
	 * Modifie la validation d'un dossier
	 * @param folder Le dossier
	 * @param validation True pour activer la validation, false pour la dsactiver
	 * @throws DataException
	 */
	public void setValidation(Folder folder, boolean validation) throws DataException;
	
	/**
	 * Modifie le nom d'un dossier
	 * @param folder Le dossier
	 * @throws DataException
	 */
	public void setName(Folder folder) throws DataException;
	
	/**
	 * Rcupration d'un intranet
	 * @param id L'identifiant d'intranet
	 * @return L'intranet correspondant
	 * @throws DataException
	 */
	public Intranet getIntranet(String id) throws DataException;

	/**
	 * Rcupration de la liste de tous les intranets
	 * @return La liste de tous les intranets
	 * @throws DataException
	 */
	public Collection getIntranets() throws DataException;
	
	/**
	 * Cration d'un intranet
	 * @param intranet L'intranet  crer
	 */
	public void createIntranet(Intranet intranet) throws DataException;
	
	/**
	 * Suppression d'un intranet
	 * @param intranet L'intranet  supprimer
	 * @throws DataException
	 */
	public void removeIntranet(Intranet intranet) throws DataException;
	
	/**
	 * Rcupration des types MIME valides
	 * @return La liste des types MIME
	 * @throws DataException
	 */
	public Collection getMIMEs() throws DataException;
	
	/**
	 * Cration d'un type MIME
	 * @param mime Le type MIME  crer
	 */
	public void createMIME(Mime mime) throws DataException;
	
	/**
	 * Suppression d'un type MIME
	 * @param extension L'extension associe au type MIME
	 * @throws DataException
	 */
	public void removeMIME(String extension) throws DataException;
	
	/**
	 * Vrifie la validit d'un type MIME
	 * @param extension L'extension  tester
	 * @return boolean True si le type est valide, false sinon
	 * @throws DataException
	 */
	public boolean isValidMIME(String extension) throws DataException;
	
	/**
	 * Rcupration d'une population
	 * @param id L'identifiant d'intranet
	 * @return La population de l'intranet
	 * @throws DataException
	 */
	public Population getPopulation(String id) throws DataException;
	
	/**
	 * Ajout d'une population
	 * @param population La population  ajouter
	 */
	public void addPopulation(Population population) throws DataException;
	
	/**
	 * Rcupration des publicateurs d'un dossier
	 * @param id L'identifiant du dossier
	 * @return La liste des publicateurs
	 * @throws DataException
	 */
	public Collection getUsersPublicators(String id) throws DataException;
	
	/**
	 * Rcupration des groupes de publicateurs d'un dossier
	 * @param id L'identifiant du dossier
	 * @return La liste des groupes de publicateurs
	 * @throws DataException
	 */
	public Collection getGroupsPublicators(String id) throws DataException;
	
	/**
	 * Ajout d'un publicateur
	 * @param id L'identifiant du dossier
	 * @param login L'identifiant de l'utilisateur
	 */
	public void addPublicator(String id, String login) throws DataException;
	
	/**
	 * Ajout d'un groupe de publicateurs
	 * @param id L'identifiant du dossier
	 * @param group L'identifiant du groupe
	 * @throws DataException
	 */
	public void addPublicators(String id, String group) throws DataException;
	
	/**
	 * Suppression d'un publicateur
	 * @param id L'identifiant du dossier
	 * @param login L'identifiant de l'utilisateur
	 * @throws DataException
	 */
	public void removePublicator(String id, String login) throws DataException;
	
	/**
	 * Suppression d'un groupe de publicateurs
	 * @param id L'identifiant du dossier
	 * @param group L'identifiant du groupe
	 * @throws DataException
	 */
	public void removePublicators(String id, String group) throws DataException;
	
	/**
	 * Recherche de documents par nom / description
	 * @param intranet L'intranet concern
	 * @param name Tout ou partie du nom du document
	 * @param description Tout ou partie de la description du document
	 * @param valid La validit des documents recherchs
	 * @return La liste des documents correspondants au(x) critre(s) de recherche
	 */
	public Collection findByName(String intranet, String name, String description, boolean valid) throws DataException;
	
	/**
	 * Recherche de tous les documents en attente de validation
	 * @param intranet L'intranet concern
	 * @return La liste des documents en attente de validation
	 * @throws DataException
	 */
	public Collection findInvalid(String intranet, String name, String description) throws DataException;
	
	/**
	 * Recherche des documents dposs par un utilisateur en attente de validation
	 * @param intranet L'intranet concern
	 * @param login L'identifiant de l'utilisateur
	 * @return La liste des documents de l'utilisateur en attente de validation
	 * @throws DataException
	 */
	public Collection findMyInvalid(String intranet, String name, String description, String login) throws DataException;
	
	/**
	 * Rcupration des super-utilisateurs
	 * @return La liste des super-utilisateurs
	 * @throws DataException
	 */
	public Collection getSuperUsers() throws DataException;
	
	/**
	 * Ajout d'un super-utilisateur
	 * @param login L'identifiant de l'utilisateur
	 */
	public void addSuperUser(String login) throws DataException;
	
	/**
	 * Suppression d'un super-utilisateur
	 * @param login L'identifiant de l'utilisateur
	 * @throws DataException
	 */
	public void removeSuperUser(String login) throws DataException;
	
	/**
	 * Rcupration des valideurs d'un dossier
	 * @param id L'identifiant du dossier
	 * @return La liste des valideurs
	 * @throws DataException
	 */
	public Collection getValidators(String id) throws DataException;
	
	/**
	 * Ajout d'un valideur
	 * @param id L'identifiant du dossier
	 * @param login L'identifiant de l'utilisateur
	 */
	public void addValidator(String id, String login) throws DataException;
	
	/**
	 * Suppression d'un valideur
	 * @param id L'identifiant du dossier
	 * @param login L'identifiant de l'utilisateur
	 * @throws DataException
	 */
	public void removeValidator(String id, String login) throws DataException;
	
	/**
     * Echange le rang de deux documents
     * @param id L'identifiant du dossier
     * @param doc L'identifiant du document
     * @param oldorder L'ancien rang du document
     * @param neworder Le nouveau rang du document
     * @throws DataException
     */
    public void swap(String id, String doc, String oldorder, String neworder) throws DataException;
}
