package org.esupportail.portal.channels.CIntranet.security;

import java.util.Collection;

import org.apache.log4j.Priority;
import org.esupportail.portal.channels.CIntranet.beans.Population;
import org.esupportail.portal.channels.CIntranet.data.DataException;
import org.esupportail.portal.channels.CIntranet.data.DataFactory;
import org.esupportail.portal.channels.CIntranet.data.IDataControler;
import org.esupportail.portal.channels.CIntranet.groups.GroupException;
import org.esupportail.portal.channels.CIntranet.groups.GroupFactory;
import org.esupportail.portal.channels.CIntranet.groups.IGroupManager;
import org.jasig.portal.services.LogService;

/**
 * SecurityManager<br>
 * <br>
 * Implmentation d'un ISecurityManager utilisant les fonctionnalits du IDatabaseManager<br>
 * et du IGroupManager pour raliser ses fonctionnalits<br>
 * <br>
 * (c)Copyright <a href="http://www.esup-portail.org">ESup-Portail 2004</a><br>
 * @author <a href="mailto:mathieu.larchet@univ-nancy2.fr">Mathieu Larchet</a>
 * @version 2.0
 * 
 */

public class SecurityManager implements ISecurityManager {
	
	private static Priority nivlog = Priority.DEBUG;
	private IDataControler data = null;
	private IGroupManager group = null;
	
	/**
	* Constructeur
	*/
	public SecurityManager() {
	}

	/**
	 * Si l'utilisateur est un super-utilisateur
	 * @param login Le login de l'utilisateur
	 * @return true si c'est un super-utilisateur, false sinon
	 * @throws SecurityException
	 */
	public boolean isSuperUser(String login) throws SecurityException {
		LogService.log(nivlog, "SecurityManager::isSuperUser()");
		
		// Source de donnes
		data = DataFactory.make();
		
		try {
			Collection su = data.getSuperUsers();
			return su.contains(login);
		}
		catch(DataException e) {
			LogService.log(Priority.ERROR, "SecurityManager::isSuperUser() : " + e);
			throw new SecurityException(e.getMessage());
		}
	}

	/**
	 * Si l'utilisateur fait partie de la population d'un intranet
	 * @param intranet L'identifiant de l'intranet
	 * @param login L'identifiant de l'utilisateur
	 * @return true si l'utilisateur fait partie de la population, false sinon
	 * @throws SecurityException
	 */
	public boolean isIntranetAccessible(String intranet, String login) throws SecurityException {
		LogService.log(nivlog, "SecurityManager::isIntranetAccessible()");
		
		// Source de donnes
		data = DataFactory.make();
		
		Population members = null;
		try {
			members = data.getPopulation(intranet);
		}
		catch(DataException e) {
			LogService.log(Priority.ERROR, "SecurityManager::isIntranetAccessible() : " + e);
			throw new SecurityException(e.getMessage());
		}
		
		Collection users = members.getUsers();
		if(users.contains(login)) {
			return true;
		}
		
		// Gestion des groupes
		group = GroupFactory.make();
		
		Collection groups = members.getGroups();
		try {
			return group.isMemberOf(login, groups);
		}
		catch(GroupException e) {
			LogService.log(Priority.ERROR, "SecurityManager::isIntranetAccessible() : " + e);
			throw new SecurityException(e.getMessage());
		}
	}

	/**
	 * Retourne une constante indiquant le statut d'un utilisateur pour un dossier donn
	 * @param folder L'identifiant du dossier
	 * @param login L'identifiant de l'utilisateur
	 * @return USER si c'est un utilisateur standard<br>
	 * PUBLICATOR si c'est un publicateur
	 * VALIDATOR si c'est un valideur
	 * @throws SecurityException
	 */
	public int getRoleOfUser(String folder, String login) throws SecurityException {
		LogService.log(nivlog, "SecurityManager::getRoleOfUser()");
		if(isValidator(login, folder)) {
			return ISecurityManager.VALIDATOR;
		}
		
		if(isPublicator(login, folder)) {
			return ISecurityManager.PUBLICATOR;
		}
		
		return ISecurityManager.USER;
	}
	
	/**
	 * Indique si l'utilisateur est un valideur pour un dossier donn
	 * @param login L'identifiant de l'utilisateur
	 * @param id L'identifiant du dossier
	 * @return true si l'utilisateur est un valideur, false sinon
	 * @throws SecurityException
	 */
	private boolean isValidator(String login, String id) throws SecurityException {
		LogService.log(nivlog, "SecurityManager::isValidator()");
		
		// Source de donnes
		data = DataFactory.make();
		
		try {
			Collection validators = data.getValidators(id);
			return validators.contains(login);
		}
		catch(DataException e) {
			LogService.log(Priority.ERROR, "SecurityManager::isValidator() : " + e);
			throw new SecurityException(e.getMessage());
		}
	}

	/**
	 * Indique si l'utilisateur est un publicateur pour un dossier donn
	 * @param login L'identifiant de l'utilisateur
	 * @param id L'identifiant du dossier
	 * @return true si l'utilisateur est un publicateur, false sinon
	 * @throws SecurityException
	 */
	private boolean isPublicator(String login, String id) throws SecurityException {
		LogService.log(nivlog, "SecurityManager::isPublicator()");
		Collection groups = null;
		
		// Source de donnes
		data = DataFactory.make();
		
		try {
			Collection publicators = data.getUsersPublicators(id);
			if(publicators.contains(login)) {
				return true;
			}
			groups = data.getGroupsPublicators(id);
		}
		catch(DataException e) {
			LogService.log(Priority.ERROR, "SecurityManager::isPublicator() : " + e);
			throw new SecurityException(e.getMessage());
		}
		
		// Cas de toute la population
		if(groups.contains("_EVERYBODY")) {
			return true;
		}
		
		// Gestion des groupes
		group = GroupFactory.make();
		
		try {
			return group.isMemberOf(login, groups);
		}
		catch(GroupException e) {
			LogService.log(Priority.ERROR, "SecurityManager::isPublicator() : " + e);
			throw new SecurityException(e.getMessage());
		}
	}
}
