package org.esupportail.portal.channels.CIntranet.storage;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Hashtable;

import org.apache.log4j.Priority;
import org.esupportail.portal.channels.CIntranet.beans.Document;
import org.esupportail.portal.channels.CIntranet.beans.Intranet;
import org.jasig.portal.services.LogService;

/**
 * BalancedFileSystemStorage<br>
 * <br>
 * Implmentation d'un IStorageControler avec un stockage des documents sur<br>
 * un systme de fichier physique local (ventuellement partag)<br>
 * Chaque intranet se voit attribuer un dossier dans lequel des dossiers sont crs<br>
 * afin qu'aucun dossier ne contienne plus d'un nombre dtermin de documents<br>
 * <br>
 * (c)Copyright <a href="http://www.esup-portail.org">ESup-Portail 2004</a><br>
 * @author <a href="mailto:mathieu.larchet@univ-nancy2.fr">Mathieu Larchet</a>
 * @version 1.2
 * 
 */

public class BalancedFileSystemStorage implements IStorageControler {

	private Priority nivlog = Priority.DEBUG;
	private String root = null;
	private int size = 100;
	
	/**
	 * Constructeur
	 */
	public BalancedFileSystemStorage() {
	}
	
	/**
	 * Sauvegarde d'un document
	 * @param intranet L'intranet auquel appartient le document
	 * @param doc Le document
	 * @param in Le flux de lecture vers le contenu du document
	 * @throws StorageException
	 */
	public void saveDocument(String intranet, Document doc, InputStream in) throws StorageException {
		LogService.log(nivlog, "BasicFileSystemStorage::saveDocument()");
		// Vrification paramtres
		if(root == null) {
			LogService.log(Priority.ERROR, "BasicFileSystemStorage::saveDocument() : root = null");
			throw new StorageException("Le rpertoire de base n'est pas correctement configur\u00E9");
		}
		
		// Vrification rpertoire
		File folder = new File(root + "/" + intranet + "/" + getFolder(doc.getId()));
		if(!folder.exists()) {
			folder.mkdir();
		}
		
		// Cration du fichier
		File file = new File(root + "/" + intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId());
		
		// Vrification existence du fichier
		if(file.exists()) {
			LogService.log(Priority.ERROR, "BasicFileSystemStorage::saveDocument() : Le document " + doc.getId() + " existe d\u00E9j\u00E0");
			throw new StorageException("Erreur lors de la sauvegarde du document");
		}
		try {
			// Cration du fichier vide
			if(!file.createNewFile())
			{
				LogService.log(Priority.ERROR, "BasicFileSystemStorage::saveDocument() : Erreur lors de la cr\u00E9ation du fichier");
				throw new StorageException("Erreur lors de la sauvegarde du document");
			}
		}
		catch(IOException e) {
			LogService.log(Priority.ERROR, "BasicFileSystemStorage::saveDocument() : Erreur lors de la cr\u00E9ation du fichier");
			throw new StorageException("Erreur lors de la sauvegarde du document");
		}
		
		// Ecriture des donnes
		try {
			OutputStream out = new FileOutputStream(file);
			
			int size = 0;
            byte[] contentBytes = new byte[8192];
            while ((size = in.read(contentBytes)) != -1) {
                out.write(contentBytes,0, size);
            }
			in.close();
			out.flush();
			out.close();
		}
		catch(FileNotFoundException e) {
			LogService.log(Priority.ERROR, "BasicFileSystemStorage::saveDocument() : Erreur lors de l'\u00E9criture du fichier");
			throw new StorageException("Erreur lors de la sauvegarde du document");
		}
		catch(IOException e) {
			LogService.log(Priority.ERROR, "BasicFileSystemStorage::saveDocument() : Erreur lors de l'\u00E9criture du fichier");
			throw new StorageException("Erreur lors de la sauvegarde du document");
		}
	}

	/**
	 * Lecture d'un document
	 * @param intranet L'intranet auquel appartient le document
	 * @param doc Le document
	 * @return Le flux de lecture vers le document stock
	 * @throws StorageException
	 */
	public InputStream loadDocument(String intranet, Document doc) throws StorageException {
		LogService.log(nivlog, "BasicFileSystemStorage::loadDocument()");
		// Vrification paramtres
		if(root == null) {
			LogService.log(Priority.ERROR, "BasicFileSystemStorage::loadDocument() : root = null");
			throw new StorageException("Le rpertoire de base n'est pas correctement configur\u00E9");
		}
		
		// Rcupration du fichier
		File loadDoc = loadDoc = new File(root + "/" + intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId());
				
		try {
			return new FileInputStream(loadDoc);
		}
		catch(FileNotFoundException e) {
			LogService.log(Priority.ERROR, "BasicFileSystemStorage::saveDocument() : Erreur lors de la lecture du document " + doc.getId());
			throw new StorageException("Erreur lors de la lecture du document");
		}
	}

	/**
	 * Mise  jour d'un document
	 * @param intranet L'intranet auquel appartient le document
	 * @param doc Le document
	 * @param in Le flux de lecture vers le nouveau contenu, null si pas de mise  jour du contenu
	 * @throws StorageException
	 */
	public synchronized void updateDocument(String intranet, Document doc, InputStream in) throws StorageException {
		LogService.log(nivlog, "BasicFileSystemStorage::updateDocument()");
		// Vrification paramtres
		if(root == null) {
			LogService.log(Priority.ERROR, "BasicFileSystemStorage::updateDocument() : root = null");
			throw new StorageException("Le rpertoire de base n'est pas correctement configur\u00E9");
		}
		
		// Document non valide
		if(!doc.isValid()) {
			if(in == null) {
				// Recopie du fichier valide
				File valid = new File(root + "/" + intranet + "/" + getFolder(doc.getId()) + "/v" + doc.getId().substring(1));
				File newfile = new File(root + "/" + intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId());
				try {
					// Cration du fichier vide
					if(!newfile.createNewFile())
					{
						LogService.log(Priority.ERROR, "BasicFileSystemStorage::updateDocument() : Erreur lors de la cr\u00E9ation du fichier");
						throw new StorageException("Erreur lors de la sauvegarde du document");
					}
					InputStream input = new FileInputStream(valid);
					OutputStream out = new FileOutputStream(newfile);
					int size = 0;
		            byte[] contentBytes = new byte[8192];
		            while((size = input.read(contentBytes)) != -1) {
		                out.write(contentBytes,0, size);
		            }
					input.close();
					out.flush();
					out.close();
					return;
				}
				catch(IOException e) {
					LogService.log(Priority.ERROR, "BasicFileSystemStorage::updateDocument() : Erreur lors de la cr\u00E9ation du fichier");
					throw new StorageException("Erreur lors de la sauvegarde du document");
				}
			}
			else {
				// Ecriture du nouveau fichier
				File newfile = new File(root + "/" + intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId());
				try {
					// Cration du fichier vide
					if(!newfile.createNewFile())
					{
						LogService.log(Priority.ERROR, "BasicFileSystemStorage::updateDocument() : Erreur lors de la cr\u00E9ation du fichier");
						throw new StorageException("Erreur lors de la sauvegarde du document");
					}
					OutputStream out = new FileOutputStream(newfile);
					int size = 0;
		            byte[] contentBytes = new byte[8192];
		            while((size = in.read(contentBytes)) != -1) {
		                out.write(contentBytes,0, size);
		            }
					in.close();
					out.flush();
					out.close();
					return;
				}
				catch(IOException e) {
					LogService.log(Priority.ERROR, "BasicFileSystemStorage::updateDocument() : Erreur lors de la cr\u00E9ation du fichier");
					throw new StorageException("Erreur lors de la sauvegarde du document");
				}
			}
		}
		
		// Document valide
		File valid = new File(root + "/" + intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId());
		valid.delete();
		try {
			// Cration du fichier vide
			if(!valid.createNewFile())
			{
				LogService.log(Priority.ERROR, "BasicFileSystemStorage::updateDocument() : Erreur lors de la cr\u00E9ation du fichier");
				throw new StorageException("Erreur lors de la sauvegarde du document");
			}
			OutputStream out = new FileOutputStream(valid);
			int size = 0;
            byte[] contentBytes = new byte[8192];
            while((size = in.read(contentBytes)) != -1) {
                out.write(contentBytes,0, size);
            }
			in.close();
			out.flush();
			out.close();
		}
		catch(IOException e) {
			LogService.log(Priority.ERROR, "BasicFileSystemStorage::updateDocument() : Erreur lors de la cr\u00E9ation du fichier");
			throw new StorageException("Erreur lors de la sauvegarde du document");
		}
	}
	
	/**
	 * Suppression d'un document
	 * @param intranet L'intranet auquel appartient le document
	 * @param doc Le document
	 * @throws StorageException
	 */
	public synchronized void destroyDocument(String intranet, Document doc) throws StorageException {
		LogService.log(nivlog, "BasicFileSystemStorage::destroyDocument()");
		// Vrification paramtres
		if(root == null) {
			LogService.log(Priority.ERROR, "BasicFileSystemStorage::destroyDocument() : root = null");
			throw new StorageException("Le rpertoire de base n'est pas correctement configur\u00E9");
		}
		
		// Rcupration du fichier
		File file = new File(root + "/" + intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId());
				
		// Suppression du fichier
		if(!file.delete()) {
			LogService.log(Priority.ERROR, "BasicFileSystemStorage::destroyDocument() : Erreur lors de la suppression du fichier " + doc.getId());
			LogService.log(Priority.FATAL, "BasicFileSystemStorage::destroyDocument() : Un document supprim\u00E9 de la base est encore stock\u00E9");
			throw new StorageException("Erreur lors de la suppression du document");
		}
	}

	/**
	 * Validation d'un document
	 * @param intranet L'intranet auquel appartient le document
	 * @param doc Le document  valider
	 * @throws StorageException
	 */
	public synchronized void validDocument(String intranet, Document doc) throws StorageException {
		LogService.log(nivlog, "BasicFileSystemStorage::validDocument()");
		// Vrification paramtres
		if(root == null) {
			LogService.log(Priority.ERROR, "BasicFileSystemStorage::validDocument() : root = null");
			throw new StorageException("Le rpertoire de base n'est pas correctement configur\u00E9");
		}
		
		// Cration des fichiers
		File validfile = new File(root + "/" + intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId());
		File oldfile = new File(root + "/" + intranet + "/" + getFolder(doc.getId()) + "/i" + doc.getId().substring(1));
		
		// Eventuellement suppression de l'ancien fichier et cration du nouveau
		validfile.delete();
		if(!oldfile.renameTo(validfile)) {
			LogService.log(Priority.ERROR, "BasicFileSystemStorage::validDocument() : Erreur lors de la validation du fichier " + doc.getId());
			LogService.log(Priority.FATAL, "BasicFileSystemStorage::validDocument() : La validation a chou\u00E9 le fichier non valid\u00E9 est encore pr\u00E9sent");
		}
	}
	
	/**
	 * Cration d'un intranet
	 * @param intranet L'intranet  crer
	 * @throws StorageException
	 */
	public void createIntranet(Intranet intranet) throws StorageException {
		LogService.log(nivlog, "BasicFileSystemStorage::createIntranet()");
		// Vrification paramtres
		if(root == null) {
			LogService.log(Priority.ERROR, "BasicFileSystemStorage::createIntranet() : root = null");
			throw new StorageException("Le rpertoire de base n'est pas correctement configur\u00E9");
		}
		
		// Cration du dossier racine et du dossier temporaire
		File newFold = new File(root + "/" + intranet.getId());
		if(newFold.exists()) {
			LogService.log(Priority.ERROR, "BasicFileSystemStorage::createIntranet() : Le dossier " + intranet.getId() + "/temp existe d\u00E9j\u00E0");
			throw new StorageException("Erreur lors de la cration de l'intranet");
		}
		if(!newFold.mkdir()) {
			LogService.log(Priority.ERROR, "BasicFileSystemStorage::createIntranet() : Erreur lors de la cr\u00E9ation du dossier " + intranet.getId());
			throw new StorageException("Erreur lors de la cr\u00E9ation de l'intranet");
		}
	}

	/**
	 * Suppression d'un intranet et de tous ses documents
	 * @param intranet L'intranet  supprimer
	 * @throws StorageException
	 */
	public synchronized void destroyIntranet(String intranet) throws StorageException {
		LogService.log(nivlog, "BasicFileSystemStorage::destroyIntranet()");
		if(root == null) {
			LogService.log(Priority.ERROR, "BasicFileSystemStorage::destroyIntranet() : root = null");
			throw new StorageException("Le rpertoire de base n'est pas correctement configur\u00E9");
		}
		LogService.log(Priority.FATAL, "BalancedFileSystemStorage::destroyIntranet() : La suppression de l'intranet " + intranet + " doit \u00EAtre r\u00E9alise manuellement");
	}
	
	
	/**
	 * Calcule le rpertoire o lire / crire un document<br>
	 * en fonction de son identifiant
	 * @param id L'identifiant du document
	 * @return Le chemin o lire / crire le document
	 */
	private String getFolder(String id) {
		int number = new Integer(id.substring(1)).intValue();
		return new Integer(number / size).toString();
	}
	
	/**
	 * Liste des paramtres spcifiques  l'implmentation du IStorageControler<br>
	 * Ici deux paramtres obligatoires :<br>
	 * - size qui contient le nombre maximal de documents par dossier<br>
	 * - path qui indique le chemin o crer les intranets
	 * @param parameters
	 */
	public void setParameters(Hashtable parameters) {
		LogService.log(nivlog, "BasicFileSystemStorage::setParameters()");
		root = (String)parameters.get("path");
		size = new Integer((String)parameters.get("size")).intValue();
	}
}
