package org.esupportail.portal.channels.CIntranet.storage;

import java.io.InputStream;
import java.util.Hashtable;

import org.apache.log4j.Priority;
import org.esupportail.portal.channels.CIntranet.beans.Document;
import org.esupportail.portal.channels.CIntranet.beans.Intranet;
import org.esupportail.portal.channels.CIntranet.utils.DAVConnection;
import org.esupportail.portal.channels.CIntranet.utils.DAVException;
import org.jasig.portal.services.LogService;

/**
 * DAVStorage<br>
 * <br>
 * Implmentation d'un IStorageControler avec un stockage des documents sur<br>
 * un serveur WebDAV<br>
 * Chaque intranet se voit attribuer un dossier dans lequel des dossiers sont crs<br>
 * afin qu'aucun dossier ne contienne plus d'un nombre dtermin de documents<br>
 * <br>
 * (c)Copyright <a href="http://www.esup-portail.org">ESup-Portail 2004</a><br>
 * @author <a href="mailto:mathieu.larchet@univ-nancy2.fr">Mathieu Larchet</a>
 * @version 2.0
 * 
 */
public class DAVStorage implements IStorageControler {

    private Priority nivlog = Priority.DEBUG;
    private String url = null;
    private String uri = null;
    private String login = null;
    private String pass = null;
    private int size = 100;
    
    /**
	 * Constructeur
	 */
	public DAVStorage() {
	}
    
	/**
	 * Sauvegarde d'un document
	 * @param intranet L'intranet auquel appartient le document
	 * @param doc Le document
	 * @param in Le flux de lecture vers le contenu du document
	 * @throws StorageException
	 */
    public void saveDocument(String intranet, Document doc, InputStream in) throws StorageException {
        LogService.log(nivlog, "DAVStorage::saveDocument()");
		// Vrification paramtres
		if(url == null) {
			LogService.log(Priority.ERROR, "DAVStorage::saveDocument() : url = null");
			throw new StorageException("L'URL n'est pas correctement configur\u00E9e");
		}
		
		// Cration ressource WebDAV
		DAVConnection con = new DAVConnection();
		
		try {
		    con.connect(url, uri, login, pass);
		    
		    // Vrification rpertoire
		    String path = intranet + "/" + getFolder(doc.getId()) + "/";
		    
		    // Cration rpertoire
		    if(!con.dirExists(path)) {
		        con.createDir(path);
		    }
		    
		    // Cration du fichier
			String file = path + doc.getId();
		    con.createFile(file, in);
		    con.close();
		}
		catch(DAVException e) {
		    LogService.log(Priority.ERROR, "DAVStorage::saveDocument() : " + e.getMessage());
		    con.close();
			throw new StorageException("Erreur lors de la sauvegarde du document");
		}
    }

    /**
	 * Lecture d'un document
	 * @param intranet L'intranet auquel appartient le document
	 * @param doc Le document
	 * @return Le flux de lecture vers le document stock
	 * @throws StorageException
	 */
    public InputStream loadDocument(String intranet, Document doc) throws StorageException {
        LogService.log(nivlog, "BalancedFileSystemStorage::loadDocument()");
		// Vrification paramtres
		if(url == null) {
		    LogService.log(Priority.ERROR, "DAVStorage::loadDocument() : url = null");
			throw new StorageException("L'URL n'est pas correctement configur\u00E9e");
		}
		
		// Cration ressource WebDAV
		DAVConnection con = new DAVConnection();
		InputStream result = null;
		
		try {
		    con.connect(url, uri, login, pass);
		    
		    // Rcupration du fichier
			String file = intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId();
			result = con.getFile(file);
			con.close();
		}
		catch(DAVException e) {
		    LogService.log(Priority.ERROR, "DAVStorage::loadDocument() : " + e.getMessage());
		    con.close();
			throw new StorageException("Erreur lors de la lecture du document");
		}
		con.close();
		return result;
    }

    /**
	 * Mise  jour d'un document
	 * @param intranet L'intranet auquel appartient le document
	 * @param doc Le document
	 * @param in Le flux de lecture vers le nouveau contenu, null si pas de mise  jour du contenu
	 * @throws StorageException
	 */
    public synchronized void updateDocument(String intranet, Document doc, InputStream in) throws StorageException {
        LogService.log(nivlog, "DAVStorage::updateDocument()");
		// Vrification paramtres
		if(url == null) {
		    LogService.log(Priority.ERROR, "DAVStorage::updateDocument() : url = null");
			throw new StorageException("L'URL n'est pas correctement configur\u00E9e");
		}
		
		// Cration ressource WebDAV
		DAVConnection con = new DAVConnection();
		
		try {
		    con.connect(url, uri, login, pass);
		    if(!doc.isValid()) {
		        if(in == null) {
		            String oldfile = intranet + "/" + getFolder(doc.getId()) + "/v" + doc.getId().substring(1);
			        String newfile = intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId();
			        con.renameFile(oldfile, newfile);
		        }
		        else {
		            String file = intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId();
			        con.createFile(file, in);
		        }
		    }
		    else {
		        String file = intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId();
		        con.deleteFile(file);
		        con.createFile(file, in);
		    }
		    con.close();
		}
		catch(DAVException e) {
		    LogService.log(Priority.ERROR, "DAVStorage::updateDocument() : " + e.getMessage());
		    con.close();
			throw new StorageException("Erreur lors de la mise \u00E0 jour du document");
		}
    }

    /**
	 * Suppression d'un document
	 * @param intranet L'intranet auquel appartient le document
	 * @param doc Le document
	 * @throws StorageException
	 */
    public synchronized void destroyDocument(String intranet, Document doc) throws StorageException {
        LogService.log(nivlog, "DAVStorage::destroyDocument()");
		// Vrification paramtres
		if(url == null) {
		    LogService.log(Priority.ERROR, "DAVStorage::destroyDocument() : url = null");
			throw new StorageException("L'URL n'est pas correctement configur\u00E9e");
		}
		
		// Cration ressource WebDAV
		DAVConnection con = new DAVConnection();
		
		try {
		    con.connect(url, uri, login, pass);
		    String file = intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId();
		    con.deleteFile(file);
		    con.close();
		}
		catch(DAVException e) {
		    LogService.log(Priority.ERROR, "DAVStorage::destroyDocument() : " + e.getMessage());
		    con.close();
			throw new StorageException("Erreur lors de la suppression du document");
		}
    }

    /**
	 * Validation d'un document
	 * @param intranet L'intranet auquel appartient le document
	 * @param doc Le document  valider
	 * @throws StorageException
	 */
    public synchronized void validDocument(String intranet, Document doc) throws StorageException {
        LogService.log(nivlog, "DAVStorage::validDocument()");
		// Vrification paramtres
		if(url == null) {
		    LogService.log(Priority.ERROR, "DAVStorage::validDocument() : url = null");
			throw new StorageException("L'URL n'est pas correctement configur\u00E9e");
		}
		
		// Cration ressource WebDAV
		DAVConnection con = new DAVConnection();
		
		try {
		    con.connect(url, uri, login, pass);
		    String validfile = intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId();
		    String oldfile = intranet + "/" + getFolder(doc.getId()) + "/i" + doc.getId().substring(1);
		    if(con.fileExists(validfile)) {
		        con.deleteFile(validfile);
		    }
		    con.renameFile(oldfile, validfile);
		    con.close();
		}
		catch(DAVException e) {
		    LogService.log(Priority.ERROR, "DAVStorage::validDocument() : " + e.getMessage());
		    con.close();
			throw new StorageException("Erreur lors de la validation du document");
		}
    }

    /**
	 * Invalidation d'un document
	 * @param intranet L'intranet auquel appartient le document
	 * @param doc Le document  invalider
	 * @throws StorageException
	 */
    public synchronized void invalidDocument(String intranet, Document doc) throws StorageException {
        LogService.log(nivlog, "DAVStorage::invalidDocument()");
		// Vrification paramtres
		if(url == null) {
		    LogService.log(Priority.ERROR, "DAVStorage::invalidDocument() : url = null");
			throw new StorageException("L'URL n'est pas correctement configur\u00E9e");
		}
		
		// Cration ressource WebDAV
		DAVConnection con = new DAVConnection();
		
		try {
		    con.connect(url, uri, login, pass);
		    String invalidfile = intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId();
		    String oldfile = intranet + "/" + getFolder(doc.getId()) + "/v" + doc.getId().substring(1);
		    con.renameFile(oldfile, invalidfile);
		    con.close();
		}
		catch(DAVException e) {
		    LogService.log(Priority.ERROR, "DAVStorage::invalidDocument() : " + e.getMessage());
		    con.close();
			throw new StorageException("Erreur lors de l'invalidation du document");
		}
    }

    /**
	 * Cration d'un intranet
	 * @param intranet L'intranet  crer
	 * @throws StorageException
	 */
    public synchronized void createIntranet(Intranet intranet) throws StorageException {
        LogService.log(nivlog, "DAVStorage::createIntranet()");
		// Vrification paramtres
		if(url == null) {
		    LogService.log(Priority.ERROR, "DAVStorage::createIntranet() : url = null");
			throw new StorageException("L'URL n'est pas correctement configur\u00E9e");
		}
		
		// Cration ressource WebDAV
		DAVConnection con = new DAVConnection();
		
		try {
		    con.connect(url, uri, login, pass);
		    String path = intranet.getId() + "/";
		    if(con.dirExists(path)) {
		        con.close();
		        LogService.log(Priority.ERROR, "DAVStorage::createIntranet() : Le dossier " + intranet.getId() + " existe d\u00E9j\u00E0");
		        throw new StorageException("Le dossier " + intranet.getId() + " existe d\u00E9j\u00E0");
		    }
		    con.createDir(path);
		    con.close();
		}
		catch(DAVException e) {
		    LogService.log(Priority.ERROR, "DAVStorage::createIntranet() : " + e.getMessage());
		    con.close();
			throw new StorageException("Erreur lors de la cr\u00E9ation de l'intranet");
		}
    }

    /**
	 * Suppression d'un intranet et de tous ses documents
	 * @param intranet L'intranet  supprimer
	 * @throws StorageException
	 */
    public synchronized void destroyIntranet(String intranet) throws StorageException {
        LogService.log(nivlog, "DAVStorage::destroyIntranet()");
		if(url == null) {
		    LogService.log(Priority.ERROR, "DAVStorage::destroyIntranet() : url = null");
			throw new StorageException("L'URL n'est pas correctement configur\u00E9e");
		}
		LogService.log(Priority.FATAL, "DAVStorage::destroyIntranet() : La suppression de l'intranet " + intranet + " doit \u00EAtre r\u00E9alis\u00E9e manuellement");
    }

    /**
	 * Calcule le rpertoire o lire / crire un document<br>
	 * en fonction de son identifiant
	 * @param id L'identifiant du document
	 * @return Le chemin o lire / crire le document
	 */
	private String getFolder(String id) {
		int number = new Integer(id.substring(1)).intValue();
		return new Integer(number / size).toString();
	}
    
    /**
	 * Liste des paramtres spcifiques  l'implmentation du IStorageControler<br>
	 * Ici cinq paramtres obligatoires :<br>
	 * - url qui contient l'adresse du serveur WebDAV
	 * - uri qui contient le chemin depuis la racine
	 * - login qui contient le nom d'utilisateur
	 * - password qui contient le mot de passe
	 * - size qui contient le nombre maximal de documents par dossier<br>
	 * @param parameters
	 */
    public void setParameters(Hashtable parameters) {
        LogService.log(nivlog, "DAVStorage::setParameters()");
		url = (String)parameters.get("url");
		uri = (String)parameters.get("uri");
		login = (String)parameters.get("login");
		pass = (String)parameters.get("password");
		size = new Integer((String)parameters.get("size")).intValue();
		
		if(!uri.startsWith("/")) {
		    uri = "/" + uri;
		}
		if(!uri.endsWith("/")) {
		    uri = uri + "/";
		}
    }
}
