package org.esupportail.portal.channels.CIntranet.utils;

import java.io.IOException;
import java.io.InputStream;

import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpURL;
import org.apache.commons.httpclient.URIException;
import org.apache.log4j.Priority;
import org.apache.webdav.lib.WebdavResource;
import org.jasig.portal.services.LogService;

/**
 * DAVConnection<br>
 * <br>
 * Cette classe propose des mthodes de base pour utiliser des services WebDAV<br>
 * <br>
 * (c)Copyright <a href="http://www.esup-portail.org">ESup-Portail 2004</a><br>
 * @author <a href="mailto:mathieu.larchet@univ-nancy2.fr">Mathieu Larchet</a>
 * @version 2.0
 * 
 */
public class DAVConnection {
    
    private static Priority nivlog = Priority.DEBUG;
    
    private String url = null;
    private String uri = null;
    private String login = null;
    private String password = null;
    
    private HttpURL httpUrl = null;
    private WebdavResource res = null;
    
    /**
     * Constructeur
     */
    public DAVConnection() {
    }
    
    /**
     * Ouvre une connexion WebDAV
     * @param url L'URL de connexion
     * @param login L'identifiant
     * @param password Le mot de passe
     * @throws DAVException
     */
    public void connect(String url, String uri, String login, String password) throws DAVException {
        LogService.log(nivlog, "DAVConnection::connect()");
        
        this.url = url;
        this.uri = uri;
        this.login = login;
        this.password = password;
        
        try {
            httpUrl = new HttpURL(this.url + this.uri);
            httpUrl.setUserinfo(this.login, this.password);
        }
        catch(URIException e) {
            LogService.log(Priority.ERROR, "DAVConnection::connect() : Mauvaise URL " + this.url + this.uri);
            throw new DAVException("Mauvaise URL : " + this.url + this.uri);
        }
        
        try {
            res = new WebdavResource(httpUrl);
        }
        catch(HttpException e) {
            LogService.log(Priority.ERROR, "DAVConnection::connect() : Erreur de connexion \u00E0 " + this.url + " erreur " + e.getReasonCode());
            throw new DAVException("Erreur de connexion \u00E0 " + this.url + " erreur " + e.getReasonCode());
        }
        catch(IOException e) {
            LogService.log(Priority.ERROR, "DAVConnection::connect() : Erreur de connexion \u00E0 " + this.url + " erreur " + e.getMessage());
            throw new DAVException("Erreur de connexion \u00E0 " + this.url + " erreur " + e.getMessage());
        }
    }
    
    /**
     * Ferme la connexion
     */
    public void close() {
        LogService.log(nivlog, "DAVConnection::close()");
        if(res != null) {
            try {	
                res.close();
            }
            catch(IOException e) {
                LogService.log(Priority.ERROR, "DAVConnection::close() : Erreur de d\u00E9connexion " + e.getMessage());
            }
        }
    }

    /**
     * Teste l'existence d'un rpertoire
     * @param path Le chemin relatif  tester
     * @return boolean true si le rpertoire existe, false sinon
     */
    public boolean dirExists(String path) {
        LogService.log(nivlog, "DAVConnection::dirExists()");
        
        boolean result = false;
        try {
            res.setPath(uri + path);
            result = true;
        }
        catch(HttpException e) {
            // Le rpertoire n'existe pas
        }
        catch(IOException e) {
            LogService.log(Priority.ERROR, "DAVConnection::dirExists() : " + e.getMessage());
        }
        return result;
    }
    
    /**
     * Teste l'existence d'un fichier
     * @param file Le chemin du fichier  tester
     * @return boolean true si le fichier existe, false sinon
     */
    public boolean fileExists(String file) {
        LogService.log(nivlog, "DAVConnection::fileExists()");
        
        return dirExists(file);
    }
    
    /**
     * Cre un rpertoire
     * @param path Le chemin du rpertoire  crer
     * @throws DAVException
     */
    public void createDir(String path) throws DAVException {
        LogService.log(nivlog, "DAVConnection::createDir()");
        
        try {
            res.mkcolMethod(uri + path);
            int httpCode = res.getStatusCode();
            
            if(httpCode != 201) {
                LogService.log(Priority.ERROR, "DAVConnection::createDir() : Erreur lors de la cr\u00E9ation du r\u00E9pertoire " + path + " erreur " + httpCode);
                throw new DAVException("Erreur lors de la cr\u00E9ation du r\u00E9pertoire");
            }
        }
        catch(HttpException e) {
            LogService.log(Priority.ERROR, "DAVConnection::createDir() : Erreur lors de la cr\u00E9ation du r\u00E9pertoire " + path + " erreur " + e.getReasonCode());
            throw new DAVException("Erreur lors de la cr\u00E9ation du r\u00E9pertoire");
        }
        catch(IOException e) {
            LogService.log(Priority.ERROR, "DAVConnection::createDir() : " + e.getMessage());
            throw new DAVException("Erreur lors de la cr\u00E9ation du r\u00E9pertoire");
        }
    }
    
    /**
     * Cre un fichier
     * @param file Le chemin du fichier  crer
     * @param in Les donnes du fichier
     * @throws DAVException
     */
    public void createFile(String file, InputStream in) throws DAVException {
        LogService.log(nivlog, "DAVConnection::createFile()");
        
        try {
            res.putMethod(uri + file, in);
            in.close();
            int httpCode = res.getStatusCode();
            if(httpCode != 201) {
                LogService.log(Priority.ERROR, "DAVConnection::createFile() : Erreur lors de la cr\u00E9ation du fichier " + file + " erreur " + httpCode);
                throw new DAVException("Erreur lors de la cr\u00E9ation du fichier");
            }
        }
        catch(HttpException e) {
            LogService.log(Priority.ERROR, "DAVConnection::createFile() : Erreur lors de la cr\u00E9ation du fichier " + file + " erreur " + e.getReasonCode());
            throw new DAVException("Erreur lors de la cr\u00E9ation du r\u00E9pertoire");
        }
        catch(IOException e) {
            LogService.log(Priority.ERROR, "DAVConnection::createFile() : " + e.getMessage());
            throw new DAVException("Erreur lors de la cr\u00E9ation du fichier");
        }
    }
    
    /**
     * Renomme un fichier
     * @param source L'ancien chemin du fichier
     * @param dest Le nouveau chemin du fichier
     * @throws DAVException
     */
    public void renameFile(String source, String dest) throws DAVException {
        LogService.log(nivlog, "DAVConnection::renameFile()");
        
        try {
            res.setPath(uri + source);
            res.moveMethod(uri + dest);
            int httpCode = res.getStatusCode();
            if(httpCode != 201) {
                LogService.log(Priority.ERROR, "DAVConnection::renameFile() : Erreur lors du renommage du fichier " + source + " erreur " + httpCode);
                throw new DAVException("Erreur lors du renommage du fichier");
            }
        }
        catch(HttpException e) {
            LogService.log(Priority.ERROR, "DAVConnection::renameFile() : Erreur lors du renommage du fichier " + source + " erreur " + e.getReasonCode());
            throw new DAVException("Erreur lors du renommage du fichier");
        }
        catch(IOException e) {
            LogService.log(Priority.ERROR, "DAVConnection::renameFile() : " + e.getMessage());
            throw new DAVException("Erreur lors du renommage du fichier");
        }
    }
    
    /**
     * Supprime un fichier
     * @param file Le chemin du fichier  supprimer
     * @throws DAVException
     */
    public void deleteFile(String file) throws DAVException {
        LogService.log(nivlog, "DAVConnection::deleteFile()");
        
        try {
            res.deleteMethod(uri + file);
            int httpCode = res.getStatusCode();
            if(httpCode != 204) {
                LogService.log(Priority.ERROR, "DAVConnection::deleteFile() : Erreur lors de la suppression du fichier " + file + " erreur " + httpCode);
                throw new DAVException("Erreur lors de la suppression du fichier");
            }
        }
        catch(HttpException e) {
            LogService.log(Priority.ERROR, "DAVConnection::deleteFile() : Erreur lors de la suppression du fichier " + file + " erreur " + e.getReasonCode());
            throw new DAVException("Erreur lors de la suppression du fichier");
        }
        catch(IOException e) {
            LogService.log(Priority.ERROR, "DAVConnection::deleteFile() : " + e.getMessage());
            throw new DAVException("Erreur lors de la suppression du fichier");
        }
    }
    
    /**
     * Lit un fichier
     * @param file Le chemin du fichier  lire
     * @return InputStream les donnes du fichier
     * @throws DAVException
     */
    public InputStream getFile(String file) throws DAVException {
        LogService.log(nivlog, "DAVConnection::getFile()");
        InputStream result = null;
        
        try {
            result = res.getMethodData(uri + file);
            int httpCode = res.getStatusCode();
            if(httpCode != 200) {
                LogService.log(Priority.ERROR, "DAVConnection::getFile() : Erreur lors de la lecture du fichier " + file + " erreur " + httpCode);
                throw new DAVException("Erreur lors de la lecture du fichier");
            }
        }
        catch(HttpException e) {
            LogService.log(Priority.ERROR, "DAVConnection::getFile() : Erreur lors de la lecture du fichier " + file + " erreur " + e.getReasonCode());
            throw new DAVException("Erreur lors de la lecture du fichier");
        }
        catch(IOException e) {
            LogService.log(Priority.ERROR, "DAVConnection::getFile() : " + e.getMessage());
            throw new DAVException("Erreur lors de la lecture du fichier");
        }
        return result;
    }
}
