package org.esupportail.portal.channels.CIntranet.data;

import java.sql.Connection;
import java.util.Collection;
import java.util.Iterator;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.esupportail.portal.channels.CIntranet.beans.Document;
import org.esupportail.portal.channels.CIntranet.beans.Folder;
import org.esupportail.portal.channels.CIntranet.beans.FolderTree;
import org.esupportail.portal.channels.CIntranet.beans.Intranet;
import org.esupportail.portal.channels.CIntranet.beans.Mime;
import org.esupportail.portal.channels.CIntranet.beans.Population;
import org.esupportail.portal.channels.CIntranet.beans.SubFolder;
import org.esupportail.portal.channels.CIntranet.config.Config;
import org.esupportail.portal.utils.database.Query;

/**
 * DataControler<br>
 * <br>
 * Controleur d'accs aux donnes<br>
 * <br>
 * (c)Copyright <a href="http://www.esup-portail.org">ESup-Portail 2004</a><br>
 * @author <a href="mailto:mathieu.larchet@univ-nancy2.fr">Mathieu Larchet</a>
 * @version 2.0
 * 
 */
public class DataControler implements IDataControler {
    
    protected static final Log log = LogFactory.getLog(DataControler.class);
        
    private boolean session = false;
    private Query query = null;
    
    /**
     * Constructeur
     */
    public DataControler() {
    }

    /**
     * Dmarre une session
     */
    private void sessionStart() {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::sessionStart()");
        }
        this.session = true;
        query = Config.getInstance().getConnexionDefault();
        query.setAutoCommit(false);
        query.setTransactionIsolation(Connection.TRANSACTION_SERIALIZABLE);
    }

    /**
     * Clt une session
     */
    private void sessionStop() {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::sessionStop()");
        }
        query.close();
        query = null;
        session = false;
    }

    /**
     * Valide les oprations de la session
     */
    public void commit() {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::commit()");
        }
        if(session) {
            query.commit();
            sessionStop();
        }
    }
    
    /**
     * Annule les oprations de la session
     */
    public void rollback() {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::rollback()");
        }
        if(session) {
            query.rollback();
            sessionStop();
        }
    }

    /**
     * Dmarre une opration
     */
    private void open() {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::open()");
        }
        if(!session) {
            query = Config.getInstance().getConnexionDefault();
        }
    }
    
    /**
     * Clt une opration
     */
    
    private void close() {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::close()");
        }
        if(!session) {
            query.close();
            query = null;
        }
    }
    
    /**
	 * Rcupration d'un document
	 * @param intranet L'identifiant de l'intranet auquel appartient le document
	 * @param id L'identifiant du dossier auquel appartient le document
	 * @return Le document
	 * @throws DataException
	 */
	public Document getDocument(String intranet, String id) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::getDocument()");
	    }
	    open();
	    IDataDocument data = DataFactory.makeDocument();
	    Document res = null;
	    try {
	        res = data.getDocument(query, intranet, id);
	    }
	    catch(DataException e) {
	        close();
	        throw e;
	    }
	    close();
	    return res;
	}
	
	/**
	 * Cration d'un document
	 * @param doc Le document  crer
	 * @param valid Si le document est valide true, false sinon
	 */
	public void createDocument(Document doc, boolean valid) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::createDocument()");
	    }
	    sessionStart();
	    IDataDocument dataDoc = DataFactory.makeDocument();
	    IDataOrder dataOrd = DataFactory.makeOrder();
	    boolean exist = false;
	    try {
	        exist = dataDoc.documentExist(query, doc);
	    }
	    catch(DataException e) {
	        sessionStop();
	        throw e;
	    }
	    if(exist) {
	        sessionStop();
	        throw new DataException("Un document portant ce nom existe d\u00E9j\u00E0");
	    }
	    try {
	        int id = dataDoc.getNextDocument(query, doc.getIntranet());
	        if(valid) {
	            doc.setId("v" + id);
	        }
	        else {
	            doc.setId("i"+ id);
	        }
	        dataOrd.push(query, doc.getFolder());
	        dataDoc.createDocument(query, doc);
	    }
	    catch(DataException e) {
	        rollback();
	        throw e;
	    }
	}
	
	/**
	 * Suppression d'un document
	 * @param doc Le document  supprimer
	 * @throws DataException
	 */
	public void removeDocument(Document doc) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::removeDocument()");
	    }
	    sessionStart();
	    IDataDocument dataDoc = DataFactory.makeDocument();
	    IDataOrder dataOrd = DataFactory.makeOrder();
	    boolean locked = false;
        try {
            locked = dataDoc.documentIsLocked(query, doc.getId(), doc.getIntranet());
        }
        catch(DataException e) {
            sessionStop();
            throw e;
        }
        if(locked) {
            throw new DataException("Impossible de modifier un document verrouill\u00E9");
        }
	    if(!doc.isValid()) {
	        try {
	            dataDoc.unlockDocument(query, "v" + doc.getId().substring(1), doc.getIntranet());
	        }
	        catch(DataException e) {
	            rollback();
	            throw e;
	        }
	    }
	    try {
	        dataDoc.removeDocument(query, doc.getId(), doc.getIntranet());
	        dataOrd.pull(query, doc.getFolder(), doc.getOrder());
	    }
	    catch(DataException e) {
	        rollback();
	        throw e;
	    }
	}
	
	/**
	 * Mise  jour d'un document
	 * @param doc Le document  mettre  jour
	 * @throws DataException
	 */
	public void updateDocument(Document doc) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::updateDocument()");
	    }
	    sessionStart();
	    IDataDocument data = DataFactory.makeDocument();
	    if(doc.isValid()) {
	        boolean locked = false;
	        try {
	            locked = data.documentIsLocked(query, doc.getId(), doc.getIntranet());
	        }
	        catch(DataException e) {
	            sessionStop();
	            throw e;
	        }
	        if(locked) {
	            throw new DataException("Impossible de modifier un document verrouill\u00E9");
	        }
	        try {
	            data.updateDocument(query, doc);
	        }
	        catch(DataException e) {
	            sessionStop();
	            throw e;
	        }
	    }
	    else {
	        try {
	            data.lockDocument(query, "v" + doc.getId().substring(1), doc.getIntranet());
	            data.createDocument(query, doc);
	        }
	        catch(DataException e) {
	            rollback();
	            throw e;
	        }
	    }
	}
	
	/**
	 * Validation d'un document
	 * @param doc Le document  valider
	 * @throws DataException
	 */
	public void validDocument(Document doc) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::validDocument()");
	    }
	    sessionStart();
	    IDataDocument data = DataFactory.makeDocument();
	    try {
	        data.removeDocument(query, doc.getId(), doc.getIntranet());
	        data.validDocument(query, doc.getId(), doc.getIntranet());
	    }
	    catch(DataException e) {
	        rollback();
	        throw e;
	    }
	}
	
	/**
	 * Invalidation d'un document
	 * @param doc Le document  invalider
	 * @throws DataException
	 */
	public void invalidDocument(Document doc) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::invalidDocument()");
	    }
	    sessionStart();
	    IDataDocument data = DataFactory.makeDocument();
	    try {
	        data.invalidDocument(query, doc.getId(), doc.getIntranet());
	    }
	    catch(DataException e) {
	        rollback();
	        throw e;
	    }
	}
	
	/**
	 * Rcupration d'un dossier
	 * @param id L'identifiant du dossier
	 * @param sub Si true alors on rcupre galement les sous-dossiers<br>
	 * et les documents
	 * @return Le dossier
	 * @throws DataException
	 */
	public Folder getFolder(String id, boolean sub) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::getFolder()");
	    }
	    open();
	    IDataFolder dataFol = DataFactory.makeFolder();
	    IDataDocument dataDoc = null;
	    if(sub) {
	        dataDoc = DataFactory.makeDocument();
	    }
	    Folder res = null;
	    try {
	        res = dataFol.getFolder(query, id);
	        if(sub) {
	            res.setDocuments(dataDoc.getDocuments(query, id));
	            res.setSubFolders(dataFol.getSubFolders(query, id));
	            res.setPath(dataFol.getPath(query, id));
	        }
	    }
	    catch(DataException e) {
	        close();
	        throw e;
	    }
	    close();
	    return res;
	}
	
	/**
	 * Cration d'un dossier
	 * @param folder Le dossier  crer
	 * @return L'identifiant du dossier cr
	 */
	public void createFolder(Folder folder) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::createFolder()");
	    }
	    sessionStart();
	    IDataFolder dataFol = DataFactory.makeFolder();
	    IDataPublicator dataPub = DataFactory.makePublicator();
	    IDataValidator dataVal = DataFactory.makeValidator();
	    
	    boolean exist = false;
	    try {
	        exist = dataFol.folderExist(query, folder);
	    }
	    catch(DataException e) {
	        sessionStop();
	        throw e;
	    }
	    if(exist) {
	        throw new DataException("Un dossier portant ce nom existe d\u00E9j\u00E0");
	    }
	    try {
	        folder.setId(new Integer(dataFol.getNextFolder(query)).toString());
	        dataFol.createFolder(query, folder);
	        Collection validators = dataVal.getValidators(query, folder.getParent());
	        Collection users = dataPub.getUsersPublicators(query, folder.getParent());
	        Collection groups = dataPub.getGroupsPublicators(query, folder.getParent());
	        dataVal.addValidator(query, folder.getId(), validators);
	        dataPub.addPublicator(query, folder.getId(), users);
	        dataPub.addPublicators(query, folder.getId(), groups);
	    }
	    catch(DataException e) {
	        rollback();
	        throw e;
	    } 
	    commit();
	}
	
	/**
	 * Suppression d'un dossier
	 * @param id L'identifiant du dossier
	 * @throws DataException
	 */
	public void removeFolder(String id) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::removeFolder()");
	    }
	    sessionStart();
	    IDataFolder dataFol = DataFactory.makeFolder();
	    IDataPublicator dataPub = DataFactory.makePublicator();
	    IDataValidator dataVal = DataFactory.makeValidator();
	    Folder tmp = null;
	    try {
	        tmp = dataFol.getFolder(query, id);
	    }
	    catch(DataException e) {
	        sessionStop();
	        throw e;
	    }
	    if(tmp.isRoot()) {
	        throw new DataException("Impossible de supprimer la racine de l'intranet");
	    }
	    boolean empty = false;
	    try {
	        empty = dataFol.isFolderEmpty(query, id);
	    }
	    catch(DataException e) {
	        sessionStop();
	        throw e;
	    }
	    if(!empty) {
	        throw new DataException("Impossible de supprimer un dossier non vide");
	    }
	    try {
	        dataFol.removeFolder(query, id);
	        dataVal.removeAllValidators(query, id);
	        dataPub.removeAllPublicators(query, id);
	    }
	    catch(DataException e) {
	        rollback();
	        throw e;
	    }
	    commit();
	}
	
	/**
	 * Modifie la validation d'un dossier
	 * @param folder Le dossier
	 * @param validation True pour activer la validation, false pour la dsactiver
	 * @throws DataException
	 */
	public void setValidation(Folder folder, boolean validation) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::updateFolder()");
	    }
	    open();
	    IDataFolder dataFol = DataFactory.makeFolder();
	    boolean empty = false;
	    try {
	        empty = dataFol.isFolderEmpty(query, folder.getId());
	    }
	    catch(DataException e) {
	        sessionStop();
	        throw e;
	    }
	    if(!empty) {
	        throw new DataException("Impossible de modifier un dossier non vide");
	    }
	    try {
	        dataFol.setValidation(query, folder.getId(), validation);
	    }
	    catch(DataException e) {
	        close();
	        throw e;
	    }
	    close();
	}
	
	/**
	 * Modifie le nom d'un dossier
	 * @param folder Le dossier
	 * @throws DataException
	 */
	public void setName(Folder folder) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::updateFolder()");
	    }
	    open();
	    IDataFolder dataFol = DataFactory.makeFolder();
	    boolean empty = false;
	    try {
	        empty = dataFol.isFolderEmpty(query, folder.getId());
	    }
	    catch(DataException e) {
	        sessionStop();
	        throw e;
	    }
	    if(!empty) {
	        throw new DataException("Impossible de modifier un dossier non vide");
	    }
	    boolean exist = false;
	    try {
	        exist = dataFol.folderExist(query, folder);
	    }
	    catch(DataException e) {
	        sessionStop();
	        throw e;
	    }
	    if(exist) {
	        throw new DataException("Un dossier portant ce nom existe d\u00E9j\u00E0");
	    }
	    try {
	        dataFol.setName(query, folder.getId(), folder.getName());
	    }
	    catch(DataException e) {
	        close();
	        throw e;
	    }
	    close();
	}
	
	/**
	 * Rcupration d'un intranet
	 * @param id L'identifiant d'intranet
	 * @return L'intranet correspondant
	 * @throws DataException
	 */
	public Intranet getIntranet(String id) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::getIntranet()");
	    }
	    open();
	    IDataIntranet data = DataFactory.makeIntranet();
	    Intranet res = null;
	    try {
	        res = data.getIntranet(query, id);
	    }
	    catch(DataException e) {
	        close();
	        throw e;
	    }
	    close();
	    return res;
	}

	/**
	 * Rcupration de la liste de tous les intranets
	 * @return La liste de tous les intranets
	 * @throws DataException
	 */
	public Collection getIntranets() throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::getIntranets()");
	    }
	    open();
	    IDataIntranet data = DataFactory.makeIntranet();
	    Collection res = null;
	    try {
	        res = data.getIntranets(query);
	    }
	    catch(DataException e) {
	        close();
	        throw e;
	    }
	    close();
	    return res;
	}
	
	/**
	 * Cration d'un intranet
	 * @param intranet L'intranet  crer
	 */
	public void createIntranet(Intranet intranet) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::createIntranet()");
	    }
	    sessionStart();
	    IDataFolder dataFol = DataFactory.makeFolder();
	    IDataIntranet dataInt = DataFactory.makeIntranet();
	    boolean exist = false;
	    try {
	        exist = dataInt.intranetExist(query, intranet);
	    }
	    catch(DataException e) {
	        sessionStop();
	        throw e;
	    }
	    if(exist) {
	        throw new DataException("Un intranet porte d\u00E9j\u00E0 cet identifiant");
	    }
	    Folder folder = new Folder();
		folder.setIntranet(intranet.getId());
		folder.setName(intranet.getName());
		folder.setParent("0");
		folder.setValidation(true);
		try {
		    folder.setId(new Integer(dataFol.getNextFolder(query)).toString());
		    dataFol.createFolder(query, folder);
		    dataInt.createIntranet(query, intranet);
		}
		catch(DataException e) {
		    rollback();
		    throw e;
		}
	}
	
	/**
	 * Suppression d'un intranet
	 * @param intranet L'intranet  supprimer
	 * @throws DataException
	 */
	public void removeIntranet(Intranet intranet) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::removeIntranet()");
	    }
	    sessionStart();
	    IDataIntranet dataInt = DataFactory.makeIntranet();
	    IDataFolder dataFol = DataFactory.makeFolder();
	    IDataPopulation dataPop = DataFactory.makePopulation();
	    IDataPublicator dataPub = DataFactory.makePublicator();
	    IDataValidator dataVal = DataFactory.makeValidator();
	    
	    try {
	        Collection folders = dataFol.getFolderChildren(query, intranet.getRoot());
	        Iterator i = folders.iterator();
	        while(i.hasNext()) {
	            SubFolder tmp = (SubFolder)i.next();
	            dataPub.removeAllPublicators(query, tmp.getId());
	            dataVal.removeAllValidators(query, tmp.getId());
	            dataFol.removeFolder(query, tmp.getId());
	        }
	        dataPub.removeAllPublicators(query, intranet.getRoot());
	        dataVal.removeAllValidators(query, intranet.getRoot());
	        dataFol.removeFolder(query, intranet.getRoot());
	        dataInt.removeIntranet(query, intranet);
	    }
	    catch(DataException e) {
	        rollback();
	        throw e;
	    }
	}
		
	/**
	 * Rcupration des types MIME valides
	 * @return La liste des types MIME
	 * @throws DataException
	 */
	public Collection getMIMEs() throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::getMIMEs()");
	    }
	    open();
	    IDataMIME data = DataFactory.makeMIME();
	    Collection res = null;
	    try {
	        res = data.getMIMEs(query);
	    }
	    catch(DataException e) {
	        close();
	        throw e;
	    }
	    close();
	    return res;
	}
	
	/**
	 * Cration d'un type MIME
	 * @param mime Le type MIME  crer
	 */
	public void createMIME(Mime mime) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::createMIME()");
	    }
	    sessionStart();
	    IDataMIME data = DataFactory.makeMIME();
	    boolean exist = false;
	    try {
	        exist = data.isValidMIME(query, mime.getExtension());
	    }
	    catch(DataException e) {
	        sessionStop();
	        throw e;
	    }
	    if(exist) {
	        throw new DataException("Cette extension est d\u00E9j\u00E0 associ\u00E9e \u00E0 un type MIME");
	    }
	    try {
	        data.createMIME(query, mime);
	    }
	    catch(DataException e) {
	        rollback();
	        throw e;
	    }
	    commit();
	}
	
	/**
	 * Suppression d'un type MIME
	 * @param extension L'extension associe au type MIME
	 * @throws DataException
	 */
	public void removeMIME(String extension) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::removeMIME()");
	    }
	    sessionStart();
	    IDataMIME data = DataFactory.makeMIME();
	    try {
	        data.removeMIME(query, extension);
	    }
	    catch(DataException e) {
	        rollback();
	        throw e;
	    }
	    commit();
	}
	
	/**
	 * Vrifie la validit d'un type MIME
	 * @param extension L'extension  tester
	 * @return boolean True si le type est valide, false sinon
	 * @throws DataException
	 */
	public boolean isValidMIME(String extension) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::isValidMIME()");
	    }
	    open();
	    boolean res = false;
	    IDataMIME data = DataFactory.makeMIME();
	    try {
	        res = data.isValidMIME(query, extension);
	    }
	    catch(DataException e) {
	        close();
	        throw e;
	    }
	    close();
	    return res;
	}
	
	/**
	 * Rcupration d'une population
	 * @param id L'identifiant d'intranet
	 * @return La population de l'intranet
	 * @throws DataException
	 */
	public Population getPopulation(String id) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::getPopulation()");
	    }
	    open();
	    IDataPopulation data = DataFactory.makePopulation();
	    Population res = null;
	    try {
	        res = data.getPopulation(query, id);
	    }
	    catch(DataException e) {
	        close();
	        throw e;
	    }
	    close();
	    return res;
	}
	
	/**
	 * Ajout d'une population
	 * @param population La population  ajouter
	 */
	public void addPopulation(Population population) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::addPopulation()");
	    }
	    sessionStart();
	    IDataPopulation data = DataFactory.makePopulation();
	    try {
	        data.addPopulation(query, population);
	    }
	    catch(DataException e) {
	        rollback();
	        throw e;
	    }
	    commit();
	}
	
	/**
	 * Rcupration des publicateurs d'un dossier
	 * @param id L'identifiant du dossier
	 * @return La liste des publicateurs
	 * @throws DataException
	 */
	public Collection getUsersPublicators(String id) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::getUsersPublicators()");
	    }
	    open();
	    IDataPublicator data = DataFactory.makePublicator();
	    Collection res = null;
	    try {
	        res = data.getUsersPublicators(query, id);
	    }
	    catch(DataException e) {
	        close();
	        throw e;
	    }
	    close();
	    return res;
	}
	
	/**
	 * Rcupration des groupes de publicateurs d'un dossier
	 * @param id L'identifiant du dossier
	 * @return La liste des groupes de publicateurs
	 * @throws DataException
	 */
	public Collection getGroupsPublicators(String id) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::getGroupsPublicators()");
	    }
	    open();
	    IDataPublicator data = DataFactory.makePublicator();
	    Collection res = null;
	    try {
	        res = data.getGroupsPublicators(query, id);
	    }
	    catch(DataException e) {
	        close();
	        throw e;
	    }
	    close();
	    return res;
	}
	
	/**
	 * Ajout d'un publicateur
	 * @param id L'identifiant du dossier
	 * @param login L'identifiant de l'utilisateur
	 */
	public void addPublicator(String id, String login) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::addPublicator()");
	    }
	    sessionStart();
	    IDataFolder dataFol = DataFactory.makeFolder();
	    IDataPublicator dataPub = DataFactory.makePublicator();
	    try {
	        dataPub.addPublicator(query, id, login);
	        Collection children = dataFol.getFolderChildren(query, id);
	        Iterator i = children.iterator();
	        while(i.hasNext()) {
	            SubFolder tmp = (SubFolder)i.next();
	            dataPub.addPublicator(query, tmp.getId(), login);
	        }
	    }
	    catch(DataException e) {
	        rollback();
	        throw e;
	    }
	    commit();
	}
	
	/**
	 * Ajout d'un groupe de publicateurs
	 * @param id L'identifiant du dossier
	 * @param group L'identifiant du groupe
	 * @throws DataException
	 */
	public void addPublicators(String id, String group) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::addPublicators()");
	    }
	    sessionStart();
	    IDataFolder dataFol = DataFactory.makeFolder();
	    IDataPublicator dataPub = DataFactory.makePublicator();
	    try {
	        dataPub.addPublicators(query, id, group);
	        Collection children = dataFol.getFolderChildren(query, id);
	        Iterator i = children.iterator();
	        while(i.hasNext()) {
	            SubFolder tmp = (SubFolder)i.next();
	            dataPub.addPublicators(query, tmp.getId(), group);
	        }
	    }
	    catch(DataException e) {
	        rollback();
	        throw e;
	    }
	    commit();
	}
	
	/**
	 * Suppression d'un publicateur
	 * @param id L'identifiant du dossier
	 * @param login L'identifiant de l'utilisateur
	 * @throws DataException
	 */
	public void removePublicator(String id, String login) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::removePublicator()");
	    }
	    sessionStart();
	    IDataFolder dataFol = DataFactory.makeFolder();
	    IDataPublicator dataPub = DataFactory.makePublicator();
	    try {
	        dataPub.removePublicator(query, id, login);
	        Collection children = dataFol.getFolderChildren(query, id);
	        Iterator i = children.iterator();
	        while(i.hasNext()) {
	            SubFolder tmp = (SubFolder)i.next();
	            dataPub.removePublicator(query, tmp.getId(), login);
	        }
	    }
	    catch(DataException e) {
	        rollback();
	        throw e;
	    }
	    commit();
	}
	
	/**
	 * Suppression d'un groupe de publicateurs
	 * @param id L'identifiant du dossier
	 * @param group L'identifiant du groupe
	 * @throws DataException
	 */
	public void removePublicators(String id, String group) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::removePublicators()");
	    }
	    sessionStart();
	    IDataFolder dataFol = DataFactory.makeFolder();
	    IDataPublicator dataPub = DataFactory.makePublicator();
	    try {
	        dataPub.removePublicators(query, id, group);
	        Collection children = dataFol.getFolderChildren(query, id);
	        Iterator i = children.iterator();
	        while(i.hasNext()) {
	            SubFolder tmp = (SubFolder)i.next();
	            dataPub.removePublicators(query, tmp.getId(), group);
	        }
	    }
	    catch(DataException e) {
	        rollback();
	        throw e;
	    }
	    commit();
	}
	
	/**
	 * Rcupration des super-utilisateurs
	 * @return La liste des super-utilisateurs
	 * @throws DataException
	 */
	public Collection getSuperUsers() throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::getSuperUsers()");
	    }
	    open();
	    IDataSU data = DataFactory.makeSU();
	    Collection res = null;
	    try {
	        res = data.getSuperUsers(query);
	    }
	    catch(DataException e) {
	        close();
	        throw e;
	    }
	    close();
	    return res;
	}
	
	/**
	 * Ajout d'un super-utilisateur
	 * @param login L'identifiant de l'utilisateur
	 */
	public void addSuperUser(String login) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::addSuperUser()");
	    }
	    open();
	    IDataSU data = DataFactory.makeSU();
	    try {
	        data.addSuperUser(query, login);
	    }
	    catch(DataException e) {
	        close();
	        throw e;
	    }
	    close();
	}
	
	/**
	 * Suppression d'un super-utilisateur
	 * @param login L'identifiant de l'utilisateur
	 * @throws DataException
	 */
	public void removeSuperUser(String login) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::removeSuperUser()");
	    }
	    open();
	    IDataSU data = DataFactory.makeSU();
	    try {
	        data.removeSuperUser(query, login);
	    }
	    catch(DataException e) {
	        close();
	        throw e;
	    }
	    close();
	}
	
	/**
	 * Rcupration des valideurs d'un dossier
	 * @param id L'identifiant du dossier
	 * @return La liste des valideurs
	 * @throws DataException
	 */
	public Collection getValidators(String id) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::getValidators()");
	    }
	    open();
	    IDataValidator data = DataFactory.makeValidator();
	    Collection res = null;
	    try {
	        res = data.getValidators(query, id);
	    }
	    catch(DataException e) {
	        close();
	        throw e;
	    }
	    close();
	    return res;
	}
	
	/**
	 * Ajout d'un valideur
	 * @param id L'identifiant du dossier
	 * @param login L'identifiant de l'utilisateur
	 */
	public void addValidator(String id, String login) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::addValidator()");
	    }
	    sessionStart();
	    IDataFolder dataFol = DataFactory.makeFolder();
	    IDataValidator dataVal = DataFactory.makeValidator();
	    try {
	        dataVal.addValidator(query, id, login);
	        Collection children = dataFol.getFolderChildren(query, id);
	        Iterator i = children.iterator();
	        while(i.hasNext()) {
	            SubFolder tmp = (SubFolder)i.next();
	            dataVal.addValidator(query, tmp.getId(), login);
	        }
	    }
	    catch(DataException e) {
	        rollback();
	        throw e;
	    }
	    commit();
	}
	
	/**
	 * Suppression d'un valideur
	 * @param id L'identifiant du dossier
	 * @param login L'identifiant de l'utilisateur
	 * @throws DataException
	 */
	public void removeValidator(String id, String login) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::removeValidator()");
	    }
	    sessionStart();
	    IDataFolder dataFol = DataFactory.makeFolder();
	    IDataValidator dataVal = DataFactory.makeValidator();
	    try {
	        dataVal.removeValidator(query, id, login);
	        Collection children = dataFol.getFolderChildren(query, id);
	        Iterator i = children.iterator();
	        while(i.hasNext()) {
	            SubFolder tmp = (SubFolder)i.next();
	            dataVal.removeValidator(query, tmp.getId(), login);
	        }
	    }
	    catch(DataException e) {
	        rollback();
	        throw e;
	    }
	    commit();
	}

	/**
     * Echange le rang de deux documents
     * @param id L'identifiant du dossier
     * @param doc L'identifiant du document
     * @param oldorder L'ancien rang du document
     * @param neworder Le nouveau rang du document
     * @throws DataException
     */
	public void swap(String id, String doc, String oldorder, String neworder) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::swap()");
	    }
	    sessionStart();
	    IDataOrder data = DataFactory.makeOrder();
	    try {
	        data.swap(query, id, doc, oldorder, neworder);
	    }
	    catch(DataException e) {
	        rollback();
	        throw e;
	    }
	    commit();
    }
	
	/**
     * Construit rcursivement l'arborescence d'un dossier
     * @param id L'identifiant du dossier
     * @return L'arbre
     * @throws DataException
     */
	public FolderTree getFolderTree(String id) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::getFolderTree()");
	    }
	    open();
	    IDataFolder data = DataFactory.makeFolder();
	    FolderTree res = null;
	    try {
	        SubFolder tmp = null;
	        tmp = data.getSubFolder(query, id);
	        while(!tmp.getParent().equals("0")) {
	            tmp = data.getSubFolder(query, tmp.getParent());
	        }
	        
	        res = new FolderTree();
	        res.setId(tmp.getId());
	        res.setName(tmp.getName());
	        res.setIntranet(tmp.getIntranet());
	        res.setParent(tmp.getParent());
	        res.setSubfolder(data.getFolderTree(query, tmp.getId()));
	    }
	    catch(DataException e) {
	        close();
	        throw e;
	    }
	    close();
	    return res;
    }
	
	protected void finalize() throws Throwable {
        super.finalize();
        if(query != null) {
            log.error("DataControler::finalize() : Des connexions n'ont pas \u00E9t\u00E9 ferm\u00E9es");
        }
    }

    
}
