package org.esupportail.portal.channels.CIntranet.security;

import java.util.Collection;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.esupportail.portal.channels.CIntranet.beans.Population;
import org.esupportail.portal.channels.CIntranet.beans.User;
import org.esupportail.portal.channels.CIntranet.data.DataException;
import org.esupportail.portal.channels.CIntranet.data.DataFactory;
import org.esupportail.portal.channels.CIntranet.data.IDataControler;
import org.esupportail.portal.channels.CIntranet.groups.GroupException;
import org.esupportail.portal.channels.CIntranet.groups.GroupFactory;
import org.esupportail.portal.channels.CIntranet.groups.IGroupManager;

/**
 * SecurityManager<br>
 * <br>
 * Implmentation d'un ISecurityManager utilisant les fonctionnalits du IDatabaseManager<br>
 * et du IGroupManager pour raliser ses fonctionnalits<br>
 * <br>
 * (c)Copyright <a href="http://www.esup-portail.org">ESup-Portail 2004</a><br>
 * @author <a href="mailto:mathieu.larchet@univ-nancy2.fr">Mathieu Larchet</a>
 * @version 2.0
 * 
 */

public class SecurityManager implements ISecurityManager {
	
    protected static final Log log = LogFactory.getLog(SecurityManager.class);
	
	/**
	* Constructeur
	*/
	public SecurityManager() {
	}

	/**
	 * Si l'utilisateur est un super-utilisateur
	 * @param login Le login de l'utilisateur
	 * @return true si c'est un super-utilisateur, false sinon
	 * @throws SecurityException
	 */
	public synchronized boolean isSuperUser(String login) throws SecurityException {
		if(log.isDebugEnabled()) {
		    log.debug("SecurityManager::isSuperUser()");
		}
		
		User user = new User();
		user.setLogin(login);
		
		// Source de donnes
		IDataControler data = DataFactory.make();
		
		try {
			Collection su = data.getSuperUsers();
			return su.contains(user);
		}
		catch(DataException e) {
			log.error("SecurityManager::isSuperUser() : " + e);
			throw new SecurityException(e.getMessage());
		}
	}

	/**
	 * Si l'utilisateur fait partie de la population d'un intranet
	 * @param intranet L'identifiant de l'intranet
	 * @param login L'identifiant de l'utilisateur
	 * @return true si l'utilisateur fait partie de la population, false sinon
	 * @throws SecurityException
	 */
	public synchronized boolean isIntranetAccessible(String intranet, String login) throws SecurityException {
	    if(log.isDebugEnabled()) {
		    log.debug("SecurityManager::isIntranetAccessible()");
	    }
		
		User user = new User();
		user.setLogin(login);
		
		// Source de donnes
		IDataControler data = DataFactory.make();
		
		Population members = null;
		try {
			members = data.getPopulation(intranet);
		}
		catch(DataException e) {
		    log.error("SecurityManager::isIntranetAccessible() : " + e);
			throw new SecurityException(e.getMessage());
		}
		
		Collection users = members.getUsers();
		if(users.contains(user)) {
			return true;
		}
		
		// Gestion des groupes
		IGroupManager group = GroupFactory.make();
		
		Collection groups = members.getGroups();
		try {
			return group.isMemberOf(login, groups);
		}
		catch(GroupException e) {
		    log.error("SecurityManager::isIntranetAccessible() : " + e);
			throw new SecurityException(e.getMessage());
		}
	}

	/**
	 * Retourne une constante indiquant le statut d'un utilisateur pour un dossier donn
	 * @param folder L'identifiant du dossier
	 * @param login L'identifiant de l'utilisateur
	 * @return USER si c'est un utilisateur standard<br>
	 * PUBLICATOR si c'est un publicateur
	 * VALIDATOR si c'est un valideur
	 * @throws SecurityException
	 */
	public synchronized int getRoleOfUser(String folder, String login) throws SecurityException {
	    if(log.isDebugEnabled()) {
		    log.debug("SecurityManager::getRoleOfUser()");
	    }
		if(isValidator(login, folder)) {
			return ISecurityManager.VALIDATOR;
		}
		
		if(isPublicator(login, folder)) {
			return ISecurityManager.PUBLICATOR;
		}
		
		return ISecurityManager.USER;
	}
	
	/**
	 * Indique si l'utilisateur est un valideur pour un dossier donn
	 * @param login L'identifiant de l'utilisateur
	 * @param id L'identifiant du dossier
	 * @return true si l'utilisateur est un valideur, false sinon
	 * @throws SecurityException
	 */
	private boolean isValidator(String login, String id) throws SecurityException {
	    if(log.isDebugEnabled()) {
		    log.debug("SecurityManager::isValidator()");
	    }
		
		User user = new User();
		user.setLogin(login);
		
		// Source de donnes
		IDataControler data = DataFactory.make();
		
		try {
			Collection validators = data.getValidators(id);
			return validators.contains(user);
		}
		catch(DataException e) {
		    log.error("SecurityManager::isValidator() : " + e);
			throw new SecurityException(e.getMessage());
		}
	}

	/**
	 * Indique si l'utilisateur est un publicateur pour un dossier donn
	 * @param login L'identifiant de l'utilisateur
	 * @param id L'identifiant du dossier
	 * @return true si l'utilisateur est un publicateur, false sinon
	 * @throws SecurityException
	 */
	private boolean isPublicator(String login, String id) throws SecurityException {
	    if(log.isDebugEnabled()) {
		    log.debug("SecurityManager::isPublicator()");
	    }
		Collection groups = null;
		
		User user = new User();
		user.setLogin(login);
		
		// Source de donnes
		IDataControler data = DataFactory.make();
		
		try {
			Collection publicators = data.getUsersPublicators(id);
			if(publicators.contains(user)) {
				return true;
			}
			groups = data.getGroupsPublicators(id);
		}
		catch(DataException e) {
		    log.error("SecurityManager::isPublicator() : " + e);
			throw new SecurityException(e.getMessage());
		}
		
		// Cas de toute la population
		if(groups.contains("_EVERYBODY")) {
			return true;
		}
		
		// Gestion des groupes
		IGroupManager group = GroupFactory.make();
		
		try {
			return group.isMemberOf(login, groups);
		}
		catch(GroupException e) {
		    log.error("SecurityManager::isPublicator() : " + e);
			throw new SecurityException(e.getMessage());
		}
	}
}
