package org.esupportail.portal.channels.CIntranet.storage;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.esupportail.portal.channels.CIntranet.beans.Document;
import org.esupportail.portal.channels.CIntranet.beans.Intranet;

/**
 * BasicFileSystemStorage<br>
 * <br>
 * Implmentation d'un IStorageControler avec un stockage des documents sur<br>
 * un systme de fichier physique local (ventuellement partag)<br>
 * Chaque intranet se voit attribuer un dossier dans lequel tous les documents<br>
 * sont stocks  la racine<br>
 * <br>
 * (c)Copyright <a href="http://www.esup-portail.org">ESup-Portail 2004</a><br>
 * @author <a href="mailto:mathieu.larchet@univ-nancy2.fr">Mathieu Larchet</a>
 * @version $Revision: 1.1.2.2 $
 * 
 */

public class BasicFileSystemStorage implements IStorageControler {

    protected static final Log log = LogFactory.getLog(BasicFileSystemStorage.class);
	private String root = null;
	
	/**
	* Constructeur
	*/
	public BasicFileSystemStorage() {
	}
	
	/**
	 * Sauvegarde d'un document
	 * @param intranet l'identifiant de l'intranet
	 * @param doc le document
	 * @param in le flux de lecture vers le contenu du document
	 * @throws StorageException
	 */
	public synchronized void saveDocument(String intranet, Document doc, InputStream in) throws StorageException {
		if(log.isDebugEnabled()) {
		    log.debug("BasicFileSystemStorage::saveDocument()");
		}
		// Vrification paramtres
		if(root == null) {
			log.error("BasicFileSystemStorage::saveDocument() : root = null");
			throw new StorageException("Le r\u00E9pertoire de base n'est pas correctement configur\u00E9");
		}
		// Cration du fichier
		File file = new File(root + "/" + intranet + "/" + doc.getId());
		
		// Vrification existence du fichier
		if(file.exists()) {
			log.error("BasicFileSystemStorage::saveDocument() : Le document " + doc.getId() + " existe d\u00E9j\u00E0");
			throw new StorageException("Erreur lors de la sauvegarde du document");
		}
		try {
			// Cration du fichier vide
			if(!file.createNewFile())
			{
				log.error("BasicFileSystemStorage::saveDocument() : Erreur lors de la cr\u00E9ation du fichier");
				throw new StorageException("Erreur lors de la sauvegarde du document");
			}
		}
		catch(IOException e) {
			log.error("BasicFileSystemStorage::saveDocument() : Erreur lors de la cr\u00E9ation du fichier");
			throw new StorageException("Erreur lors de la sauvegarde du document");
		}
		
		// Ecriture des donnes
		try {
			OutputStream out = new FileOutputStream(file);
			
			int size = 0;
            byte[] contentBytes = new byte[8192];
            while ((size = in.read(contentBytes)) != -1) {
                out.write(contentBytes,0, size);
            }
			in.close();
			out.flush();
			out.close();
		}
		catch(FileNotFoundException e) {
			log.error("BasicFileSystemStorage::saveDocument() : Erreur lors de l'\u00E9criture du fichier");
			throw new StorageException("Erreur lors de la sauvegarde du document");
		}
		catch(IOException e) {
			log.error("BasicFileSystemStorage::saveDocument() : Erreur lors de l'\u00E9criture du fichier");
			throw new StorageException("Erreur lors de la sauvegarde du document");
		}
	}

	/**
	 * Lecture d'un document
	 * @param intranet l'identifiant de l'intranet
	 * @param doc le document
	 * @return le flux de lecture vers le document stock
	 * @throws StorageException
	 */
	public synchronized InputStream loadDocument(String intranet, Document doc) throws StorageException {
	    if(log.isDebugEnabled()) {
		    log.debug("BasicFileSystemStorage::loadDocument()");
	    }
		// Vrification paramtres
		if(root == null) {
			log.error("BasicFileSystemStorage::loadDocument() : root = null");
			throw new StorageException("Le r\u00E9pertoire de base n'est pas correctement configur\u00E9");
		}
		
		// Rcupration du fichier
		File loadDoc = new File(root + "/" + intranet + "/" + doc.getId());
				
		try {
			return new FileInputStream(loadDoc);
		}
		catch(FileNotFoundException e) {
			log.error("BasicFileSystemStorage::loadDocument() : Erreur lors de la lecture du document " + doc.getId());
			throw new StorageException("Erreur lors de la lecture du document");
		}
	}
	
	/**
	 * Mise  jour d'un document
	 * @param intranet l'identifiant de l'intranet
	 * @param doc le document
	 * @param in le flux de lecture vers le nouveau contenu, null si pas de mise  jour du contenu
	 * @throws StorageException
	 */
	public synchronized void updateDocument(String intranet, Document doc, InputStream in) throws StorageException {
	    if(log.isDebugEnabled()) {
		    log.debug("BasicFileSystemStorage::updateDocument()");
	    }
		// Vrification paramtres
		if(root == null) {
			log.error("BasicFileSystemStorage::updateDocument() : root = null");
			throw new StorageException("Le r\u00E9pertoire de base n'est pas correctement configur\u00E9");
		}
		
		// Document non valide
		if(!doc.isValid()) {
			if(in == null) {
				// Recopie du fichier valide
				File valid = new File(root + "/" + intranet + "/v" + doc.getId().substring(1));
				File newfile = new File(root + "/" + intranet + "/" + doc.getId());
				try {
					// Cration du fichier vide
					if(!newfile.createNewFile())
					{
						log.error("BasicFileSystemStorage::updateDocument() : Erreur lors de la cr\u00E9ation du fichier");
						throw new StorageException("Erreur lors de la sauvegarde du document");
					}
					InputStream input = new FileInputStream(valid);
					OutputStream out = new FileOutputStream(newfile);
					int size = 0;
		            byte[] contentBytes = new byte[8192];
		            while((size = input.read(contentBytes)) != -1) {
		                out.write(contentBytes,0, size);
		            }
					input.close();
					out.flush();
					out.close();
					return;
				}
				catch(IOException e) {
					log.error("BasicFileSystemStorage::updateDocument() : Erreur lors de la cr\u00E9ation du fichier");
					throw new StorageException("Erreur lors de la sauvegarde du document");
				}
			}
			else {
				// Ecriture du nouveau fichier
				File newfile = new File(root + "/" + intranet + "/" + doc.getId());
				try {
					// Cration du fichier vide
					if(!newfile.createNewFile())
					{
						log.error("BasicFileSystemStorage::updateDocument() : Erreur lors de la cr\u00E9ation du fichier");
						throw new StorageException("Erreur lors de la sauvegarde du document");
					}
					OutputStream out = new FileOutputStream(newfile);
					int size = 0;
		            byte[] contentBytes = new byte[8192];
		            while((size = in.read(contentBytes)) != -1) {
		                out.write(contentBytes,0, size);
		            }
					in.close();
					out.flush();
					out.close();
					return;
				}
				catch(IOException e) {
					log.error("BasicFileSystemStorage::updateDocument() : Erreur lors de la cr\u00E9ation du fichier");
					throw new StorageException("Erreur lors de la sauvegarde du document");
				}
			}
		}
		
		// Document valide
		File valid = new File(root + "/" + intranet + "/" + doc.getId());
		valid.delete();
		try {
			// Cration du fichier vide
			if(!valid.createNewFile())
			{
				log.error("BasicFileSystemStorage::updateDocument() : Erreur lors de la cr\u00E9ation du fichier");
				throw new StorageException("Erreur lors de la sauvegarde du document");
			}
			OutputStream out = new FileOutputStream(valid);
			int size = 0;
            byte[] contentBytes = new byte[8192];
            while((size = in.read(contentBytes)) != -1) {
                out.write(contentBytes,0, size);
            }
			in.close();
			out.flush();
			out.close();
		}
		catch(IOException e) {
			log.error("BasicFileSystemStorage::updateDocument() : Erreur lors de la cr\u00E9ation du fichier");
			throw new StorageException("Erreur lors de la sauvegarde du document");
		}
	}
	
	/**
	 * Suppression d'un document
	 * @param intranet l'identifiant de l'intranet
	 * @param doc le document
	 * @throws StorageException
	 */
	public synchronized void destroyDocument(String intranet, Document doc) throws StorageException {
	    if(log.isDebugEnabled()) {
		    log.debug("BasicFileSystemStorage::destroyDocument()");
	    }
		// Vrification paramtres
		if(root == null) {
			log.error("BasicFileSystemStorage::destroyDocument() : root = null");
			throw new StorageException("Le r\u00E9pertoire de base n'est pas correctement configur\u00E9");
		}
		
		// Rcupration du fichier
		File file = new File(root + "/" + intranet + "/" + doc.getId());
				
		// Suppression du fichier
		if(!file.delete()) {
			log.error("BasicFileSystemStorage::destroyDocument() : Erreur lors de la suppression du fichier " + doc.getId());
			log.fatal("BasicFileSystemStorage::destroyDocument() : Un document supprim\u00E9 de la base est encore stock\u00E9");
			throw new StorageException("Erreur lors de la suppression du document");
		}
	}
	
	/**
	 * Cration d'un intranet
	 * @param intranet l'intranet  crer
	 * @throws StorageException
	 */
	public synchronized void createIntranet(Intranet intranet) throws StorageException {
	    if(log.isDebugEnabled()) {
		    log.debug("BasicFileSystemStorage::createIntranet()");
	    }
		// Vrification paramtres
		if(root == null) {
			log.error("BasicFileSystemStorage::createIntranet() : root = null");
			throw new StorageException("Le r\u00E9pertoire de base n'est pas correctement configur\u00E9");
		}
		
		// Cration du dossier racine
		File newFold = new File(root + "/" + intranet.getId());
		if(newFold.exists()) {
			log.error("BasicFileSystemStorage::createIntranet() : Le dossier " + intranet.getId() + " existe d\u00E9j\u00E0");
			throw new StorageException("Erreur lors de la cr\u00E9ation de l'intranet");
		}
		if(!newFold.mkdir()) {
			log.error("BasicFileSystemStorage::createIntranet() : Erreur lors de la cr\u00E9ation du dossier " + intranet.getId());
			throw new StorageException("Erreur lors de la cr\u00E9ation de l'intranet");
		}
	}

	/**
	 * Suppression d'un intranet et de tous ses documents
	 * @param intranet l'identifiant de l'intranet
	 * @throws StorageException
	 */
	public synchronized void destroyIntranet(String intranet) throws StorageException {
	    if(log.isDebugEnabled()) {
		    log.debug("BasicFileSystemStorage::destroyIntranet()");
	    }
		if(root == null) {
			log.error("BasicFileSystemStorage::destroyIntranet() : root = null");
			throw new StorageException("Le r\u00E9pertoire de base n'est pas correctement configur\u00E9");
		}
		log.fatal("BasicFileSystemStorage::destroyIntranet() : La suppression de l'intranet " + intranet + " doit \u00EAtre r\u00E9alis\u00E9e manuellement");
	}
	
	/**
	 * Liste des paramtres spcifiques  l'implmentation du IStorageControler<br>
	 * Ici un seul paramtre obligatoire :<br>
	 * - path qui indique le chemin o crer les intranets
	 * @param parameters les paramtres
	 */
	public void setParameters(Map parameters) {
	    if(log.isDebugEnabled()) {
		    log.debug("BasicFileSystemStorage::setParameters()");
	    }
		root = (String)parameters.get("path");
	}
}
