package org.esupportail.portal.channels.CIntranet.data;

import java.sql.Connection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.esupportail.portal.channels.CIntranet.beans.Document;
import org.esupportail.portal.channels.CIntranet.beans.Entity;
import org.esupportail.portal.channels.CIntranet.beans.Folder;
import org.esupportail.portal.channels.CIntranet.beans.FolderTree;
import org.esupportail.portal.channels.CIntranet.beans.Grant;
import org.esupportail.portal.channels.CIntranet.beans.Intranet;
import org.esupportail.portal.channels.CIntranet.beans.MIME;
import org.esupportail.portal.channels.CIntranet.config.Config;
import org.esupportail.portal.channels.CIntranet.utils.Date;
import org.esupportail.portal.utils.database.Query;

/**
 * DataControler<br>
 * <br>
 * Classe permettant d'encapsuler les accs  la base de donnes dans des transactions<br>
 * <br>
 * (c)Copyright <a href="http://www.esup-portail.org">ESup-Portail 2004</a><br>
 * @author <a href="mailto:mathieu.larchet@univ-nancy2.fr">Mathieu Larchet</a>
 * @version $Revision: 1.1.2.7 $
 * 
 */
public class DataControler {
    
    protected static Log log = LogFactory.getLog(DataControler.class);
    private Query query = null;
    
    /**
     * Ouvre une connexion
     */
    private void open() {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::start()");
        }
        query = Config.getInstance().getConnexionDefault();
    }
    
    /**
     * Ferme la connexion courante
     */
    public void close() {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::stop()");
        }
        query.close();
        query = null;
    }
    
    /**
     * Ouvre une connexion transactionnelle
     */
    private void session() {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::session()");
        }
        query = Config.getInstance().getConnexionDefault();
        query.setAutoCommit(false);
        query.setTransactionIsolation(Connection.TRANSACTION_SERIALIZABLE);
    }
    
    /**
     * Commit la transaction
     */
    public void commit() {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::commit()");
        }
        query.commit();
    }
    
    /**
     * Rollback la transaction
     */
    public void rollback() {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::rollback()");
        }
        query.rollback();
    }
    
    /**
     * Vrifie que toutes les connexions ont t fermes
     * Override Object.finalize() method
     */
    protected void finalize() throws Throwable {
        super.finalize();
        if(query != null) {
            log.error("DataControler::finalize() : Des connexions n'ont pas \u00E9t\u00E9 ferm\u00E9es");
        }
        query.close();
    }

    /*
     ************************************************
     * Oprations relatives aux intranets
     ************************************************
     */

    /**
     * Rcupre un intranet 
     * @param id L'identifiant de l'intranet  rcuprer
     * @return intranet L'intranet
     * @throws DataException
     */
    public Intranet getIntranet(String id) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::getIntranet()");
        }
        
        Intranet intranet = null;
        open();
        try {
            intranet = IntranetAccess.getIntranet(query, id);
        }
        finally {
            close();
        }
        return intranet;
    }
    
    /**
     * Rcupre tous les intranets
     * @return intranets La liste des intranets
     * @throws DataException
     */
    public List getIntranets() throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::getIntranets()");
        }
        
        List intranets = null;
        open();
        try {
            intranets = IntranetAccess.getIntranets(query);
        }
        finally {
            close();
        }
        return intranets;
    }
    
    /**
	 * Cre un intranet
	 * @param intranet L'intranet  crer
	 */
    public void createIntranet(Intranet intranet) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::createIntranet()");
        }
        
        session();
        try {
            if(IntranetAccess.intranetExist(query, intranet.getId())) {
                throw new DataException("Un intranet avec cet identifiant existe d\u00E9j\u00E0");
            }
            Folder folder = new Folder();
            folder.setId(String.valueOf(FolderAccess.getNextFolder(query)));
            folder.setDate("99999999");
            folder.setIntranet(intranet.getId());
            folder.setName(intranet.getName());
            folder.setParentid("0");
            FolderAccess.createFolder(query, folder);
            intranet.setFolder(folder.getId());
            OrderAccess.pushIntranet(query);
            IntranetAccess.createIntranet(query, intranet);
        }
        catch(DataException e) {
            rollback();
            close();
            throw e;
        }
    }

    /**
     * Modifie un intranet existant
     * @param intranet L'intranet  modifier
     * @throws DataException
     */
    public void updateIntranet(Intranet intranet) throws DataException {
    	if(log.isDebugEnabled()) {
            log.debug("DataControler::updateIntranet()");
        }
    	
    	session();
    	try {
    		Folder folder = FolderAccess.getFolder(query, intranet.getFolder());
    		folder.setName(intranet.getName());
    		FolderAccess.updateFolder(query, folder);
    		IntranetAccess.updateIntranet(query, intranet);
    		commit();
    	}
    	catch(DataException e) {
    		rollback();
    		throw e;
    	}
    	finally {
    		close();
    	}
    }
    
    /**
	 * Supprime un intranet
	 * @param id L'identifiant de l'intranet  supprimer
	 * @throws DataException
	 */
    public void removeIntranet(String id) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::removeIntranet()");
        }
        session();
        try {
            Intranet intranet = IntranetAccess.getIntranet(query, id);
            Folder racine = FolderAccess.getFolder(query, intranet.getFolder());
            List folders = FolderAccess.getFolderChildren(query, racine.getId());
            for(Iterator i = folders.iterator();i.hasNext();) {
                Folder tmp = (Folder)i.next();
                DocumentAccess.removeDocuments(query, id, tmp.getId());
                GrantAccess.removeGrants(query, tmp.getId());
                FolderAccess.deleteFolder(query, tmp.getId());
            }
            DocumentAccess.removeDocuments(query, id, racine.getId());
            GrantAccess.removeGrants(query, racine.getId());
            FolderAccess.deleteFolder(query, racine.getId());
            PopulationAccess.removeAllIntranetEntities(query, id);
            IntranetAccess.removeIntranet(query, id);
            OrderAccess.pullIntranet(query, intranet.getOrder());
        }
        catch(DataException e) {
            rollback();
            close();
            throw e;
        }
    }

    /*
     ************************************************
     * Oprations relatives aux dossiers
     ************************************************
     */
    
    /**
     * Retourne un dossier
     * @param id L'identifiant du dossier
     * @return folder Le dossier
     * @throws DataException
     */
    public Folder getFolder(String id) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::getFolder()");
        }
        Folder folder = null;
        open();
        try {
            folder = FolderAccess.getFolder(query, id);
        }
        finally {
            close();
        }
        return folder;
    }
    
    /**
     * Retourne la liste des sous-dossiers d'un dossier
     * @param id L'identifiant du dossier pre
     * @return folders La liste des dossiers
     * @throws DataException
     */
    public List getFolders(String id) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::getFolders()");
        }
        List folders = null;
        open();
        try {
            folders = FolderAccess.getFolders(query, id);
        }
        finally {
            close();
        }
        return folders;
    }
    
    /**
     * Retourne la liste des dossiers jusqu' la racine (chemin)
     * @param id L'identifiant du dossier  partir duquel on construit le chemin
     * @return path La liste chane indiquant le chemin jusqu' la racine
     * @throws DataException
     */
    public List getFolderPath(String id) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::getFolders()");
        }
        LinkedList path = new LinkedList();
        open();
        try {
            do {
                Folder tmp = FolderAccess.getFolder(query, id);
                path.addFirst(tmp);
                id = tmp.getParentid();
            }
            while(!id.equals("0"));
        }
        finally {
            close();
        }
        return path;
    }
    
    /**
     * Cre un dossier
     * @param folder Le dossier  crer
     * @throws DataException
     */
    public void createFolder(Folder folder) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::createFolder()");
        }
        session();
        try {
            if(FolderAccess.folderExist(query, folder)) {
                throw new DataException("Un dossier portant ce nom existe d\u00E9j\u00E0");
            }
            folder.setId(FolderAccess.getNextFolder(query));
            if(!folder.getParentid().equals("0")) {
                OrderAccess.pushFolder(query, folder.getParentid());
            }
            FolderAccess.createFolder(query, folder);
            List grants = GrantAccess.getGrants(query, folder.getParentid());
            for(Iterator i = grants.iterator();i.hasNext();) {
                Grant tmp = (Grant)i.next();
                tmp.setExtended(true);
                tmp.setFolderid(folder.getId());
                GrantAccess.addGrant(query, tmp);
            }
            commit();
        }
        catch(DataException e) {
            rollback();
            throw e;
        }
        finally {
            close();
        }
    }
    
    /**
     * Modifie un dossier existant
     * @param folder Le dossier  modifier
     * @throws DataException
     */
    public void updateFolder(Folder folder) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::updateFolder()");
        }
        session();
        try {
            if(FolderAccess.folderExist(query, folder)) {
                throw new DataException("Un dossier portant ce nom existe d\u00E9j\u00E0");
            }
            FolderAccess.updateFolder(query, folder);
            commit();
        }
        catch(DataException e) {
            rollback();
            throw e;
        }
        finally {
            close();
        }
    }
    
    /**
     * Supprime un dossier
     * @param intranet L'identifiant d'intranet
     * @param id L'identifiant du dossier
     * @throws DataException
     */
    public void deleteFolder(String intranet, String id) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::deleteFolder()");
        }
        session();
        try {
            if(!FolderAccess.isFolderEmpty(query, intranet, id)) {
                throw new DataException("Impossible de supprimer un dossier non vide");
            }
            Folder folder = FolderAccess.getFolder(query, id);
            FolderAccess.deleteFolder(query, id);
            if(!folder.getParentid().equals("0")) {
                OrderAccess.pullFolder(query, folder.getParentid(), folder.getOrder());
            }
            GrantAccess.removeGrants(query, id);
            commit();
        }
        catch(DataException e) {
            rollback();
            throw e;
        }
        finally {
            close();
        }
    }
    
    /**
     * Retourne l'arborescence d'un intranet
     * @param intranet L'intranet
     * @return tree L'arborescence
     * @throws DataException
     */
    public FolderTree getFolderTree(Intranet intranet) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::getFolders()");
        }
        FolderTree res = null;
        open();
        try {
	        Folder tmp = null;
	        tmp = FolderAccess.getFolder(query, intranet.getFolder());
	        res = new FolderTree();
	        res.setId(tmp.getId());
	        res.setName(tmp.getName());
	        res.setIntranet(tmp.getIntranet());
	        res.setParentid(tmp.getParentid());
	        res.setDate(tmp.getDate());
	        res.setSubfolder(FolderAccess.getFolderTree(query, res.getId()));
	    }
	    finally {
	        close();
	    }
	    return res;
    }
    
    /*
     ************************************************
     * Oprations relatives aux documents
     ************************************************
     */
    
    /**
     * Retourne un document
     * @param intranet L'identifiant d'intranet
     * @param id L'identifiant du document
     * @return doc Le document
     * @throws DataException
     */
    public Document getDocument(String intranet, String id) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::getDocument()");
        }
        Document doc = null;
        open();
        try {
            doc = DocumentAccess.getDocument(query, intranet, id);
        }
        finally {
            close();
        }
        return doc;
    }
    
    /**
     * Retourne l'identifiant du document li
     * @param intranet L'identifiant d'intranet
     * @param id L'identifiant du document pre
     * @return doc Le document
     * @throws DataException
     */
    public Document getLinkedDocument(String intranet, String id) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::getLinkedDocument()");
        }
        Document doc = null;
        open();
        try {
            String idl = DocumentAccess.getLinkedDocument(query, intranet, id);
            doc = DocumentAccess.getDocument(query, intranet, idl);
        }
        finally {
            close();
        }
        return doc;
    }
    
    /**
     * Retourne la liste des documents d'un dossier
     * @param intranet L'identifiant d'intranet
     * @param id L'identifiant du dossier
     * @return docs La liste des documents
     * @throws DataException
     */
    public List getDocuments(String intranet, String id) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::getDocuments()");
        }
        List docs = null;
        open();
        try {
            docs = DocumentAccess.getDocuments(query, intranet, id);
        }
        finally {
            close();
        }
        return docs;
    }
    
    /**
     * Cre un nouveau document
     * @param doc Le document  crer
     * @throws DataException
     */
    public void createDocument(Document doc) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::createDocument()");
        }
        session();
        try {
            if(DocumentAccess.documentExist(query, doc)) {
                throw new DataException("Un document portant ce nom existe d\u00E9j\u00E0");
            }
            doc.setId(String.valueOf(DocumentAccess.getNextDocument(query, doc.getIntranet())));
            OrderAccess.pushDocument(query, doc.getFolderid());
            DocumentAccess.createDocument(query, doc);
            String id = doc.getFolderid();
            do {
            	FolderAccess.updateFolderDate(query, id, Date.getCurrentDate());
            	Folder tmp = FolderAccess.getFolder(query, id);
                id = tmp.getParentid();
            }
            while(!id.equals("0"));
            
        }
        catch(DataException e) {
            rollback();
            close();
            throw e;
        }
    }
    
    /**
     * Modifie un document existant
     * @param doc Le document  modifier
     * @param pending Si les modifications sont mises en attente
     * @throws DataException
     */
    public void updateDocument(Document doc, boolean pending) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::updateDocument()");
        }
        if(pending) {
            session();
            try {
                DocumentAccess.setPendingModification(query, doc.getIntranet(), doc.getId(), true);
                String newid = DocumentAccess.getNextDocument(query, doc.getIntranet());
                DocumentAccess.setLinkedDocument(query, doc.getIntranet(), doc.getId(), newid);
                doc.setId(newid);
                doc.setValid(false);
                doc.setOrder(0);
                doc.setPendingModification(true);
                OrderAccess.pushDocument(query, doc.getFolderid());
                DocumentAccess.createDocument(query, doc);
            }
            catch(DataException e) {
                rollback();
                close();
                throw e;
            }
        }
        else {
            session();
            try {
                DocumentAccess.updateDocument(query, doc);
                String id = doc.getFolderid();
                do {
                	FolderAccess.updateFolderDate(query, id, Date.getCurrentDate());
                	Folder tmp = FolderAccess.getFolder(query, doc.getFolderid());
                    id = tmp.getParentid();
                }
                while(!id.equals("0"));
            }
            catch(DataException e) {
                rollback();
                close();
                throw e;
            }
        }
    }
    
    /**
     * Supprime un document
     * @param doc Le document
     * @param pending Si la suppression est mise en attente
     * @throws DataException
     */
    public void removeDocument(Document doc, boolean pending) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::removeDocument()");
        }
        if(pending) {
            open();
            try {
                DocumentAccess.setPendingDeletion(query, doc.getIntranet(), doc.getId(), true);
            }
            finally {
                close();
            }
        }
        else {
            session();
            try {
                DocumentAccess.removeDocument(query, doc.getIntranet(), doc.getId());
                OrderAccess.pullDocument(query, doc.getFolderid(), doc.getOrder());
            }
            catch(DataException e) {
                rollback();
                close();
                throw e;
            }
        }
    }
    
    /**
     * Valide / Invalide un document
     * @param doc Le document
     * @param valid La validit
     * @throws DataException
     */
    public void validDocument(Document doc, boolean valid) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::validDocument()");
        }
        if(doc.isPendingDeletion()) {
            if(valid) {
                session();
                try {
                    DocumentAccess.removeDocument(query, doc.getIntranet(), doc.getId());
                    OrderAccess.pullDocument(query, doc.getFolderid(), doc.getOrder());
                }
                catch(DataException e) {
                    rollback();
                    close();
                    throw e;
                }
            }
            else {
                session();
                try {
                    DocumentAccess.setPendingDeletion(query, doc.getIntranet(), doc.getId(), false);
                }
                catch(DataException e) {
                    rollback();
                    close();
                    throw e;
                }
            }
            return;
        }
        if(doc.isPendingModification()) {
            if(valid) {
                session();
                try {
                    String idl = DocumentAccess.getLinkedDocument(query, doc.getIntranet(), doc.getId());
                    DocumentAccess.setValid(query, doc.getIntranet(), idl, true);
                    DocumentAccess.setPendingModification(query, doc.getIntranet(), idl, false);
                    DocumentAccess.removeDocument(query, doc.getIntranet(), doc.getId());
                    OrderAccess.pullDocument(query, doc.getFolderid(), doc.getOrder());
                    String id = doc.getFolderid();
                    do {
                    	FolderAccess.updateFolderDate(query, id, Date.getCurrentDate());
                    	Folder tmp = FolderAccess.getFolder(query, doc.getFolderid());
                        id = tmp.getParentid();
                    }
                    while(!id.equals("0"));
                    commit();
                }
                catch(DataException e) {
                    rollback();
                    close();
                    throw e;
                }
            }
            else {
                session();
                try {
                    String idl = DocumentAccess.getLinkedDocument(query, doc.getIntranet(), doc.getId());
                    Document oldoc = DocumentAccess.getDocument(query, doc.getIntranet(), idl);
                    DocumentAccess.removeDocument(query, doc.getIntranet(), idl);
                    OrderAccess.pullDocument(query, oldoc.getFolderid(), oldoc.getOrder());
                    DocumentAccess.setPendingModification(query, doc.getIntranet(), doc.getId(), false);
                }
                catch(DataException e) {
                    rollback();
                    close();
                    throw e;
                }
            }
            return;
        }
        if(valid) {
            session();
            try {
                DocumentAccess.setValid(query, doc.getIntranet(), doc.getId(), true);
                FolderAccess.updateFolderDate(query, doc.getFolderid(), Date.getCurrentDate());
            }
            catch(DataException e) {
                rollback();
                throw e;
            }
        }
        else {
            session();
            try {
                DocumentAccess.removeDocument(query, doc.getIntranet(), doc.getId());
                OrderAccess.pullDocument(query, doc.getFolderid(), doc.getOrder());
            }
            catch(DataException e) {
                rollback();
                close();
                throw e;
            }
        }
    }
    
    /**
     * Cache / Affiche un document
     * @param intranet L'identifiant d'intranet
     * @param id L'identifiant du document
     * @param visible La visibilit du document
     * @throws DataException
     */
    public void setDocumentVisible(String intranet, String id, boolean visible) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::setDocumentVisible()");
        }
        open();
        try {
            DocumentAccess.setVisible(query, intranet, id, visible);
        }
        finally {
            close();
        }
    }
    
    /*
     ************************************************
     * Oprations relatives aux types MIME
     ************************************************
     */
    
    /**
	 * Rcupre tous les types MIME
	 * @return mimes La liste des types MIME
	 * @throws DataException
	 */
	public List getMIMEs() throws DataException {
	    if(log.isDebugEnabled()) {
	        log.debug("DataControler::getMIMEs()");
	    }
	    List mimes = null;
	    open();
	    try {
	        mimes = MIMEAccess.getMIMEs(query);
	    }
	    finally {
	        close();
	    }
	    return mimes;
	}
	
	/**
	 * Cre un type MIME
	 * @param mime Le type MIME  crer
	 */
	public void createMIME(MIME mime) throws DataException {
	    if(log.isDebugEnabled()) {
	        log.debug("DataControler::createMIME()");
	    }
	    session();
	    try {
	        if(MIMEAccess.mimeExist(query, mime.getId())) {
	            throw new DataException("Cette extension est d\u00E9j\u00E0 utilis\u00E9 par un autre type MIME");
	        }
	        MIMEAccess.createMIME(query, mime);
	        commit();
	    }
	    catch(DataException e) {
	        rollback();
	        throw e;
	    }
	    finally {
	        close();
	    }
	}
	
	/**
	 * Supprime un type MIME
	 * @param extension L'extension associe au type MIME
	 * @throws DataException
	 */
	public void removeMIME(String extension) throws DataException {
	    if(log.isDebugEnabled()) {
	        log.debug("DataControler::removeMIME()");
	    }
	    open();
	    try {
	        MIMEAccess.removeMIME(query, extension);
	    }
	    finally {
	        close();
	    }
	}
	
	/**
	 * Teste si un type MIME existe dj
	 * @param extension L'extension du type  tester
	 * @return boolean True si le type MIME existe, false sinon
	 * @throws DataException
	 */
	public boolean mimeExist(String extension) throws DataException {
	    if(log.isDebugEnabled()) {
	        log.debug("DataControler::mimeExist()");
	    }
	    boolean exist = false;
	    open();
	    try {
	        exist = MIMEAccess.mimeExist(query, extension);
	    }
	    finally {
	        close();
	    }
	    return exist;
	}
    
    /*
     ************************************************
     * Oprations relatives aux populations
     ************************************************
     */
    
    /**
     * Retourne la liste des entits d'un intranet 
     * @param id l'identifiant de l'intranet
     * @return entities La liste des entits
     * @throws DataException
     */
    public List getIntranetEntities(String id) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::getIntranetEntities()");
        }
        List entities = null;
        open();
        try {
            entities = PopulationAccess.getIntranetEntities(query, id);
        }
        finally {
            close();
        }
        return entities;
    }
    
    /**
     * Ajoute une entit  la population d'un intranet
     * @param en l'entit  ajouter
     * @param id l'identifiant de l'intranet
     * @throws DataException
     */
    public void addIntranetEntity(Entity en, String id) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::addIntranetEntity()");
        }
        open();
        try {
            PopulationAccess.addIntranetEntity(query, en, id);
        }
        finally {
            close();
        }
    }
    
    /**
     * Supprime une entit de la population d'un intranet
     * @param en l'entit  supprimer
     * @param id l'identifiant de l'intranet
     * @throws DataException
     */
    public void removeIntranetEntity(Entity en, String id) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::removeIntranetEntity()");
        }
        open();
        try {
            PopulationAccess.removeIntranetEntity(query, en, id);
        }
        finally {
            close();
        }
    }
    
    /*
     ************************************************
     * Oprations relatives aux administrateurs
     ************************************************
     */
    
    /**
     * Retourne la liste des administrateurs
     * @return admins La liste
     * @throws DataException
     */
    public List getAdmins() throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::getAdmins()");
        }
        List admins = null;
        open();
        try {
            admins = AdminAccess.getAdmins(query);
        }
        finally {
            close();
        }
        return admins;
    }
    
    /**
     * Ajoute un administrateur
     * @param id L'identifiant de l'utilisateur
     * @throws DataException
     */
    public void addAdmin(String id) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::addAdmin()");
        }
        open();
        try {
            AdminAccess.addAdmin(query, id);
        }
        finally {
            close();
        }
    }
    
    /**
     * Supprime un administrateur
     * @param id L'identifiant de l'utilisateur
     * @throws DataException
     */
    public void removeAdmin(String id) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::removeAdmin()");
        }
        open();
        try {
            AdminAccess.removeAdmin(query, id);
        }
        finally {
            close();
        }
    }
    
    /*
     ************************************************
     * Oprations relatives aux droits
     ************************************************
     */
    
    /**
     * Retourne la liste des droits d'un dossier
     * @param id L'identifiant du dossier
     * @return grants La liste des droits
     * @throws DataException
     */
    public List getGrants(String id) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::getGrants()");
        }
        List grants = null;
        open();
        try {
            grants = GrantAccess.getGrants(query, id);
        }
        finally {
            close();
        }
        return grants;
    }
    
    /**
     * Ajoute un droit
     * @param grant Le droit  ajouter
     * @throws DataException
     */
    public void addGrant(Grant grant) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::addGrant()");
        }
        session();
        try {
            if(GrantAccess.grantExist(query, grant)) {
                throw new DataException("Un droit existe d\u00E9j\u00E0 sur ce dossier");
            }
            GrantAccess.addGrant(query, grant);
            grant.setExtended(true);
            List folders = FolderAccess.getFolderChildren(query, grant.getFolderid());
            for(Iterator i = folders.iterator();i.hasNext();) {
                Folder tmp = (Folder)i.next();
                grant.setFolderid(tmp.getId());
                GrantAccess.removeGrant(query, grant);
                GrantAccess.addGrant(query, grant);
            }
            commit();
        }
        catch(DataException e) {
            rollback();
            throw e;
        }
        finally {
            close();
        }
    }
    
    /**
     * Supprime un droit
     * @param grant Le droit  supprimer
     * @throws DataException
     */
    public void removeGrant(Grant grant) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::removeGrant()");
        }
        session();
        try {
            GrantAccess.removeGrant(query, grant);
            grant.setExtended(true);
            List folders = FolderAccess.getFolderChildren(query, grant.getFolderid());
            for(Iterator i = folders.iterator();i.hasNext();) {
                Folder tmp = (Folder)i.next();
                grant.setFolderid(tmp.getId());
                GrantAccess.removeGrant(query, grant);
            }
            commit();
        }
        catch(DataException e) {
            rollback();
            throw e;
        }
        finally {
            close();
        }
    }
    
    /**
     * Modifie la notification
     * @param grant Le droit  modifier
     * @throws DataException
     */
    public void setNotification(Grant grant) throws DataException {
        if(log.isDebugEnabled()) {
            log.debug("DataControler::setNotification()");
        }
        session();
        try {
            GrantAccess.updateGrant(query, grant);
            grant.setExtended(true);
            List folders = FolderAccess.getFolderChildren(query, grant.getFolderid());
            for(Iterator i = folders.iterator();i.hasNext();) {
                Folder tmp = (Folder)i.next();
                grant.setFolderid(tmp.getId());
                GrantAccess.updateGrant(query, grant);
            }
            commit();
        }
        catch(DataException e) {
            rollback();
            throw e;
        }
        finally {
            close();
        }
    }
    
    /**
     * Echange le rang de deux documents
     * @param id L'identifiant du dossier
     * @param doc L'identifiant du document
     * @param oldorder L'ancien rang du document
     * @param neworder Le nouveau rang du document
     * @throws DataException
     */
	public void swapDocuments(String id, String doc, int oldorder, int neworder) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::swapDocuments()");
	    }
	    session();
	    try {
	        OrderAccess.swapDocument(query, id, doc, oldorder, neworder);
	        commit();
	    }
	    catch(DataException e) {
	        rollback();
	        throw e;
	    }
	    finally {
	        close();
	    }	    
    }
    
    /**
     * Echange le rang de deux dossiers
     * @param id L'identifiant du dossier parent
     * @param id2 L'identifiant du dossier
     * @param oldorder L'ancien rang du dossier
     * @param neworder Le nouveau rang du dossier
     * @throws DataException
     */
	public void swapFolders(String id, String id2, int oldorder, int neworder) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::swapFolders()");
	    }
	    session();
	    try {
	        OrderAccess.swapFolder(query, id, id2, oldorder, neworder);
	        commit();
	    }
	    catch(DataException e) {
	        rollback();
	        throw e;
	    }
	    finally {
	        close();
	    }	    
    }
    
    /**
     * Echange le rang de deux intranets
     * @param id L'identifiant de l'intranet
     * @param oldorder L'ancien rang de l'intranet
     * @param neworder Le nouveau rang de l'intranet
     * @throws DataException
     */
	public void swapIntranets(String id, int oldorder, int neworder) throws DataException {
	    if(log.isDebugEnabled()) {
            log.debug("DataControler::swapIntranets()");
	    }
	    session();
	    try {
	        OrderAccess.swapIntranet(query, id, oldorder, neworder);
	        commit();
	    }
	    catch(DataException e) {
	        rollback();
	        throw e;
	    }
	    finally {
	        close();
	    }	    
    }
}