package org.esupportail.portal.channels.CIntranet.storage;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.esupportail.portal.channels.CIntranet.beans.Document;
import org.esupportail.portal.channels.CIntranet.beans.Intranet;

/**
 * BalancedFileSystemStorage<br>
 * <br>
 * Implmentation d'un IStorageControler avec un stockage des documents sur<br>
 * un systme de fichier physique local (ventuellement partag)<br>
 * Chaque intranet se voit attribuer un dossier dans lequel des dossiers sont crs<br>
 * afin qu'aucun dossier ne contienne plus d'un nombre dtermin de documents<br>
 * <br>
 * (c)Copyright <a href="http://www.esup-portail.org">ESup-Portail 2004</a><br>
 * @author <a href="mailto:mathieu.larchet@univ-nancy2.fr">Mathieu Larchet</a>
 * @version $Revision: 1.1.2.2 $
 * 
 */

public class BalancedFileSystemStorage implements IStorageControler {

    protected static final Log log = LogFactory.getLog(BalancedFileSystemStorage.class);
	private String root = null;
	private int size = 100;
	
	/**
	 * Constructeur
	 */
	public BalancedFileSystemStorage() {
	}
	
	/**
	 * Sauvegarde d'un document
	 * @param intranet l'identifiant de l'intranet
	 * @param doc le document
	 * @param in le flux de lecture vers le contenu du document
	 * @throws StorageException
	 */
	public synchronized void saveDocument(String intranet, Document doc, InputStream in) throws StorageException {
		if(log.isDebugEnabled()) {
		    log.debug("BalancedFileSystemStorage::saveDocument()");
		}
		// Vrification paramtres
		if(root == null) {
			log.error("BalancedFileSystemStorage::saveDocument() : root = null");
			throw new StorageException("Le r\u00E9pertoire de base n'est pas correctement configur\u00E9");
		}
		
		// Vrification rpertoire
		File folder = new File(root + "/" + intranet + "/" + getFolder(doc.getId()));
		if(!folder.exists()) {
			folder.mkdir();
		}
		
		// Cration du fichier
		File file = new File(root + "/" + intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId());
		
		// Vrification existence du fichier
		if(file.exists()) {
			log.error("BalancedFileSystemStorage::saveDocument() : Le document " + doc.getId() + " existe d\u00E9j\u00E0");
			throw new StorageException("Erreur lors de la sauvegarde du document");
		}
		try {
			// Cration du fichier vide
			if(!file.createNewFile())
			{
				log.error("BalancedFileSystemStorage::saveDocument() : Erreur lors de la cr\u00E9ation du fichier");
				throw new StorageException("Erreur lors de la sauvegarde du document");
			}
		}
		catch(IOException e) {
			log.error("BalancedFileSystemStorage::saveDocument() : Erreur lors de la cr\u00E9ation du fichier");
			throw new StorageException("Erreur lors de la sauvegarde du document");
		}
		
		// Ecriture des donnes
		try {
			OutputStream out = new FileOutputStream(file);
			
			int size = 0;
            byte[] contentBytes = new byte[8192];
            while ((size = in.read(contentBytes)) != -1) {
                out.write(contentBytes,0, size);
            }
			in.close();
			out.flush();
			out.close();
		}
		catch(FileNotFoundException e) {
			log.error("BalancedFileSystemStorage::saveDocument() : Erreur lors de l'\u00E9criture du fichier");
			throw new StorageException("Erreur lors de la sauvegarde du document");
		}
		catch(IOException e) {
			log.error("BalancedFileSystemStorage::saveDocument() : Erreur lors de l'\u00E9criture du fichier");
			throw new StorageException("Erreur lors de la sauvegarde du document");
		}
	}

	/**
	 * Lecture d'un document
	 * @param intranet l'identifiant de l'intranet
	 * @param doc le document
	 * @param os le flux vers lequel on crit le document
	 * @throws StorageException
	 */
	public synchronized void loadDocument(String intranet, Document doc, OutputStream os) throws StorageException {
	    if(log.isDebugEnabled()) {
		    log.debug( "BalancedFileSystemStorage::loadDocument()");
	    }
		// Vrification paramtres
		if(root == null) {
			log.error("BalancedFileSystemStorage::loadDocument() : root = null");
			throw new StorageException("Le r\u00E9pertoire de base n'est pas correctement configur\u00E9");
		}
		
		// Rcupration du fichier
		File loadDoc = new File(root + "/" + intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId());
				
		try {
			BufferedInputStream in = new BufferedInputStream(new FileInputStream(loadDoc));
			BufferedOutputStream out = new BufferedOutputStream(os);
			
			int size = 0;
			byte[] contentBytes = new byte[8192];
            while ((size = in.read(contentBytes)) != -1) {
                out.write(contentBytes,0, size);
            }
			
			in.close();
			out.close();
		}
		catch(FileNotFoundException e) {
			log.error("BalancedFileSystemStorage::loadDocument() : Erreur lors de la lecture du document " + doc.getId());
			throw new StorageException("Erreur lors de la lecture du document");
		}
		catch(IOException e) {
			log.error("BalancedFileSystemStorage::loadDocument() : Erreur lors de la lecture du document " + doc.getId());
			throw new StorageException("Erreur lors de la lecture du document");
		}
	}

	/**
	 * Mise  jour d'un document
	 * @param intranet l'identifiant de l'intranet
	 * @param doc le document
	 * @param in le flux de lecture vers le nouveau contenu, null si pas de mise  jour du contenu
	 * @throws StorageException
	 */
	public synchronized void updateDocument(String intranet, Document doc, InputStream in) throws StorageException {
	    if(log.isDebugEnabled()) {
		    log.debug("BalancedFileSystemStorage::updateDocument()");
	    }
		// Vrification paramtres
		if(root == null) {
			log.error("BalancedFileSystemStorage::updateDocument() : root = null");
			throw new StorageException("Le r\u00E9pertoire de base n'est pas correctement configur\u00E9");
		}
		
		// Document non valide
		if(!doc.isValid()) {
			if(in == null) {
				// Recopie du fichier valide
				File valid = new File(root + "/" + intranet + "/" + getFolder(doc.getId()) + "/v" + doc.getId().substring(1));
				File newfile = new File(root + "/" + intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId());
				try {
					// Cration du fichier vide
					if(!newfile.createNewFile())
					{
						log.error("BalancedFileSystemStorage::updateDocument() : Erreur lors de la cr\u00E9ation du fichier");
						throw new StorageException("Erreur lors de la sauvegarde du document");
					}
					InputStream input = new FileInputStream(valid);
					OutputStream out = new FileOutputStream(newfile);
					int size = 0;
		            byte[] contentBytes = new byte[8192];
		            while((size = input.read(contentBytes)) != -1) {
		                out.write(contentBytes,0, size);
		            }
					input.close();
					out.flush();
					out.close();
					return;
				}
				catch(IOException e) {
					log.error("BalancedFileSystemStorage::updateDocument() : Erreur lors de la cr\u00E9ation du fichier");
					throw new StorageException("Erreur lors de la sauvegarde du document");
				}
			}
			else {
				// Ecriture du nouveau fichier
				File newfile = new File(root + "/" + intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId());
				try {
					// Cration du fichier vide
					if(!newfile.createNewFile())
					{
						log.error("BalancedFileSystemStorage::updateDocument() : Erreur lors de la cr\u00E9ation du fichier");
						throw new StorageException("Erreur lors de la sauvegarde du document");
					}
					OutputStream out = new FileOutputStream(newfile);
					int size = 0;
		            byte[] contentBytes = new byte[8192];
		            while((size = in.read(contentBytes)) != -1) {
		                out.write(contentBytes,0, size);
		            }
					in.close();
					out.flush();
					out.close();
					return;
				}
				catch(IOException e) {
					log.error("BalancedFileSystemStorage::updateDocument() : Erreur lors de la cr\u00E9ation du fichier");
					throw new StorageException("Erreur lors de la sauvegarde du document");
				}
			}
		}
		
		// Document valide
		File valid = new File(root + "/" + intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId());
		valid.delete();
		try {
			// Cration du fichier vide
			if(!valid.createNewFile())
			{
				log.error("BalancedFileSystemStorage::updateDocument() : Erreur lors de la cr\u00E9ation du fichier");
				throw new StorageException("Erreur lors de la sauvegarde du document");
			}
			OutputStream out = new FileOutputStream(valid);
			int size = 0;
            byte[] contentBytes = new byte[8192];
            while((size = in.read(contentBytes)) != -1) {
                out.write(contentBytes,0, size);
            }
			in.close();
			out.flush();
			out.close();
		}
		catch(IOException e) {
			log.error("BalancedFileSystemStorage::updateDocument() : Erreur lors de la cr\u00E9ation du fichier");
			throw new StorageException("Erreur lors de la sauvegarde du document");
		}
	}
	
	/**
	 * Suppression d'un document
	 * @param intranet l'identifiant de l'intranet
	 * @param doc le document
	 * @throws StorageException
	 */
	public synchronized void destroyDocument(String intranet, Document doc) throws StorageException {
	    if(log.isDebugEnabled()) {
		    log.debug("BalancedFileSystemStorage::destroyDocument()");
	    }
		// Vrification paramtres
		if(root == null) {
			log.error("BalancedFileSystemStorage::destroyDocument() : root = null");
			throw new StorageException("Le r\u00E9pertoire de base n'est pas correctement configur\u00E9");
		}
		
		// Rcupration du fichier
		File file = new File(root + "/" + intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId());
				
		// Suppression du fichier
		if(!file.delete()) {
			log.error("BalancedFileSystemStorage::destroyDocument() : Erreur lors de la suppression du fichier " + doc.getId());
			log.fatal("BalancedFileSystemStorage::destroyDocument() : Un document supprim\u00E9 de la base est encore stock\u00E9");
			throw new StorageException("Erreur lors de la suppression du document");
		}
	}

	/**
	 * Cration d'un intranet
	 * @param intranet l'intranet  crer
	 * @throws StorageException
	 */
	public synchronized void createIntranet(Intranet intranet) throws StorageException {
	    if(log.isDebugEnabled()) {
		    log.debug("BalancedFileSystemStorage::createIntranet()");
	    }
		// Vrification paramtres
		if(root == null) {
			log.error("BalancedFileSystemStorage::createIntranet() : root = null");
			throw new StorageException("Le r\u00E9pertoire de base n'est pas correctement configur\u00E9");
		}
		
		// Cration du dossier racine
		File newFold = new File(root + "/" + intranet.getId());
		if(newFold.exists()) {
			log.error("BalancedFileSystemStorage::createIntranet() : Le dossier " + intranet.getId() + " existe d\u00E9j\u00E0");
			throw new StorageException("Erreur lors de la cr\u00E9ation de l'intranet");
		}
		if(!newFold.mkdir()) {
			log.error("BalancedFileSystemStorage::createIntranet() : Erreur lors de la cr\u00E9ation du dossier " + intranet.getId());
			throw new StorageException("Erreur lors de la cr\u00E9ation de l'intranet");
		}
	}

	/**
	 * Suppression d'un intranet et de tous ses documents
	 * @param intranet l'identifiant de l'intranet
	 * @throws StorageException
	 */
	public synchronized void destroyIntranet(String intranet) throws StorageException {
	    if(log.isDebugEnabled()) {
		    log.debug("BalancedFileSystemStorage::destroyIntranet()");
	    }
		if(root == null) {
			log.error("BalancedFileSystemStorage::destroyIntranet() : root = null");
			throw new StorageException("Le rpertoire de base n'est pas correctement configur\u00E9");
		}
		log.fatal("BalancedFileSystemStorage::destroyIntranet() : La suppression de l'intranet " + intranet + " doit \u00EAtre r\u00E9alise manuellement");
	}
	
	
	/**
	 * Calcule le rpertoire o lire / crire un document<br>
	 * en fonction de son identifiant
	 * @param id L'identifiant du document
	 * @return Le chemin o lire / crire le document
	 */
	private String getFolder(String id) {
		int number = new Integer(id).intValue();
		return new Integer(number / size).toString();
	}
	
	/**
	 * Liste des paramtres spcifiques  l'implmentation du IStorageControler<br>
	 * Ici deux paramtres obligatoires :<br>
	 * - size qui contient le nombre maximal de documents par dossier<br>
	 * - path qui indique le chemin o crer les intranets
	 * @param parameters les paramtres
	 */
	public void setParameters(Map parameters) {
	    if(log.isDebugEnabled()) {
		    log.debug("BalancedFileSystemStorage::setParameters()");
	    }
		root = (String)parameters.get("path");
		size = new Integer((String)parameters.get("size")).intValue();
	}
}
