package org.esupportail.portal.channels.CIntranet.storage;

import java.io.IOException;
import java.io.InputStream;

import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpURL;
import org.apache.commons.httpclient.URIException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.webdav.lib.WebdavResource;
import org.esupportail.portal.channels.CIntranet.storage.StorageFactory;

/**
 * DAVConnection<br>
 * <br>
 * Cette classe propose des mthodes de base pour utiliser des services WebDAV<br>
 * <br>
 * (c)Copyright <a href="http://www.esup-portail.org">ESup-Portail 2004</a><br>
 * @author <a href="mailto:mathieu.larchet@univ-nancy2.fr">Mathieu Larchet</a>
 * @version $Revision: 1.1.2.1 $
 * 
 */
public class DAVConnection {
    
    protected static final Log log = LogFactory.getLog(StorageFactory.class);
    
    private String url = null;
    private String uri = null;
    private String login = null;
    private String password = null;
    
    private HttpURL httpUrl = null;
    private WebdavResource res = null;
    
    /**
     * Constructeur
     */
    public DAVConnection() {
    }
    
    /**
     * Ouvre une connexion WebDAV
     * @param url l'URL de connexion
     * @param login l'identifiant
     * @param password le mot de passe
     * @throws DAVException
     */
    public void connect(String url, String login, String password) throws DAVException {
        if(log.isDebugEnabled()) {
            log.debug("DAVConnection::connect()");
        }
        
        this.url = url;
        this.login = login;
        this.password = password;
        
        try {
            httpUrl = new HttpURL(this.url);
            httpUrl.setUserinfo(this.login, this.password);
            uri = httpUrl.getCurrentHierPath() + "/";
        }
        catch(URIException e) {
            log.error("DAVConnection::connect() : Mauvaise URL " + this.url + this.uri);
            throw new DAVException("Mauvaise URL : " + this.url);
        }
        
        try {
            res = new WebdavResource(httpUrl);
        }
        catch(HttpException e) {
            log.error("DAVConnection::connect() : Erreur de connexion \u00E0 " + this.url + " erreur " + e.getReasonCode());
            throw new DAVException("Erreur de connexion \u00E0 " + this.url + " erreur " + e.getReasonCode());
        }
        catch(IOException e) {
            log.error("DAVConnection::connect() : Erreur de connexion \u00E0 " + this.url + " erreur " + e.getMessage());
            throw new DAVException("Erreur de connexion \u00E0 " + this.url + " erreur " + e.getMessage());
        }
    }
    
    /**
     * Ferme la connexion
     */
    public void close() {
        if(log.isDebugEnabled()) {
            log.debug("DAVConnection::close()");
        }
        if(res != null) {
            try {	
                res.close();
            }
            catch(IOException e) {
                log.error("DAVConnection::close() : Erreur de d\u00E9connexion " + e.getMessage());
            }
        }
    }

    /**
     * Teste l'existence d'un rpertoire
     * @param path le chemin relatif  tester
     * @return exist true | false
     */
    public boolean dirExists(String path) {
        if(log.isDebugEnabled()) {
            log.debug("DAVConnection::dirExists()");
        }
        
        boolean result = false;
        try {
            res.setPath(uri + path);
            result = true;
        }
        catch(HttpException e) {
            // Le rpertoire n'existe pas
        }
        catch(IOException e) {
            log.error("DAVConnection::dirExists() : " + e.getMessage());
        }
        return result;
    }
    
    /**
     * Teste l'existence d'un fichier
     * @param file le chemin du fichier  tester
     * @return exist true | false
     */
    public boolean fileExists(String file) {
        if(log.isDebugEnabled()) {
            log.debug("DAVConnection::fileExists()");
        }
        
        return dirExists(file);
    }
    
    /**
     * Cre un rpertoire
     * @param path le chemin du rpertoire  crer
     * @throws DAVException
     */
    public void createDir(String path) throws DAVException {
        if(log.isDebugEnabled()) {
            log.debug("DAVConnection::createDir()");
        }
        
        try {
            res.mkcolMethod(uri + path);
            int httpCode = res.getStatusCode();
            
            if(httpCode != 201) {
                log.error("DAVConnection::createDir() : Erreur lors de la cr\u00E9ation du r\u00E9pertoire " + path + " erreur " + httpCode);
                throw new DAVException("Erreur lors de la cr\u00E9ation du r\u00E9pertoire");
            }
        }
        catch(HttpException e) {
            log.error("DAVConnection::createDir() : Erreur lors de la cr\u00E9ation du r\u00E9pertoire " + path + " erreur " + e.getReasonCode());
            throw new DAVException("Erreur lors de la cr\u00E9ation du r\u00E9pertoire");
        }
        catch(IOException e) {
            log.error("DAVConnection::createDir() : " + e.getMessage());
            throw new DAVException("Erreur lors de la cr\u00E9ation du r\u00E9pertoire");
        }
    }
    
    /**
     * Cre un fichier
     * @param file le chemin du fichier  crer
     * @param in les donnes du fichier
     * @throws DAVException
     */
    public void createFile(String file, InputStream in) throws DAVException {
        if(log.isDebugEnabled()) {
            log.debug("DAVConnection::createFile()");
        }
        
        try {
            res.putMethod(uri + file, in);
            in.close();
            int httpCode = res.getStatusCode();
            if(httpCode != 201) {
                log.error("DAVConnection::createFile() : Erreur lors de la cr\u00E9ation du fichier " + file + " erreur " + httpCode);
                throw new DAVException("Erreur lors de la cr\u00E9ation du fichier");
            }
        }
        catch(HttpException e) {
            log.error("DAVConnection::createFile() : Erreur lors de la cr\u00E9ation du fichier " + file + " erreur " + e.getReasonCode());
            throw new DAVException("Erreur lors de la cr\u00E9ation du r\u00E9pertoire");
        }
        catch(IOException e) {
            log.error("DAVConnection::createFile() : " + e.getMessage());
            throw new DAVException("Erreur lors de la cr\u00E9ation du fichier");
        }
    }
    
    /**
     * Renomme un fichier
     * @param source l'ancien chemin du fichier
     * @param dest le nouveau chemin du fichier
     * @throws DAVException
     */
    public void renameFile(String source, String dest) throws DAVException {
        if(log.isDebugEnabled()) {
            log.debug("DAVConnection::renameFile()");
        }
        
        try {
            res.setPath(uri + source);
            res.moveMethod(uri + dest);
            int httpCode = res.getStatusCode();
            if(httpCode != 201) {
                log.error("DAVConnection::renameFile() : Erreur lors du renommage du fichier " + source + " erreur " + httpCode);
                throw new DAVException("Erreur lors du renommage du fichier");
            }
        }
        catch(HttpException e) {
            log.error("DAVConnection::renameFile() : Erreur lors du renommage du fichier " + source + " erreur " + e.getReasonCode());
            throw new DAVException("Erreur lors du renommage du fichier");
        }
        catch(IOException e) {
            log.error("DAVConnection::renameFile() : " + e.getMessage());
            throw new DAVException("Erreur lors du renommage du fichier");
        }
    }
    
    /**
     * Supprime un fichier
     * @param file le chemin du fichier  supprimer
     * @throws DAVException
     */
    public void deleteFile(String file) throws DAVException {
        if(log.isDebugEnabled()) {
            log.debug("DAVConnection::deleteFile()");
        }
        
        try {
            res.deleteMethod(uri + file);
            int httpCode = res.getStatusCode();
            if(httpCode != 204) {
                log.error("DAVConnection::deleteFile() : Erreur lors de la suppression du fichier " + file + " erreur " + httpCode);
                throw new DAVException("Erreur lors de la suppression du fichier");
            }
        }
        catch(HttpException e) {
            log.error("DAVConnection::deleteFile() : Erreur lors de la suppression du fichier " + file + " erreur " + e.getReasonCode());
            throw new DAVException("Erreur lors de la suppression du fichier");
        }
        catch(IOException e) {
            log.error("DAVConnection::deleteFile() : " + e.getMessage());
            throw new DAVException("Erreur lors de la suppression du fichier");
        }
    }
    
    /**
     * Lit un fichier
     * @param file le chemin du fichier  lire
     * @return data les donnes du fichier
     * @throws DAVException
     */
    public InputStream getFile(String file) throws DAVException {
        if(log.isDebugEnabled()) {
            log.debug("DAVConnection::getFile()");
        }
        InputStream result = null;
        
        try {
            result = res.getMethodData(uri + file);
            int httpCode = res.getStatusCode();
            if(httpCode != 200) {
                log.error("DAVConnection::getFile() : Erreur lors de la lecture du fichier " + file + " erreur " + httpCode);
                throw new DAVException("Erreur lors de la lecture du fichier");
            }
        }
        catch(HttpException e) {
            log.error("DAVConnection::getFile() : Erreur lors de la lecture du fichier " + file + " erreur " + e.getReasonCode());
            throw new DAVException("Erreur lors de la lecture du fichier");
        }
        catch(IOException e) {
            log.error("DAVConnection::getFile() : " + e.getMessage());
            throw new DAVException("Erreur lors de la lecture du fichier");
        }
        return result;
    }
}
