package org.esupportail.portal.channels.CIntranet.storage;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.esupportail.portal.channels.CIntranet.beans.Document;
import org.esupportail.portal.channels.CIntranet.beans.Intranet;

/**
 * DAVStorage<br>
 * <br>
 * Implmentation d'un IStorageControler avec un stockage des documents sur<br>
 * un serveur WebDAV<br>
 * Chaque intranet se voit attribuer un dossier dans lequel des dossiers sont crs<br>
 * afin qu'aucun dossier ne contienne plus d'un nombre dtermin de documents<br>
 * <br>
 * (c)Copyright <a href="http://www.esup-portail.org">ESup-Portail 2004</a><br>
 * @author <a href="mailto:mathieu.larchet@univ-nancy2.fr">Mathieu Larchet</a>
 * @version $Revision: 1.1.2.2 $
 * 
 */
public class DAVStorage implements IStorageControler {

    protected static final Log log = LogFactory.getLog(DAVStorage.class);
    private String url = null;
    private String login = null;
    private String pass = null;
    private int size = 100;
    
    /**
	 * Constructeur
	 */
	public DAVStorage() {
	}
    
	/**
	 * Sauvegarde d'un document
	 * @param intranet l'identifiant de l'intranet
	 * @param doc le document
	 * @param in le flux de lecture vers le contenu du document
	 * @throws StorageException
	 */
    public synchronized void saveDocument(String intranet, Document doc, InputStream in) throws StorageException {
        if(log.isDebugEnabled()) {
		    log.debug("DAVStorage::saveDocument()");
        }
		// Vrification paramtres
		if(url == null) {
			log.error("DAVStorage::saveDocument() : url = null");
			throw new StorageException("L'URL n'est pas correctement configur\u00E9e");
		}
		
		// Cration ressource WebDAV
		DAVConnection con = new DAVConnection();
		
		try {
		    con.connect(url, login, pass);
		    
		    // Vrification rpertoire
		    String path = intranet + "/" + getFolder(doc.getId()) + "/";
		    
		    // Cration rpertoire
		    if(!con.dirExists(path)) {
		        con.createDir(path);
		    }
		    
		    // Cration du fichier
			String file = path + doc.getId();
		    con.createFile(file, in);
		    con.close();
		}
		catch(DAVException e) {
		    log.error("DAVStorage::saveDocument() : " + e.getMessage());
		    con.close();
			throw new StorageException("Erreur lors de la sauvegarde du document");
		}
    }

    /**
	 * Lecture d'un document
	 * @param intranet l'identifiant de l'intranet
	 * @param doc le document
	 * @param os le flux vers lequel on crit le document
	 * @throws StorageException
	 */
    public synchronized void loadDocument(String intranet, Document doc, OutputStream os) throws StorageException {
        if(log.isDebugEnabled()) {
		    log.debug("DAVStorage::loadDocument()");
        }
		// Vrification paramtres
		if(url == null) {
		    log.error("DAVStorage::loadDocument() : url = null");
			throw new StorageException("L'URL n'est pas correctement configur\u00E9e");
		}
		
		// Cration ressource WebDAV
		DAVConnection con = new DAVConnection();
		InputStream result = null;
		
		try {
		    con.connect(url, login, pass);
		    
		    // Rcupration du fichier
			String file = intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId();
			result = con.getFile(file);
			
			// Ecriture du flux
			BufferedInputStream in = new BufferedInputStream(result);
			BufferedOutputStream out = new BufferedOutputStream(os);
			
			int size = 0;
			byte[] contentBytes = new byte[8192];
            while ((size = in.read(contentBytes)) != -1) {
                out.write(contentBytes,0, size);
            }
			
			in.close();
			out.close();
			
			con.close();
		}
		catch(DAVException e) {
		    log.error("DAVStorage::loadDocument() : " + e.getMessage());
		    con.close();
			throw new StorageException("Erreur lors de la lecture du document");
		}
		catch(IOException e) {
			log.error("BasicFileSystemStorage::loadDocument() : Erreur lors de la lecture du document " + doc.getId());
			con.close();
			throw new StorageException("Erreur lors de la lecture du document");
		}
    }

    /**
	 * Mise  jour d'un document
	 * @param intranet l'identifiant de l'intranet
	 * @param doc le document
	 * @param in le flux de lecture vers le nouveau contenu, null si pas de mise  jour du contenu
	 * @throws StorageException
	 */
    public synchronized void updateDocument(String intranet, Document doc, InputStream in) throws StorageException {
        if(log.isDebugEnabled()) {
		    log.debug("DAVStorage::updateDocument()");
        }
		// Vrification paramtres
		if(url == null) {
		    log.error("DAVStorage::updateDocument() : url = null");
			throw new StorageException("L'URL n'est pas correctement configur\u00E9e");
		}
		
		// Cration ressource WebDAV
		DAVConnection con = new DAVConnection();
		
		try {
		    con.connect(url, login, pass);
		    if(!doc.isValid()) {
		        if(in == null) {
		            String oldfile = intranet + "/" + getFolder(doc.getId()) + "/v" + doc.getId().substring(1);
			        String newfile = intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId();
			        con.renameFile(oldfile, newfile);
		        }
		        else {
		            String file = intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId();
			        con.createFile(file, in);
		        }
		    }
		    else {
		        String file = intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId();
		        con.deleteFile(file);
		        con.createFile(file, in);
		    }
		    con.close();
		}
		catch(DAVException e) {
		    log.error("DAVStorage::updateDocument() : " + e.getMessage());
		    con.close();
			throw new StorageException("Erreur lors de la mise \u00E0 jour du document");
		}
    }

    /**
	 * Suppression d'un document
	 * @param intranet l'identifiant de l'intranet
	 * @param doc le document
	 * @throws StorageException
	 */
    public synchronized void destroyDocument(String intranet, Document doc) throws StorageException {
        if(log.isDebugEnabled()) {
		    log.debug("DAVStorage::destroyDocument()");
        }
		// Vrification paramtres
		if(url == null) {
		    log.error("DAVStorage::destroyDocument() : url = null");
			throw new StorageException("L'URL n'est pas correctement configur\u00E9e");
		}
		
		// Cration ressource WebDAV
		DAVConnection con = new DAVConnection();
		
		try {
		    con.connect(url, login, pass);
		    String file = intranet + "/" + getFolder(doc.getId()) + "/" + doc.getId();
		    con.deleteFile(file);
		    con.close();
		}
		catch(DAVException e) {
		    log.error("DAVStorage::destroyDocument() : " + e.getMessage());
		    con.close();
			throw new StorageException("Erreur lors de la suppression du document");
		}
    }

    /**
	 * Cration d'un intranet
	 * @param intranet l'intranet  crer
	 * @throws StorageException
	 */
    public synchronized void createIntranet(Intranet intranet) throws StorageException {
        if(log.isDebugEnabled()) {
		    log.debug("DAVStorage::createIntranet()");
        }
		// Vrification paramtres
		if(url == null) {
		    log.error("DAVStorage::createIntranet() : url = null");
			throw new StorageException("L'URL n'est pas correctement configur\u00E9e");
		}
		
		// Cration ressource WebDAV
		DAVConnection con = new DAVConnection();
		
		try {
		    con.connect(url, login, pass);
		    String path = intranet.getId() + "/";
		    if(con.dirExists(path)) {
		        con.close();
		        log.error("DAVStorage::createIntranet() : Le dossier " + intranet.getId() + " existe d\u00E9j\u00E0");
		        throw new StorageException("Le dossier " + intranet.getId() + " existe d\u00E9j\u00E0");
		    }
		    con.createDir(path);
		    con.close();
		}
		catch(DAVException e) {
		    log.error("DAVStorage::createIntranet() : " + e.getMessage());
		    con.close();
			throw new StorageException("Erreur lors de la cr\u00E9ation de l'intranet");
		}
    }

    /**
	 * Suppression d'un intranet et de tous ses documents
	 * @param intranet l'identifiant de l'intranet
	 * @throws StorageException
	 */
    public synchronized void destroyIntranet(String intranet) throws StorageException {
        if(log.isDebugEnabled()) {
		    log.debug("DAVStorage::destroyIntranet()");
        }
		if(url == null) {
		    log.error("DAVStorage::destroyIntranet() : url = null");
			throw new StorageException("L'URL n'est pas correctement configur\u00E9e");
		}
		log.fatal("DAVStorage::destroyIntranet() : La suppression de l'intranet " + intranet + " doit \u00EAtre r\u00E9alis\u00E9e manuellement");
    }

    /**
	 * Calcule le rpertoire o lire / crire un document<br>
	 * en fonction de son identifiant
	 * @param id L'identifiant du document
	 * @return Le chemin o lire / crire le document
	 */
	private String getFolder(String id) {
		int number = new Integer(id).intValue();
		return new Integer(number / size).toString();
	}
    
    /**
	 * Liste des paramtres spcifiques  l'implmentation du IStorageControler<br>
	 * Ici cinq paramtres obligatoires :<br>
	 * - url qui contient l'adresse du serveur WebDAV et le chemin de stockage
	 * - login qui contient le nom d'utilisateur
	 * - password qui contient le mot de passe
	 * - size qui contient le nombre maximal de documents par dossier<br>
	 * @param parameters les paramtres
	 */
    public void setParameters(Map parameters) {
        if(log.isDebugEnabled()) {
		    log.debug("DAVStorage::setParameters()");
        }
		url = (String)parameters.get("url");
		login = (String)parameters.get("login");
		pass = (String)parameters.get("password");
		size = new Integer((String)parameters.get("size")).intValue();
		
		if(!url.endsWith("/")) {
		    url = url + "/";
		}
    }
}
