/**
 * ESUP-Portail Practice - Copyright (c) 2004 ESUP-Portail consortium 
 * 
 * This program is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software Foundation.
 * 
 * See the license terms site for more details: http://www.esup-portail.org/license
 * 
 * @author Alexandre Boisseau <thetyrex at users.sourceforge.net>
 * @author Pascal Aubry  <pascal.aubry at univ-rennes1.fr>
*/

package org.esupportail.portal.channels.CPracticeChannel;

import java.io.UnsupportedEncodingException;

import org.esupportail.portal.channels.util.format.EsupXMLEscaper;

/**
 * A file, part of a practice.
 * Data-object, mapped to the file table of the database.
 */
public final class CFile {
	
	/**
	 * The unique id (primary key) of the file
	 */
	private long id;
	
	/**
	 * @return the unique id of the file in the database.
	 */
	public long getId() {
		return id;
	}
	
	/**
	 * The name of the file, as given by the client when. 
	 */
	private String filename;
	
	/**
	 * @return the name of the file, as given by the client when uploading the file.
	 * @uml.property name="filename"
	 */
	public String getFilename() {
		return filename;
	}
	
	/** 
	 * The content of the file.
	 */
	private byte[] content;

	/**
	 * @return The content of the file
	 */
	public byte[] getContent() {
		return content;
	}
	

	/** 
	 * The id of the practice the file was uploaded with.
	 */
	private long cPracticeId;
	/**
	 * @param practiceId The cPracticeId to set.
	 */
	public void setCPracticeId(final long practiceId) {
		cPracticeId = practiceId;
	}
	/**
	 * @return The cPracticeId.
	 */
	public long getCPracticeId() {
		return cPracticeId;
	}
	
	/** 
	 * Default constructor, needed by JDO. 
	 */
	protected CFile() {	
		// needed by JDO
	}
	
	/**
	 * Construct a CFile.
	 * @param cPracticeId the id of the practice the file was uploaded with.
	 * @param filename the name of the file.
	 * @param content the content of the file.
	 */
	protected CFile(final long cPracticeId, final String filename, final byte[] content) {
		this();
		this.content = content;
		this.cPracticeId = cPracticeId;
		this.filename = filename;
	}

	/**
	 * Convert a CFile object into a string.
	 * 
	 * @return a string
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		return toXML(false);
	}

	/**
	 * Convert a CFile object into an XML string.
	 * 
	 * @param content true to output the content of the file, false otherwise
	 * @return an XML string.
	 */
	public String toXML(final boolean content) {

		String xmlContent = "";	
		
		if (content) {
			try {
				xmlContent = new String(getContent(), "ISO-8859-1");
			} catch (UnsupportedEncodingException e) {
				xmlContent = "";
			}
		}

		return "\n<file>"
			+ "\n<file-id>" + id + "</file-id>"
			+ "\n<practice-id>" + cPracticeId + "</practice-id>"
			+ "\n<filename>" + EsupXMLEscaper.escape(filename) + "</filename>"
			+ "\n<content>" + EsupXMLEscaper.escape(xmlContent) + "</content>"
			+ "\n</file>";
	}
}