/**
 * ESUP-Portail Practice - Copyright (c) 2004 ESUP-Portail consortium 
 * 
 * This program is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software Foundation.
 * 
 * See the license terms site for more details: http://www.esup-portail.org/license
 * 
 * @author Alexandre Boisseau <thetyrex at users.sourceforge.net>
 * @author Pascal Aubry  <pascal.aubry at univ-rennes1.fr>
*/

package org.esupportail.portal.channels.CPracticeChannel;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.sql.Timestamp;
import java.util.Date;
import java.util.Iterator;
import java.util.Vector;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import org.esupportail.portal.channels.util.format.TimestampFormater;
import org.esupportail.portal.channels.util.format.EsupXMLEscaper;

/**
 * A practice, as sent by a student to a teacher.
 * Data-object mapped to the practice table of the database.
 */
public final class CPractice {

	/**
	 * The unique id (primary key) of the practice.
	 * 
	 * @uml.property name="id"
	 */
	private long id;
	/**
	 * @return the unique id of the practice.
	 * 
	 * @uml.property name="id"
	 */
	long getId() {
		return id;
	}

	/**
	 * The date when the practice was stored (set on creation).
	 * 
	 * @uml.property name="date"
	 */
	private Timestamp date;

	/**
	 * @return the date when the practice was stored.
	 * 
	 * @uml.property name="date"
	 */
	public Timestamp getDate() {
		return date;
	}

	/**
	 * Small text or message writen by the sender when sending the practice.
	 * 
	 * @uml.property name="description"
	 */
	private String description;

	/**
	 * @return the description of the practice.
	 * 
	 * @uml.property name="description"
	 */
	public String getDescription() {
		return description;
	}

	//	/**
	//	 * @param description the description of the practice.
	//	 * 
	//	 * @uml.property name="description"
	//	 */
	//	public void setDescription(final String description) {
	//		this.description = description;
	//	}

	/**
	 * The user that sent the practice. This field is not stored into database (cf senderId)
	 * 
	 * @uml.property name="sender"
	 * @uml.associationEnd multiplicity="(1 1)" inverse="sent:CPracticeChannel.CUser"
	 */
	private CUser sender;


	/**
	 * @return the user that sent the practice.
	 * 
	 * @uml.property name="sender"
	 */
	public CUser getSender() {
		return sender;
	}

	/**
	 * The sender's id (this field only is stored into the database).
	 */
	private String senderId;

	/**
	 * Retreive the sender's id.
	 * 
	 * @return a String instance, or null if not set.
	 */
	private String getSenderId() {
		return senderId;
	}

	/**
	 * The user the practice was sent to. This field is not stored into database (cf receiverId)
	 * 
	 * @uml.property name="receiver"
	 * @uml.associationEnd multiplicity="(0 1)" inverse="received:CPracticeChannel.CUser"
	 */
	private CUser receiver;

	
	/**
	 * @return the user the practice was sent to.
	 * 
	 * @uml.property name="receiver"
	 */
	public CUser getReceiver() {
		return receiver;
	}

	/**
	 * The receiver's id (this field only is stored into the database).
	 */
	private String receiverId;

	/**
	 * Retreive the receiver's id.
	 * 
	 * @return a String instance, or null if not set.
	 */
	private String getReceiverId() {
		return receiverId;
	}

	/**
	 * The list of the files that were uploading with the practice.
	 */
	private Vector files;

	/**
	 * @return Returns the list of the files that were uploaded with the practice.
	 */
	public Vector getFiles() {
		return files;
	}
	/**
	 * @param files The list of the files uploaded with the practice.
	 */
	public void setFiles(final Vector files) {
		this.files = files;
	}
	
	/**
	 * Default constructor.
	 */
	CPractice() {
		this.date = new java.sql.Timestamp(new Date().getTime());
	}
	
	/**
	 * Constructor.
	 * @param sender the user that sent the practice
	 * @param receiver the user that received the practice
	 * @param description the description of the practice
	 */
	CPractice(final CUser sender, final CUser receiver, final String description) {
		this();
		this.description = description;
		this.sender = sender;
		this.senderId = sender.getId();
		this.receiver = receiver;
		this.receiverId = receiver.getId();
	}
	
	/**
	 * Constructor.
	 * @param sender the user that sent the practice
	 * @param receiver the user that received the practice
	 * @param description the description of the practice
	 * @param files the list of the files uploaded with the practice.
	 */
	CPractice(final CUser sender, final CUser receiver, 
  		  final String description, final Vector files) {
		this(sender, receiver, description);
		this.files = files;
	}
	
	/**
	 * Verify if a user can do things on a practice (practically, he can if 
	 * he sent or received the practice).
	 * 
	 * @param userId the user's id
	 * @return true if the user is granted, false otherwise.
	 */
	boolean isUserGranted(final String userId) {
		if (userId.equals(getSenderId())) {
			return true;
		}
		if (userId.equals(getReceiverId())) {
			return true;
		}
		return false;
	}

	/**
	 * Get the prefix to use when zipping data.
	 * 
	 * @return a String.
	 */
	String getZipPrefix() {
		return getSenderId() + "-" + TimestampFormater.shortFormat(getDate()) + "-" + getId();
	}

	/**
	 * Add itself to an opened ZIP stream.
	 * 
	 * @param out the output zip stream
	 * @throws IOException IOException.
	 */
	void addToZipStream(final ZipOutputStream out) throws IOException {
	    byte[] buf = new byte[1024];
		
	    for (int i = 0; i < getFiles().size(); i++) {

	    	// get file #i
	    	CFile file = (CFile) getFiles().get(i);

	    	// get its content
	    	InputStream in = (InputStream) new ByteArrayInputStream(file.getContent());

	    	// Add a ZIP entry to output stream.
	    	out.putNextEntry(new ZipEntry(getZipPrefix() + "\\" + file.getFilename()));	    

	    	// Transfer bytes from the file to the ZIP stream
	    	int len;
	    	while ((len = in.read(buf)) > 0) {
	    		out.write(buf, 0, len);
	    	}	    

	    	// Complete the entry
	    	out.closeEntry();
	    	in.close();
	    }
	}	

	/** 
	 * Convert a CPractice object into a String
	 * 
	 * @return a string
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		return toXML(false, -1);
	}

	/** 
	 * Convert a CPractice object into an XML String
	 * 
	 * @param truncateDescription true to truncate the description (to the length 
	 *        specified by descriptionLength), false otherwise
	 * @param descriptionLength the length the description should be truncated to, used
	 *        when truncateDescription is true only.
	 * @return an XML string.
	 */
	String toXML(final boolean truncateDescription, final int descriptionLength) {

		String truncatedDescription;
		if (getDescription() == null) {
			truncatedDescription = "";
		} else if (truncateDescription && getDescription().length() >= descriptionLength) {
			truncatedDescription = getDescription().substring(0, descriptionLength);
		} else {
			truncatedDescription = getDescription();
		}

		String xml = "\n<practice>"
			+ "\n<id>" + getId() + "</id>"
			+ "\n<date>" + TimestampFormater.longFormat(getDate()) + "</date>"
			+ "\n<description>" + EsupXMLEscaper.escape(truncatedDescription) + "</description>"
			+ "\n<sender>" + getSender().toXML() + "\n</sender>"
			+ "\n<receiver>" + getReceiver().toXML() + "\n</receiver>"
			+ "\n<files>";
		
		if (!getFiles().isEmpty()) {
			Iterator iterFiles = getFiles().iterator();
			CFile file = null;
			while (iterFiles.hasNext()) {
				file = (CFile) iterFiles.next();
				xml = xml + file.toXML(false);			
			}
		}
		
		xml = xml + "\n</files>"
		+ "\n</practice>";
		
		return xml;	
	}
}