/**
 * ESUP-Portail Practice - Copyright (c) 2004 ESUP-Portail consortium 
 * 
 * This program is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software Foundation.
 * 
 * See the license terms site for more details: http://www.esup-portail.org/license
 * 
 * @author Alexandre Boisseau <thetyrex at users.sourceforge.net>
 * @author Pascal Aubry  <pascal.aubry at univ-rennes1.fr>
*/

// TODO Revoir les logs : les pramtrages de logs uPortal sont crass par les paramtrages de logs OJB
// ( voir aprs la runion ESUP du 28 mai)

// TODO Plantage lorsqu'on essaye plusieurs download total de suite.

// TODO faire les fichier UPGRADE, INSTALL, ...

package org.esupportail.portal.channels.CPracticeChannel;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;
import java.util.zip.ZipOutputStream;

import javax.jdo.Query;

import org.apache.commons.io.IOUtils;
import org.apache.commons.io.output.ByteArrayOutputStream;
import org.apache.ojb.jdori.sql.OjbStoreFatalInternalException;
import org.esupportail.portal.channels.CJDOChannel;
import org.esupportail.portal.channels.CPracticeChannel.util.sort.CPracticeDateComparer;
import org.esupportail.portal.channels.CPracticeChannel.util.sort.CPracticeReceiverComparer;
import org.esupportail.portal.channels.CPracticeChannel.util.sort.CPracticeSenderComparer;
import org.esupportail.portal.channels.util.format.EsupXMLEscaper;
import org.esupportail.portal.channels.util.format.TimestampFormater;
import org.jasig.portal.MultipartDataSource;
import org.jasig.portal.PortalException;
import org.jasig.portal.security.provider.RestrictedPerson;
import org.jasig.portal.utils.XSLT;

/**
 * The main class of the packages, implementing uPortal interfaces.
 */
public final class CPracticeChannel extends CJDOChannel {

	/**
	 * The current user (that instanciated the channel). Initialized by 
	 * setStaticData() on a per-session basis.
	 * 
	 * @uml.property name="currentUser"
	 * @uml.associationEnd multiplicity="(0 1)"
	 */
	private CUser currentUser = null;

	/**
	 * set the current user.
	 * 
	 * @param user the current user
	 * 
	 * @uml.property name="currentUser"
	 */
	private void setCurrentUser(final CUser user) {
		this.currentUser = user;
	}

	/**
	 * @return Returns the currentUser.
	 * 
	 * @uml.property name="currentUser"
	 */
	private CUser getCurrentUser() {
		return currentUser;
	}


	// integers to store the internal state of the application.
	/** Constant value for appState */
	private static final int FATAL_ERROR_STATE = 1;
	/** Constant value for appState */
	private static final int DISPLAY_PRACTICES_STATE = 2;
	/** Constant value for appState */
	private static final int DISPLAY_FILE_STATE = 3;
	/** Constant value for appState */ 
	private static final int DELETE_PRACTICE_STATE = 4;
	/** Constant value for appState */
	private static final int ENTER_TEACHER_STATE = 51;
	/** Constant value for appState */	
	private static final int ENTER_DESCRIPTION_STATE = 52;
	/** Constant value for appState */
	private static final int ENTER_FILES_STATE = 53;
	/** Constant value for appState */
	private static final int DOWNLOAD_FILE_STATE = 500;
	/** Constant value for appState */
	private static final int DOWNLOAD_PRACTICE_STATE = 501;
	/** Constant value for appState */
	private static final int DOWNLOAD_ALL_STATE = 502;
	

	/**
	 * Retreive the default state of the application.
	 * 
	 * @return an integer.
	 */
	protected int getDefaultAppState() {
		return DISPLAY_PRACTICES_STATE;
	}

	/**
	 * The configuration of the application.
	 * 
	 * @uml.property name="config"
	 * @uml.associationEnd multiplicity="(0 1)"
	 */
	private CPracticeChannelConfig config;

	/**
	 * @return Returns the config.
	 * 
	 * @uml.property name="config"
	 */
	private CPracticeChannelConfig getConfig() {
		return config;
	}


	/**
	 * Read a file and set the internal configuration.
	 * 
	 * @throws PortalException PortalException  
	 */
	protected void readConfig() throws PortalException {
		try {
			config = CPracticeChannelConfig.getInstance();
		} catch (Exception ex) {
			throw new PortalException("Error while loading configuration: " + ex);
		}
		
	}

	/**
	 * Constructor.
	 * 
	 * @throws PortalException PortalException  
	 */
	public CPracticeChannel() throws PortalException {
		super();

		// intitialize the session parameters
		sessionParams = new CSessionParams();
	}
	
	/**
	 * Set specific static (per session) data.
	 * 
	 * @throws PortalException PortalException.
	 * @see org.jasig.portal.IChannel#setStaticData(org.jasig.portal.ChannelStaticData)
	 */
	protected void setSpecificStaticData() throws PortalException {
		setCurrentUser(getCUser(getCurrentUserId()));
		if (getCurrentUser() == null) {
			reportLog("Error when get the current user : User \"" + getCurrentUserId() + "\" not found by uPortal");
			throw new PortalException("User \"" + getCurrentUserId() + "\" not found by uPortal");
		}
		
	}

	/**
	 * change the state of the application depending on the previous state and 
	 * the action done by the client.
	 * 
	 * @throws PortalException PortalException.
	 */
	protected void transition() throws PortalException {
		
		switch (getAppState()) {
			case DISPLAY_PRACTICES_STATE :
				transitionDisplayPractices();
				break;
			case DISPLAY_FILE_STATE :
				transitionDisplayFile();
				break;
			case DELETE_PRACTICE_STATE :
				transitionDeletePractice();
				break;
			case ENTER_TEACHER_STATE :
				transitionEnterTeacher();
				break;
			case ENTER_DESCRIPTION_STATE : 
				transitionEnterDescription();
				break;
			case ENTER_FILES_STATE : 
				transitionEnterFiles();
				break;
			case ABOUT_STATE : 
				transitionAbout();
				break;
			default :
				reportLog("Error in transition() : Unknown state \"" + getAppState() + "\".");
				throw new PortalException("Unknown state \"" + getAppState() + "\".");
		}
	}

	/**
	 * Set next application state when when appState == DISPLAY_PRACTICES_STATE.
	 *
	 * @throws PortalException PortalException.
	 */
	private void transitionDisplayPractices() throws PortalException {	

		String action = getRuntimeParameter("action");

		if (action == null) {
			return;
		}
		
		if (action.equals("refresh")) {
			return;
		}
		
		if (action.equals("delete")) {
		
			long practiceId = Long.parseLong(getRuntimeParameter("practiceId"));
			CPractice practice = getCPractice(practiceId);
			if (!practice.isUserGranted(getCurrentUserId())) {
				reportLog("Error in transitionDisplayPractices() : User \"" + getCurrentUserId() 
						+ "\" was not granted to delete practice \"" + practiceId + "\".");
				throw new PortalException("User \"" + getCurrentUserId() 
						+ "\" was not granted to delete practice \"" + practiceId + "\".");
			}
			getSessionParams().setPracticeId(practiceId);
			setAppState(DELETE_PRACTICE_STATE);
			return;

		}
		
		if (action.equals("sort")) {

			if (getSessionParams().getSortOpt().equals(CSessionParams.DATE_SORT)) {
				getSessionParams().setSortOpt(CSessionParams.USER_SORT);
			} else {
				getSessionParams().setSortOpt(CSessionParams.DATE_SORT);
			}
			return;

		}
		
		if (action.equals("sendPractice")) {
		
			setAppState(ENTER_TEACHER_STATE);
			return;
			
		}
		
		if (action.equals("displayFile")) {

			long fileId = Long.parseLong(getRuntimeParameter("fileId"));
			CFile downloadFile = getFile(fileId); 
			CPractice practice = getCPractice(downloadFile.getCPracticeId());
			if (!practice.isUserGranted(getCurrentUserId())) {
				reportLog("Error in transitionDisplayPractices() : User \"" + currentUser.getId() 
						+ "\" was not granted to view file \"" + fileId + "\".");
				throw new PortalException("User \"" + currentUser.getId() 
						+ "\" was not granted to view file \"" + fileId + "\".");
			}
			getSessionParams().setFileId(fileId);
			setAppState(DISPLAY_FILE_STATE);
			return;

		}
		
		if (action.equals("downloadFile")) {

			long fileId = Long.parseLong(getRuntimeParameter("fileId"));
			CFile downloadFile = getFile(fileId);
			CPractice practice = getCPractice(downloadFile.getCPracticeId());
			if (!practice.isUserGranted(getCurrentUserId())) {
				reportLog("Error in transitionDisplayPractices() : User \"" + currentUser.getId() 
						+ "\" was not granted download file \"" + fileId + "\".");
				throw new PortalException("User \"" + getCurrentUserId() 
						+ "\" was not granted to download file \"" + fileId + "\".");
			}
			getSessionParams().setFileId(fileId);
			setAppState(DOWNLOAD_FILE_STATE);
			return;

		}
		
		if (action.equals("downloadPractice")) {

			long practiceId = Long.parseLong(getRuntimeParameter("practiceId"));
			CPractice practice = getCPractice(practiceId);
			if (!practice.isUserGranted(getCurrentUserId())) {
				reportLog("Error in transitionDisplayPractices() : User \"" + currentUser.getId() 
						+ "\" was not granted download practice \"" + practiceId + "\".");
				throw new PortalException("User \"" + getCurrentUserId() 
						+ "\" was not granted to download practice \"" + practiceId + "\".");
			}
			getSessionParams().setPracticeId(practiceId);
			setAppState(DOWNLOAD_PRACTICE_STATE);
			return;

		}
		
		if (action.equals("downloadAll")) { 

			getSessionParams().setSent(getRuntimeParameter("sent").equals("true"));
			setAppState(DOWNLOAD_ALL_STATE);
			return;

		}
		
		reportLog("Error in transitionDisplayPractices() : Unknown action.");
		throw new PortalException("Unknown action.");
	}	
	
	/**
	 * Set next application state when setRuntimeData() when appState == DISPLAY_FILE_STATE.
	 *
	 * @throws PortalException PortalException.
	 */
	private void transitionDisplayFile() throws PortalException {
		String action = getRuntimeParameter("action");

		if (action == null) {
			return;
		}
		
		if (action.equals("back")) {
			setAppState(DISPLAY_PRACTICES_STATE);
			return;
		}

		reportLog("Error in transitionDisplayFile() : Unknown action.");
		throw new PortalException("Unknown action.");
	}	
	
	/**
	 * Set next application state when setRuntimeData() when appState == DELETE_PRACTICE_STATE.
	 *
	 * @throws PortalException PortalException.
	 */
	private void transitionDeletePractice() throws PortalException {
		String action = getRuntimeParameter("action");

		if (action == null) {
			return;
		}
		
		if (action.equals("doIt")) {
			CPractice practice = getCPractice(getSessionParams().getPracticeId());
			getPersistenceManager().currentTransaction().begin();
			getPersistenceManager().deletePersistent(practice);
			getPersistenceManager().currentTransaction().commit();
			setAppState(DISPLAY_PRACTICES_STATE);
			return;

		}
		
		if (action.equals("cancel")) {
		
			setAppState(DISPLAY_PRACTICES_STATE);
			return;
			
		}

		reportLog("Error in transitionDeletePractice() : Unknown action.");
		throw new PortalException("Unknown action.");
	}

	/**
	 * Set next application state when setRuntimeData() when appState == ENTER_TEACHER_STATE.
	 *
	 * @throws PortalException PortalException.
	 */
	private void transitionEnterTeacher() throws PortalException {
		String action = getRuntimeParameter("action");

		if (action == null) {
			return;
		}
		
		if (action.equals("next")) {

			String receiverId = getRuntimeParameter("receiverId");
			if (receiverId == null) {
				reportLog("ReceiverId parameter should not be null here.");
				throw new PortalException("ReceiverId parameter should not be null here.");
			}
			receiverId = receiverId.trim();
			getSessionParams().setReceiverId(receiverId);
			if (receiverId.equals("")) {
				setErrorMsg("Veuillez entrer l'uid du destinataire.");
				return;
			}
			CUser receiver = getCUser(receiverId);
			if (receiver == null) {
				setErrorMsg("L'utilisateur \"" + receiverId + "\" n'existe pas.");
				return;
			}
			if (!receiver.canReceive()) {
				setErrorMsg("Cet utilisateur n'est pas habilit  recevoir des TPs.");
				return;
			}
			setAppState(ENTER_DESCRIPTION_STATE);
			return;

		}
		
		if (action.equals("cancel")) {

			setAppState(DISPLAY_PRACTICES_STATE);
			return;

		}

		reportLog("Error in transitionEnterTeacher() : Unknown action.");
		throw new PortalException("Unknown action.");
	}
	
	/**
	 * Set next application state when setRuntimeData() when appState == ENTER_DESCRIPTION_STATE.
	 *
	 * @throws PortalException PortalException.
	 */
	private void transitionEnterDescription() throws PortalException {
		String action = getRuntimeParameter("action");

		if (action == null) {
			return;
		}
		
		if (action.equals("next")) {

			String description = getRuntimeParameter("description");
			if (description == null) {
				reportLog("Description parameter should not be null here.");
				throw new PortalException("Description parameter should not be null here.");
			}
			getSessionParams().setDescription(description);
			setAppState(ENTER_FILES_STATE);
			return;

		}
		
		if (action.equals("back")) {	

			setAppState(ENTER_TEACHER_STATE);
			return;

		}
		
		if (action.equals("cancel")) {

			setAppState(DISPLAY_PRACTICES_STATE);
			return;

		}

		reportLog("Error in transitionEnterDescription() : Unknown action.");
		throw new PortalException("Unknown action.");
	}
	
	/**
	 * Set next application state when setRuntimeData() when appState == ENTER_FILES_STATE.
	 *
	 * @throws PortalException PortalException.
	 */
	private void transitionEnterFiles() throws PortalException {	
		Vector files = new Vector();
		
		String action = getRuntimeParameter("action");	
		
		if (action == null) {
			return;
		}
		
		if (action.equals("sendPractice")) {	

			MultipartDataSource src;		
			CFile uploadedFile = null;	
			
			for (int i = 0; i < getConfig().getUploadFileNumber(); i++) {
				// Test if file is present (first file is needed, next are optional)
				try {
					src = getUploadedFile("file" + i);
				} catch (java.lang.NullPointerException ex) {
					if (i == 0) {
						setErrorMsg("Vous devez spcifier au moins un fichier.");
						return;
					}
					break;
				}
				// Test if this file contains data, and create a CFile instance. The cpracticeId to bound is set later.
				try {
					if (src.getInputStream().available() == 0) {
						setErrorMsg("Vous devez spcifier un fichier valide.");
						return;
					}
					byte[] buffer = IOUtils.toByteArray(src.getInputStream());
					uploadedFile = new CFile(0, src.getName(), buffer);
				} catch (IOException ex) {
					setErrorMsg("Erreur de tlchargement de fichier :\n" + ex);
					return;
				} catch (Exception ex) {
					reportLog("Error : " + ex);
					throw new PortalException(ex.getMessage());
				}
				files.addElement(uploadedFile);
			}
			
			CUser receiver = getCUser(getSessionParams().getReceiverId());
			if (receiver == null) {	
				reportLog("User \"" + getSessionParams().getReceiverId() + "\" not found by uPortal");
				throw new PortalException("User \"" + getSessionParams().getReceiverId() + "\" not found by uPortal");
			}

			try {
				// start a new transaction
				getPersistenceManager().currentTransaction().begin();
				// create a practice
				CPractice practice = new CPractice(currentUser, receiver, getSessionParams().getDescription());
				// mark the practice persistent in the database
				getPersistenceManager().makePersistent(practice);
				Iterator iter = files.iterator();
				while (iter.hasNext()) {
					CFile file = (CFile) iter.next();
					file.setCPracticeId(practice.getId());
				}
				practice.setFiles(files);
				getPersistenceManager().currentTransaction().commit();
			} catch (OjbStoreFatalInternalException ex) {
				setErrorMsg("Un de vos fichiers excde la taille autoris.");
				return;
			} 
			
			// files were successfully uploaded.
			setAppState(DISPLAY_PRACTICES_STATE);
			return;
			
		}
		
		if (action.equals("back")) {	

			setAppState(ENTER_DESCRIPTION_STATE);
			return;

		} 
		
		if (action.equals("cancel")) {

			setAppState(DISPLAY_PRACTICES_STATE);
			return;

		}

		reportLog("Error in transitionEnterFiles() : Unknown action.");
		throw new PortalException("Unknown action.");
	}	
			
	/**
	 * Build the XML output that will be later used by renderXML(), or
	 * call setDownloadInfo() to set download data.
	 * 
	 * @throws PortalException PortalException.
	 */
	protected void buildOutput() throws PortalException {
		
		switch (getAppState()) {
			case DOWNLOAD_FILE_STATE :
				setAppState(getDefaultAppState());
				buildDownloadFile();
				break;
			case DOWNLOAD_PRACTICE_STATE :
				setAppState(getDefaultAppState());
				buildDownloadPractice();
				break;
			case DOWNLOAD_ALL_STATE :
				setAppState(getDefaultAppState());
				buildDownloadAll();
				break;
			case DISPLAY_PRACTICES_STATE :
				buildXMLDisplayPractices();
				break;
			case DISPLAY_FILE_STATE :
				buildXMLDisplayFile();
				break;
			case DELETE_PRACTICE_STATE :
				buildXMLDeletePractice();
				break;
			case ENTER_TEACHER_STATE :
				buildXMLEnterTeacher();
				break;
			case ENTER_DESCRIPTION_STATE : 
				buildXMLEnterDescription();
				break;
			case ENTER_FILES_STATE : 
				buildXMLEnterFiles();
				break;
			case ABOUT_STATE : 
				buildXMLAbout();
				break;
			default :
				reportLog("Error in buildOutput() : Unknown state \"" + getAppState() + "\".");
				throw new PortalException("Unknown state.");
		}
	}

	/**
	 * Build the XML output when appState == DISPLAY_PRACTICES_STATE.
	 */
	private void buildXMLDisplayPractices() {
		Collection practiceSentList = null;
		Collection practiceReceivedList = null;
		List lpracticeSentList = null;
		List lpracticeReceivedList = null;
		Iterator iter;
		CPractice pratice;

		// Search the for practices to be displayed
		getPersistenceManager().currentTransaction().begin();
		Query query = getPersistenceManager().newQuery(CPractice.class, "sender.id == \"" + currentUser.getId() + "\"");
		practiceSentList = (Collection) query.execute();	
		query = getPersistenceManager().newQuery(CPractice.class, "receiver.id == \"" + currentUser.getId() + "\"");
		practiceReceivedList = (Collection) query.execute();	
		getPersistenceManager().currentTransaction().commit();

		lpracticeSentList = new Vector(practiceSentList);		
		if (getSessionParams().getSortOpt().equals(CSessionParams.DATE_SORT)) {
			Collections.sort((List) lpracticeSentList, new CPracticeDateComparer());
		} else {
			Collections.sort((List) lpracticeSentList, new CPracticeReceiverComparer());
		}

		lpracticeReceivedList = new Vector(practiceReceivedList);
		
		if (getSessionParams().getSortOpt().equals(CSessionParams.DATE_SORT)) {
			Collections.sort((List) lpracticeReceivedList, new CPracticeDateComparer());
		} else {
			Collections.sort((List) lpracticeReceivedList, new CPracticeSenderComparer());				
		}
				
		String output = XML_PROLOGUE 
				+ "\n<display-practices>";
		
		if (getSessionParams().getSortOpt().equals(CSessionParams.DATE_SORT)) {
			output = output + "\n<sort-text>Tri par personne</sort-text>"; 
		} else {
			output = output + "\n<sort-text>Tri par date</sort-text>"; 
		} 
		
		if (!lpracticeSentList.isEmpty()) {
			output = output + "\n<sent>";
			iter = lpracticeSentList.iterator();
			while (iter.hasNext()) {
				pratice = (CPractice) iter.next();
				output = output + pratice.toXML(true, getConfig().getOutputDescriptionLength());
			}
			output = output + "\n</sent>";
		}	

		if (currentUser.canReceive() && !lpracticeReceivedList.isEmpty()) {
			output = output + "\n<received>";
			iter = lpracticeReceivedList.iterator();
			while (iter.hasNext()) {
				pratice = (CPractice) iter.next();
				output = output + pratice.toXML(true, getConfig().getOutputDescriptionLength());
			}
			output = output + "\n</received>";
		}
		
		output = output 
				+ XMLErrorMsg()
				+ "\n</display-practices>";
		
		setXMLOutput(output);
	}
	
	
	/**
	 * Build the XML output when appState == DISPLAY_FILE_STATE.
	 * @throws PortalException PortalException.
	 */
	private void buildXMLDisplayFile() throws PortalException {
		
		CFile file = getFile(getSessionParams().getFileId());
		CPractice practice = getCPractice(file.getCPracticeId());
		setXMLOutput(XML_PROLOGUE
					+ "\n<display-file>"
					+ practice.toXML(false, 0)
					+ file.toXML(true)
					+ XMLErrorMsg()
					+ "\n</display-file>");
	}
	
	/**
	 * Build the XML output when appState == DELETE_PRACTICE_STATE.
	 * @throws PortalException PortalException.
	 */
	private void buildXMLDeletePractice() throws PortalException {
		CPractice practice = getCPractice(getSessionParams().getPracticeId());
		setXMLOutput(XML_PROLOGUE
					+ "\n<delete-practice>"
					+ practice.toXML(false, 0)
					+ XMLErrorMsg()
					+ "\n</delete-practice>");
	}
	
	/**
	 * Build the XML output when appState == ENTER_TEACHER_STATE.
	 */
	private void buildXMLEnterTeacher() {
		CUser receiver = getCUser(getSessionParams().getReceiverId());
		
		String receiverStr = "";
		if (receiver != null) {
			receiverStr = receiver.toXML();
		}
		setXMLOutput(XML_PROLOGUE
					+ "\n<enter-teacher>"
					+ "\n<receiver>" + receiverStr + "\n</receiver>"
					+ XMLErrorMsg()
					+ "\n</enter-teacher>");
	}

	/**
	 * Build the XML output when appState == ENTER_DESCRIPTION_STATE.
	 */
	private void buildXMLEnterDescription() { 
		CUser receiver = getCUser(getSessionParams().getReceiverId());
		String receiverStr = "";
		if (receiver != null) {
			receiverStr = receiver.toXML();
		}

		setXMLOutput(XML_PROLOGUE
					+ "\n<enter-description>"
					+ "\n<receiver>" + receiverStr + "\n</receiver>"
					+ "\n<description>" 
					+ EsupXMLEscaper.escape(getSessionParams().getDescription()) 
					+ "\n</description>"
					+ XMLErrorMsg()
					+ "\n</enter-description>");
	}
	
	/**
	 * Build the XML output when appState == ENTER_FILES_STATE.
	 */
	private void buildXMLEnterFiles() {

		CUser receiver = getCUser(getSessionParams().getReceiverId());
		String receiverStr = "";
		if (receiver != null) {
			receiverStr = receiver.toXML();
		}

		setXMLOutput(XML_PROLOGUE
					+ "\n<enter-files>"
					+ "\n<receiver>"
					+ receiverStr
					+ "\n</receiver>"
					+ "\n<description>" + EsupXMLEscaper.escape(getSessionParams().getDescription()) + "</description>"
					+ XMLErrorMsg()
					+ "\n</enter-files>");
	}

	/**
	 * Build download data when appState == DOWNLOAD_FILE_STATE.
	 * 
	 * @throws PortalException PortalException
	 */
	private void buildDownloadFile() throws PortalException {
		CFile file = getFile(getSessionParams().getFileId());
		setDownloadInfo(file.getContent(), file.getFilename());
	}

	/**
	 * Build download data when appState == DOWNLOAD_PRACTICE_STATE.
	 * 
	 * @throws PortalException PortalException
	 */
	private void buildDownloadPractice() throws PortalException {
		byte[] downloadContent = getZippedPracticeContent(getSessionParams().getPracticeId());
		if (downloadContent == null) {
			reportLog("An error occured while building the zip file corresponding to practice \"" 
									+ getSessionParams().getPracticeId() 
									+ "\" for user \"" + getCurrentUserId() + "\".");
			throw new PortalException("An error occured while building the zip file corresponding to practice \"" 
									+ getSessionParams().getPracticeId() 
									+ "\" for user \"" 
									+ getCurrentUserId() 
									+ "\".");
		}
		String downloadFilename = getZippedPracticeFilename(getSessionParams().getPracticeId());
		setDownloadInfo(downloadContent, downloadFilename);
	}

	/**
	 * Build download data when appState == DOWNLOAD_ALL_STATE.
	 * 
	 * @throws PortalException PortalException
	 */
	private void buildDownloadAll() throws PortalException {
		byte[] downloadContent = getZippedPracticesContent(getSessionParams().getSent());
		if (downloadContent == null) {
			String str;
			if (getSessionParams().getSent()) {
				str = "by";
			} else {
				str = "to";
			}
			reportLog("An error occured while building the zip file corresponding to practices sent " 
					+ str + " user \""  + getCurrentUserId()  + "\".");	
			throw new PortalException(
					"An error occured while building the zip file corresponding to practices sent " 
					+ str
					+ " user \"" 
					+ getCurrentUserId() 
					+ "\".");				
		}
		String downloadFilename = getZippedPracticesFilename();
		setDownloadInfo(downloadContent, downloadFilename);
	} 

	/**
	 * Retrieve the name of the main XSL file to use for the XSL transformation.
	 * 
	 * @return a String instance.
	 */
	protected String getXSLMainFileName() {
		return "esup-practice.xsl";
	}

	/**
	 * Set specific parameters for the XSL transformation.
	 * 
	 * @param xslt an XSL transformer
	 * This method should be overridden if specific XLS parameters are used.
	 * @throws PortalException PortalException.
	 */
	protected void setXSLParameters(final XSLT xslt) throws PortalException {
		super.setXSLParameters(xslt);
		xslt.setStylesheetParameter("sortOpt", getSessionParams().getSortOpt());
		xslt.setStylesheetParameter("uploadFileNumber", "" + getConfig().getUploadFileNumber());
	}

	/**
	 * Retrieve the (CUser) user corresponding to a uid from the database. If the 	
	 * user is not found in the database, then it's created if found in the LDAP
	 * directory (otherwise null is returned).
	 * 
	 * @param userId the id of the user.
	 * @return a CUser instance or null if not found.
	 */
	private CUser getCUser(final String userId) {
		CUser user = null;
		
		// return null if the id was not set (happens in buildXMLEnterTeacher()).
		if (userId.equals("")) {
			return null;
		}

		getPersistenceManager().currentTransaction().begin();
		Query query = getPersistenceManager().newQuery(CUser.class, "id == \"" + userId + "\"");
		Collection users = (Collection) query.execute();
		getPersistenceManager().currentTransaction().commit();

		if (!users.isEmpty()) {
			Iterator iter = users.iterator();
			return (CUser) iter.next();
		}		

		
		// try to fetch the user from the LDAP directory
		RestrictedPerson person = getPortalPerson(userId); 	
		if (person == null) {
			// no user corresponding in the LDAP directory
			return null;
		}

		boolean canReceive = false;
		Iterator iter = getConfig().getReceivers().iterator();
		while (iter.hasNext() && !canReceive) {
			Receiver receiver = (Receiver) iter.next();
			
			if (person.getAttribute(receiver.getAttribute()) != null) {
				if (((String) person.getAttribute(receiver.getAttribute())).equals(receiver.getValue())) {
					canReceive = true;
				}
			}
		}
		
		user = new CUser(userId, 
						(String) person.getAttribute(getConfig().getUportalAttributeDisplayName()), 
						(String) person.getAttribute(getConfig().getUportalAttributeEmail()), 
						canReceive);
		
		// Mark persistent this user.
		getPersistenceManager().currentTransaction().begin();
		getPersistenceManager().makePersistent(user);
		getPersistenceManager().currentTransaction().commit();
	
		return user;
	}
	
	/**
	 * Retreive a practice from the database.
	 * 
	 * @param practiceId the id of the practice.
	 * @return a CPractice instance.
	 * @throws PortalException when the practice is not found in the database.
	 */
	private CPractice getCPractice(final long practiceId) throws PortalException {

		CPractice practice = null;

		getPersistenceManager().currentTransaction().begin();
		Query newQuery = getPersistenceManager().newQuery(CPractice.class, "id == " + practiceId);
		Collection results = (Collection) newQuery.execute();
		getPersistenceManager().currentTransaction().commit();
		Iterator iter = results.iterator();
		practice = (CPractice) iter.next();
		if (practice == null) {
			reportLog("Practice \"" + practiceId + "\" not found.");
			throw new PortalException("Practice \"" + practiceId + "\" not found.");
		}
		return practice;
	}

	/**
	 * Retreive a file from the database.
	 * 
	 * @param fileId the id of the file.
	 * @return A CFile instance.
	 * @throws PortalException when the practice is not found in the database.
	 */
	private CFile getFile(final long fileId) throws PortalException {	

		CFile file = null;
		Collection results = null;

		getPersistenceManager().currentTransaction().begin();
		Query newQuery = getPersistenceManager().newQuery(CFile.class, "id == " + fileId);
		results = (Collection) newQuery.execute();
		Iterator iter = results.iterator();
		file = (CFile) iter.next();
		if (file == null) {
			reportLog("File \"" + fileId + "\" not found.");
			throw new PortalException("File \"" + fileId + "\" not found.");
		}
		getPersistenceManager().currentTransaction().commit();
		
		return file;
	}

	/**
	 * Retreive a practice as a zipped array of bytes.
	 * 
	 * @param practiceId the id of the practice
	 * @return an array of bytes, or null if an error occured when zipping.
	 * @throws PortalException PortalException.
	 */
	private byte[] getZippedPracticeContent(final long practiceId) throws PortalException {

		// creat a simple stream
	    ByteArrayOutputStream outputStream = new ByteArrayOutputStream();

		// get the practice
		CPractice practice = getCPractice(practiceId);

	    try {
			// zip the simple stream
	    	ZipOutputStream out = new ZipOutputStream(outputStream);
		    // add the practice to the stream
	        practice.addToZipStream(out);
	        // close the stream
	        out.close();
	    } catch (IOException ex) {
	    	reportLog("IOException when the creation of a zip file : " + ex);
	    	return null;
	    }

	    // return an array of bytes
	    return outputStream.toByteArray();
	}	
	
	/**
	 * Retrieve the filename of a zipped practice.
	 * 
	 * @param practiceId the id of the practice
	 * @return a string.
	 * @throws PortalException PortalException.
	 */
	private String getZippedPracticeFilename(final long practiceId) throws PortalException {
		// get the practice
		CPractice practice = getCPractice(practiceId);
		return practice.getZipPrefix() + ".zip";
	}	
	
	/**
	 * Retreive a set of practices as a zipped array of bytes.
	 * 
	 * @param sent A boolean set to true to zip sent practices, false to zip received practices.
	 * @return an array of bytes, or null if an error occured when zipping.
	 * @throws PortalException PortalException.
	 */
	private byte[] getZippedPracticesContent(final boolean sent) throws PortalException {
		Query newQuery;
		Collection practices = null;

		// creat a simple stream
	    ByteArrayOutputStream outputStream = new ByteArrayOutputStream();

	    try {
			// zip the simple stream
	    	ZipOutputStream out = new ZipOutputStream(outputStream);

			getPersistenceManager().currentTransaction().begin();
			if (sent) {
				newQuery = getPersistenceManager().newQuery(CPractice.class, 
															"sender.id == \"" + getCurrentUserId() + "\"");
			} else {
				newQuery = getPersistenceManager().newQuery(CPractice.class, 
															"receiver.id == \"" + getCurrentUserId() + "\"");
			}
			practices = (Collection) newQuery.execute();
			getPersistenceManager().currentTransaction().commit();

			Iterator iter = practices.iterator();
	    	while (iter.hasNext()) {
	    		CPractice practice = (CPractice) iter.next();	
				// add the practice to the stream
	    		practice.addToZipStream(out);
	    	}

	        // close the stream
	        out.close();
	    } catch (IOException ex) {
	    	reportLog("IOException when the creation of a zip file : " + ex);
	    	return null;
	    }

	    // return an array of bytes
	    return outputStream.toByteArray();
	}	

	/**
	 * Retreive the filename of zipped practices.
	 * 
	 * @return a string.
	 * @throws PortalException PortalException.
	 */
	private String getZippedPracticesFilename() throws PortalException {
		return getConfig().getDownloadZipFilesPrefix() 
			 + "-" 
			 + TimestampFormater.shortFormat(new java.sql.Timestamp(new Date().getTime())) 
			 + ".zip";
	}	


	/**
	 * The class implementing session parameters of the channel.
	 * Other session varaibles, such as currentUser or appState, 
	 * are implemented directly into the main channel class
	 */
	private final class CSessionParams {
		
		/**
		 * A file id.
		 */
		private long fileId = 0;
		/**
		 * @return Returns the fileId.
		 */
		private long getFileId() {
			return fileId;
		}
		/**
		 * @param fileId The fileId to set.
		 */
		private void setFileId(final long fileId) {
			this.fileId = fileId;
		}

		/**
		 * A boolean.
		 */
		private boolean sent;
		/**
		 * @return Returns the sent.
		 */
		private boolean getSent() {
			return sent;
		}
		/**
		 * @param sent The sent to set.
		 */
		private void setSent(final boolean sent) {
			this.sent = sent;
		}

		/**
		 * A practice Id (session variable).
		 */
		private long practiceId = 0;
		/**
		 * @return Returns the practiceId.
		 */
		private long getPracticeId() {
			return practiceId;
		}
		/**
		 * @param practiceId The practiceId to set.
		 */
		private void setPracticeId(final long practiceId) {
			this.practiceId = practiceId;
		}

		/**
		 * A user Id (to store the user the practice will be sent to, session variable).
		 */
		private String receiverId = "";
		/**
		 * @return Returns the receiverId.
		 */
		private String getReceiverId() {
			return receiverId;
		}
		/**
		 * @param receiverId The receiverId to set.
		 */
		private void setReceiverId(final String receiverId) {
			this.receiverId = receiverId;
		}

		/**
		 * The description of the practice to be sent (session variable).
		 */
		private String description = "";
		/**
		 * @return Returns the description.
		 */
		private String getDescription() {
			return description;
		}
		/**
		 * @param description The description to set.
		 */
		private void setDescription(final String description) {
			this.description = description;
		}

		/**  sort practices by date */
		private static final String DATE_SORT = "DATE";
		/**  sort practices by sender/receiver name */
		private static final String USER_SORT = "SENDER";
		/**  Default sort option  */
		private static final String DEFAULT_SORT = DATE_SORT;
		/** the sorting option */
		private String sortOpt = DEFAULT_SORT;
		
		/**
		 * @return Returns the sortOpt.
		 */
		private String getSortOpt() {
			return sortOpt;
		}
		/**
		 * @param sortOpt The sortOpt to set.
		 */
		private void setSortOpt(final String sortOpt) {
			this.sortOpt = sortOpt;
		}
	}

	/**
	 * a CSessionParams instance to store the session parameters.
	 * 
	 * @uml.property name="sessionParams"
	 * @uml.associationEnd multiplicity="(1 1)" inverse="this$0:org.esupportail.portal.channels.CPracticeChannel.CPracticeChannel$CSessionParams"
	 */
	private CSessionParams sessionParams = null;

	/**
	 * @return Returns the sessionParams.
	 * 
	 * @uml.property name="sessionParams"
	 */
	private CSessionParams getSessionParams() {
		return sessionParams;
	}

}

