/**
 * ESUP-Portail Practice - Copyright (c) 2004 ESUP-Portail consortium 
 * 
 * This program is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software Foundation.
 * 
 * See the license terms site for more details: http://www.esup-portail.org/license
 * 
 * @author Alexandre Boisseau <thetyrex at users.sourceforge.net>
 * @author Pascal Aubry  <pascal.aubry at univ-rennes1.fr>
*/

package org.esupportail.portal.channels.CPracticeChannel;

import java.io.IOException;
import java.util.Iterator;
import java.util.Vector;
import org.apache.commons.digester.Digester;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import java.net.URL;

/**
 * This class is used by digester to read theconfiguration file of the application.
 */
public final class CPracticeChannelConfig {
	
	
	/** 
	 * A singleton is used to insure that a single instance of the class is created.
	 */
	private static CPracticeChannelConfig singleton;
	
	/** 
	 * The location of the configuration file.
	 */
	private static String configFilename = "/properties/channels/CPracticeChannel/CPracticeChannelConfig.xml";
	
	/** 
	 * The maximum number of files a user can upload when sending a practice.
	 */
	private int uploadFileNumber;

	/**
	 * @return The maximum number of files a user can upload when sending a practice.
	 * 
	 * @uml.property name="uploadFileNumber"
	 */
	public int getUploadFileNumber() {
		return uploadFileNumber;
	}

	/** 
	 * The name of the uPortal attribute retrieved to get users' display name.
	 */
	private String uportalAttributeDisplayName;

	/**
	 * @return The name of the uPortal attribute retrieved to get users' display name.
	 * 
	 * @uml.property name="uportalAttributeDisplayName"
	 */
	public String getUportalAttributeDisplayName() {
		return uportalAttributeDisplayName;
	}

	
	/** 
	 * The name of the uPortal attribute retrieved to get users' display name.
	 */
	private String uportalAttributeEmail;

	/**
	 * @return The name of the uPortal attribute retrieved to get users' display name.
	 * 
	 * @uml.property name="uportalAttributeEmail"
	 */
	public String getUportalAttributeEmail() {
		return uportalAttributeEmail;
	}

	
	/** 
	 * The string used to prefix folders names when downloading zip files.
	 */
	private String downloadZipFilesPrefix;

	/**
	 * @return the string used to prefix folders names when downloading zip files.
	 * 
	 * @uml.property name="downloadZipFilesPrefix"
	 */
	public String getDownloadZipFilesPrefix() {
		return downloadZipFilesPrefix;
	}

	
	/** 
	 * The maximum length of descriptions that will be displayed .
	 */
	private int outputDescriptionLength;

	/**
	 * @return Returns the descriptionLength.
	 * 
	 * @uml.property name="outputDescriptionLength"
	 */
	public int getOutputDescriptionLength() {
		return outputDescriptionLength;
	}

	/**
	 * A vector to store name/value pairs. These pairs are used to known on which 
	 * values of uPortal attributes the application should rely on to decide whether
	 * a user can receive practices or not
	 * 
	 * @uml.property name="receivers"
	 * @uml.associationEnd multiplicity="(0 -1)" elementType="org.esupportail.portal.channels.CPracticeChannel.Receiver"
	 */
	private Vector receivers;

	/**
	 * @return a vector of name/value pairs of uportal attributes the application 
	 * should rely on to decide whether a user can receive practices or not
	 * 
	 * @uml.property name="receivers"
	 */
	public Vector getReceivers() {
		return receivers;
	}

	
	
	/**
	 * Constructor.
	 * @throws Exception Exception
	 */
	private CPracticeChannelConfig() throws Exception {
		Digester dig;
		URL resourceURL;
		receivers = new Vector();
		
		try {
			Class requestingClass = getClass();
			resourceURL = requestingClass.getResource(configFilename);
			if (resourceURL == null) {
				throw new Exception(configFilename + " does not exist ");
			}
			dig = new Digester();
		} catch (IOException ex) {
			throw new Exception("Impossible to load file " + configFilename + " : \n" + ex);
		} catch (SAXException ex) {
			System.out.println(configFilename + " must be well-formed " + ex);
			throw new Exception(ex);
		}
		
		dig.push(this);
		dig.addCallMethod("config", "addConfig", 5);
		dig.addCallParam("config/uportal-attributes/display-name", 0);
		dig.addCallParam("config/uportal-attributes/email", 1);
		dig.addCallParam("config/upload/file-number", 2);
		dig.addCallParam("config/download/zipfiles-prefix", 3);
		dig.addCallParam("config/output/description-length", 4);    	
		dig.parse(new InputSource(resourceURL.toExternalForm()));
		
		dig.push(this);
		dig.addObjectCreate("config/receivers/receiver", Receiver.class);
		dig.addCallMethod("config/receivers/receiver/attribute", "setAttribute", 1);
		dig.addCallParam("config/receivers/receiver/attribute", 0);
		dig.addCallMethod("config/receivers/receiver/value", "setValue", 1);
		dig.addCallParam("config/receivers/receiver/value", 0);
		dig.addSetNext("config/receivers/receiver", "addListReceiver");
		dig.parse(new InputSource(resourceURL.toExternalForm()));
	}
	
	
	/**
	 * Callback used for non iterative configuration parameters.
	 *
	 * @param uportalAttributeDisplayName callback parameter
	 * @param uportalAttributeEmail callback parameter
	 * @param uploadFileNumber callback parameter
	 * @param downloadZipFilesPrefix callback parameter
	 * @param outputDescriptionLength callback parameter
	 */
	public void addConfig(final String uportalAttributeDisplayName, 
			final String uportalAttributeEmail,
			final String uploadFileNumber,
			final String downloadZipFilesPrefix, 
			final String outputDescriptionLength) {
		this.uportalAttributeDisplayName = uportalAttributeDisplayName;
		this.uportalAttributeEmail = uportalAttributeEmail;
		this.uploadFileNumber = Integer.parseInt(uploadFileNumber);
		this.downloadZipFilesPrefix = downloadZipFilesPrefix;
		this.outputDescriptionLength = Integer.parseInt(outputDescriptionLength);
	}
	
	/**
	 * The getInstance method is overwritten to insure that at most one instance
	 * of our class lives at any moment.
	 *
	 * @return An instance of CPracticeChannelConfig.
	 * @throws Exception Exception
	 */
	public static CPracticeChannelConfig getInstance() throws Exception {
		if (singleton == null) {
			singleton = new CPracticeChannelConfig();
		}
		return singleton;
	}
	
	/**
	 * Callback used for the 'receivers' parameters. Adds a Receiver object
	 * to the list.
	 * @param receiver another parameter.
	 */
	public void addListReceiver(final Receiver receiver) {
		receivers.add(receiver);
	}
	
	
	/**
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		String s = "";
		Iterator iter = receivers.iterator();
		int i = 1;
		Receiver r = null;
		while (iter.hasNext()) {
			r = (Receiver) iter.next();
			s = s + "\n   receiver" + i + " : ";
			s = s + "attribut=" + r.getAttribute() + " ; value=" + r.getValue();
			i++;
		}	
		return "Object CPracticeChannelConfig : " 
		+ "\n - DisplayName = " + getUportalAttributeDisplayName()
		+ "\n - Email = " + getUportalAttributeEmail()
		+ "\n - ZipFilesPrefix = " + getDownloadZipFilesPrefix()
		+ "\n - DescriptionLength = " + getOutputDescriptionLength()
		+ "\n - FileNumber = " + getUploadFileNumber()
		+ "\n - receivers : " + s;
	}
	
}



