/**
 * ESUP-Portail Practice - Copyright (c) 2004 ESUP-Portail consortium 
 * 
 * This program is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software Foundation.
 * 
 * See the license terms site for more details: http://www.esup-portail.org/license
 * 
 * @author Alexandre Boisseau <thetyrex at users.sourceforge.net>
 * @author Pascal Aubry  <pascal.aubry at univ-rennes1.fr>
*/

package org.esupportail.portal.channels.CPracticeChannel;

import org.esupportail.portal.channels.util.format.EsupXMLEscaper;

/**
 * A user of the application.
 * Data-object, mapped to the user table of the database.
 */
public class CUser {

	/**
	 * The user's unique id.
	 * 
	 * @uml.property name="id"
	 */
	private String id;

	/**
	 * @return the user's unique id.
	 * 
	 * @uml.property name="id"
	 */
	public final String getId() {
		return id;
	}

	/**
	 * The user's complete name, as displayed by the application.
	 * 
	 * @uml.property name="displayName"
	 */
	private String displayName;

	/**
	 * @return the user's complete name.
	 * 
	 * @uml.property name="displayName"
	 */
	public final String getDisplayName() {
		return displayName;
	}

	/**
	 * The user's email.
	 * 
	 * @uml.property name="email"
	 */
	private String email;

	/**
	 * @return the user's email.
	 * 
	 * @uml.property name="email"
	 */
	public final String getEmail() {
		return email;
	}

	/**
	 * A flag to remember if a user can receive practices or not.
	 * 
	 * @uml.property name="canReceive"
	 */
	private boolean canReceive;

	/**
	 * @return true if the user can receive practices, false otherwise.
	 * 
	 * @uml.property name="canReceive"
	 */
	public final boolean canReceive() {
		return canReceive;
	}
	
	/**
	 * Constructor.
	 * 
	 * @param id the user's unique id.
	 * @param displayName the user's display name.
	 * @param email The user's email.
	 * @param canReceive true to allow the user to receive practices, false otherwise.
	 */
	public CUser(final String id, 
		     final String displayName, 
		     final String email, 
		     final boolean canReceive) {
		this.id = id;
		this.displayName = displayName;
		this.email = email;
		this.canReceive = canReceive;
	}
		
	/**
	 * Convert a CUser object into a string.
	 * 
	 * @return a string
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		return toXML();
	}

	/**
	 * Convert a CUser object into an XML string.
	 * 
	 * @return an XML string
	 */
	public String toXML() {
		String xml = "\n<esup-user>";
		xml = xml + "\n<id>" + getId() + "</id>";
		xml = xml + "\n<display-name>" + EsupXMLEscaper.escape(getDisplayName()) + "</display-name>";
		xml = xml + "\n<email>" + EsupXMLEscaper.escape(getEmail()) + "</email>";
		xml = xml + "\n</esup-user>";
		return xml;
	}

	/**
	 * Default constructor, needed by JDO.
	 */
	public CUser() {
		// needed by JDO
	}

}