/*
ESUP-portail is a french academic project developed under the GPL (General Public License) augmented according to the following :
A binary or source file developped by ESUP-portail can be used or compiled with products under Apache license.
The official english text of the GPL can be found here : http://www.gnu.org/licenses/gpl.html .
A non official french translation can be found here : http://www.linux-France.org/article/these/gpl.html .
The different kinds of licenses governing the products developed by the Apache foundation can be found here : http://www.apache.org/licenses .
It follows that you can as well as use download contents as well modify and redistribute them provided you respect the GPL terms.
Downloading and using such contents do not provide any guaranty.
Be sure that you have well understood the terms of the license before using the contents it covers.
The ESUP-portail distribution includes the following distributions :
    * UPortal : 
      software developed by JA-SIG (Java Architecture - Special Interest Group)
      You can find the license page here : http://mis105.udel.edu/ja-sig/uportal/license.html
    * CAS :
      SSO solution developed by Yale University
      You can find the project page here : http://www.yale.edu/tp/auth
    * Cocoon :
      XML framework distributed by the Apache foundation under Apache license;
      Please find the full text here : http://cocoon.apache.org/2.1/license.html
    * Mod_dav:
      A DAV module for Apache web server
      You can find the project page here : http://www.webdav.org/mod_dav
    * IMP :
      webmail from Horde application framework
      You can find the project page here : http://www.horde.org
    * . To be completed
*/
package org.esupportail.portal.channels.CSMS.changeProfile;

import java.util.Hashtable;

import javax.naming.Context;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.DirContext;
import javax.naming.directory.InitialDirContext;
import javax.naming.directory.ModificationItem;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;

import org.esupportail.portal.channels.CSMS.exception.ExceptionAccesLdap;
import org.esupportail.portal.channels.CSMS.exception.ExceptionConfigurationCanal;
import org.esupportail.portal.channels.CSMS.properties.ParseProprietesCanal;
import org.esupportail.portal.channels.CSMS.properties.ProprietesCanal;
import org.jasig.portal.services.LogService;

/**
 * Id: AccesLdap.java,v 1.0 10 mai 2004
 * Copyright (c) 2004 Esup Portail (www.esup-portail.org)
 * Classes: AccesLdap
 * Original Author: Yohan Colmant
 * Permet d'interroger l'annuaire LDAP
 */
public class AccesLdap {
	
	/**
	 * Mode de contact par courriel
	 */
	public static String MAIL = "courriel";
	
	/**
	 * Mode de contact par sms
	 */
	public static String SMS = "sms";
	
	/**
	 * Mode de contact par courriel et sms
	 */
	public static String MAIL_AND_SMS = "both";
	
	
	/**
	 * URL de l'annuaire LDAP
	 */
	private static String ldapUrl = null;
	
	/**
	 * Compte d'acces a l'annuaire LDAP
	 */
	private static String ldaplogin = null;

	/**
	 * Mot de passe d'acces a l'annuaire LDAP
	 */
	private static String ldapPassword = null;
	
	/**
	 * Groupe des people de l'annuaire LDAP
	 */
	private static String ldapPeople = null;
	
	
	/**
	 * Contexte d'interrogation de l'annuaire
	 */
	private static DirContext context = null;
	
	
	
	/**
	 * Initialise le contexte LDAP
	 * @throws NamingException
	 * @throws ExceptionConfigurationCanal
	 */
	private static void initContext() throws NamingException, ExceptionConfigurationCanal {
		ProprietesCanal prop = ParseProprietesCanal.getInstance();
		 
		ldapUrl = prop.getLdapUrl();
		ldaplogin = prop.getLoginLdap();
		ldapPassword = prop.getPasswordLdap();
		ldapPeople = prop.getLdapPeople();		
		
		Hashtable env = new Hashtable();
		
		env.put(Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.ldap.LdapCtxFactory");
		env.put(Context.PROVIDER_URL, ldapUrl);		
		
		env.put(Context.SECURITY_AUTHENTICATION, "simple");
		env.put(Context.SECURITY_PRINCIPAL, ldaplogin);
		env.put(Context.SECURITY_CREDENTIALS, ldapPassword);		
		
		context = new InitialDirContext(env);						
	}
	
	
	/**
	 * retourne le mode de contact pour l'etudiant dont le login est donne en parametre
	 * @param login le login de l'etudiant
	 * @return le mode de contact: MAIL, SMS ou MAIL_AND_SMS ou code d'activation
	 * @throws ExceptionAccesLdap
	 * @throws ExceptionConfigurationCanal
	 */
	public static String getContactMode(String login) throws ExceptionAccesLdap, ExceptionConfigurationCanal {
		
		try {
			if (context==null)
				initContext();
				
			String mode = "";
		
			SearchControls constraints = new SearchControls();
			constraints.setSearchScope(SearchControls.OBJECT_SCOPE);
		
			String baseDN = "uid="+login+","+ldapPeople;
			String filtre="(objectClass=*)";
		
			NamingEnumeration ne = context.search(baseDN, filtre, constraints);

			// on regarde chaque element trouve
			while (ne.hasMoreElements()) {
				SearchResult searchRes = (SearchResult)ne.nextElement();
				Attributes atts =  searchRes.getAttributes();		     
				Attribute att = atts.get(ParseProprietesCanal.getInstance().getContactModeAttribute());
				if (att == null)
					return null;
				else {
					NamingEnumeration enum = att.getAll();
					mode = (String)enum.nextElement();
				}
			}		
			return mode;
		}
		catch(NamingException e) {
			LogService.log(LogService.ERROR, "AccessLdap::getContactMode: "+e);
			throw new ExceptionAccesLdap();
		}
	}
	
	
	
	/**
	 * Affecte le mode de contact pour l'etudiant dont le login est passe en parametre
	 * @param login le login de l'etudiant
	 * @param contactMode le mode de contact: MAIL, SMS ou MAIL_AND_SMS ou code d'activation
	 * @throws ExceptionAccesLdap
	 * @throws ExceptionConfigurationCanal
	 */
	public static void setContactMode(String login, String contactMode) throws ExceptionAccesLdap, ExceptionConfigurationCanal {
		
		try {
			if (context==null)
				initContext();
		
			// Specify the changes to make
			ModificationItem[] mods = new ModificationItem[1];
			
			// Replace the "uvhcContactMode" attribute with a new value
			mods[0] = new ModificationItem(DirContext.REPLACE_ATTRIBUTE,  new BasicAttribute(ParseProprietesCanal.getInstance().getContactModeAttribute(), contactMode));
			
			// Perform the requested modifications on the named object
			String baseDN = "uid="+login+","+ldapPeople;
			context.modifyAttributes(baseDN, mods);
		}
		catch(NamingException e) {
			LogService.log(LogService.ERROR, "AccessLdap::setContactMode: "+e);
			throw new ExceptionAccesLdap();
		}
	}
	
	
	
	/**
	 * retourne le numero de telephone de l'etudiant dont le login est donne en parametre
	 * @param login le login de l'etudiant
	 * @return le numero de telephone de l'etudiant
	 * @throws ExceptionAccesLdap
	 * @throws ExceptionConfigurationCanal
	 */
	public static String getPhoneNumber(String login) throws ExceptionAccesLdap, ExceptionConfigurationCanal {
		
		try {
			if (context==null)
				initContext();
		
			String mobile = "";
			
			SearchControls constraints = new SearchControls();
			constraints.setSearchScope(SearchControls.OBJECT_SCOPE);
			
			String baseDN = "uid="+login+","+ldapPeople;
			String filtre="(objectClass=*)";
			
			NamingEnumeration ne = context.search(baseDN, filtre, constraints);
			
			// 	on regarde chaque element trouve
			while (ne.hasMoreElements()) {
				SearchResult searchRes = (SearchResult)ne.nextElement();
				Attributes atts =  searchRes.getAttributes();		     
				Attribute att = atts.get(ParseProprietesCanal.getInstance().getMobileAttribute());
				if (att == null)
					return null;
				else {
					NamingEnumeration enum = att.getAll();
					mobile = (String)enum.nextElement();
				}	    
			}
		
			return phoneFromLdapFormat(mobile);
		}
		catch(NamingException e) {
			LogService.log(LogService.ERROR, "AccessLdap::getPhoneNumber: "+e);
			throw new ExceptionAccesLdap();
		}
	}
	
	
	
	/**
	 * Affecte le numero de telephone de l'etudiant dont le login est passe en parametre
	 * @param login le login de l'etudiant
	 * @param phoneNumber le numero de telephone de l'etudiant
	 * @throws ExceptionAccesLdap
	 * @throws ExceptionConfigurationCanal
	 */
	public static void setPhoneNumber(String login, String phoneNumber) throws ExceptionAccesLdap, ExceptionConfigurationCanal {
		
		try {
			if (context==null)
				initContext();
		
			// Specify the changes to make
			ModificationItem[] mods = new ModificationItem[1];

			// Replace the "uvhcContactMode" attribute with a new value
			mods[0] = new ModificationItem(DirContext.REPLACE_ATTRIBUTE,  new BasicAttribute(ParseProprietesCanal.getInstance().getMobileAttribute(), phoneToLdapFormat(phoneNumber)));

			// Perform the requested modifications on the named object
			String baseDN = "uid="+login+","+ldapPeople;
			context.modifyAttributes(baseDN, mods);
		}
		catch(NamingException e) {
			LogService.log(LogService.ERROR, "AccessLdap::setPhoneNumber: "+e);
			throw new ExceptionAccesLdap();
		}
	}
	
	
	
	/**
	 * Efface le numero de telephone de l'etudiant dont le login est passe en parametre
	 * @param login le login de l'etudiant
	 * @throws ExceptionAccesLdap
	 * @throws ExceptionConfigurationCanal
	 */
	public static void deletePhoneNumber(String login) throws ExceptionAccesLdap, ExceptionConfigurationCanal {
		
		try {
			if (context==null)
				initContext();
		
			// Specify the changes to make
			ModificationItem[] mods = new ModificationItem[1];
			
			// Replace the "uvhcContactMode" attribute with a new value
			mods[0] = new ModificationItem(DirContext.REMOVE_ATTRIBUTE,  new BasicAttribute(ParseProprietesCanal.getInstance().getMobileAttribute()));
			
			// Perform the requested modifications on the named object
			String baseDN = "uid="+login+","+ldapPeople;
			context.modifyAttributes(baseDN, mods);	
		}
		catch(NamingException e) {
			e.printStackTrace();
			LogService.log(LogService.ERROR, "AccessLdap::deletePhoneNumber: "+e);
			throw new ExceptionAccesLdap();
		}
	}
	
	
	
	/**
	 * Recupere l'adresse email d'un user donn
	 * @param login le login de la personne cite
	 * @return le mail de la personne
	 * @throws ExceptionAccesLdap
	 * @throws ExceptionConfigurationCanal
	 */
	public static String getMail(String login) throws ExceptionAccesLdap, ExceptionConfigurationCanal {
		
		try {
			if (context==null)
				initContext();
			
			// on recherche dans l'annuaire le nom de cet expediteur
			String baseDN = ldapPeople;
			String providerUrl = ldapUrl;
			String filtre = "";
			
			// on lance la recherche
			Hashtable env = new Hashtable();
			env.put(Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.ldap.LdapCtxFactory");
			env.put(Context.PROVIDER_URL, providerUrl);

			DirContext ctx = new InitialDirContext(env);

			SearchControls constraints = new SearchControls();
			constraints.setSearchScope(SearchControls.SUBTREE_SCOPE);			      
		      			
			filtre = "(uid="+login+")";
			NamingEnumeration ne = ctx.search(baseDN	, filtre, constraints);

			// on regarde chaque element trouve
			while (ne.hasMoreElements()) {
				SearchResult searchRes = (SearchResult)ne.nextElement();
				Attributes atts =  searchRes.getAttributes();		     
				Attribute att = atts.get(ParseProprietesCanal.getInstance().getMailAttribute());
				if (att==null)
					return null;
				NamingEnumeration enum = att.getAll();
				return (String)enum.nextElement();				
			}
			
			return null;
		}
		catch (NamingException e) {
			LogService.log(LogService.ERROR, "AccessLdap::getMail: "+e);
			throw new ExceptionAccesLdap();
		}				
	}
	
	

	/**
	 * Recupere le nom complet d'un user donn
	 * @param login le login de la personne cite
	 * @return le nom complet de la personne
	 * @throws ExceptionAccesLdap
	 * @throws ExceptionConfigurationCanal
	 */
	public static String getDisplayName(String login) throws ExceptionAccesLdap, ExceptionConfigurationCanal {
		
		try {
			if (context==null)
				initContext();
			
			// on recherche dans l'annuaire le nom de cet expediteur
			String baseDN = ldapPeople;
			String providerUrl = ldapUrl;
			String filtre = "";
			
			// on lance la recherche
			Hashtable env = new Hashtable();
			env.put(Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.ldap.LdapCtxFactory");
			env.put(Context.PROVIDER_URL, providerUrl);

			DirContext ctx = new InitialDirContext(env);

			SearchControls constraints = new SearchControls();
			constraints.setSearchScope(SearchControls.SUBTREE_SCOPE);			      
		      			
			filtre = "(uid="+login+")";
			NamingEnumeration ne = ctx.search(baseDN	, filtre, constraints);

			// on regarde chaque element trouve
			while (ne.hasMoreElements()) {
				SearchResult searchRes = (SearchResult)ne.nextElement();
				Attributes atts =  searchRes.getAttributes();		     
				Attribute att = atts.get(ParseProprietesCanal.getInstance().getDisplayNameAttribute());
				if (att==null)
					return login;
				NamingEnumeration enum = att.getAll();
				return (String)enum.nextElement();				
			}
			
			return null;
		}
		catch (NamingException e) {
			LogService.log(LogService.ERROR, "AccessLdap::getDisplayName: "+e);
			throw new ExceptionAccesLdap();
		}				
	}
	
	
	
	
	/**
	 * Transforme un numero de telephone du format standard au format LDAP
	 * @param phoneNumber numero de telephone au format 06********
	 * @return un numero de telephone au format +33 6 ** ** ** **
	 */
	private static String phoneToLdapFormat(String phoneNumber) {
		String newPhone = "+33 6 ";
		
		newPhone += phoneNumber.substring(2,4);
		newPhone += " ";
		
		newPhone += phoneNumber.substring(4,6);
		newPhone += " ";
		
		newPhone += phoneNumber.substring(6,8);
		newPhone += " ";
		
		newPhone += phoneNumber.substring(8,10);
		
		return newPhone;
	}
	
	
	/**
	 * Transforme un numero de telephone du format LDAP au format standard
	 * @param phoneNumber numero de telephone au format +33 6 ** ** ** **
	 * @return un numero de telephone au format 06********
	 */
	private static String phoneFromLdapFormat(String phoneNumber) {
		String newPhone = "0" + phoneNumber.substring(3);	
		newPhone = newPhone.replaceAll(" ","");		
		return newPhone;
	}
	
	
	
	
	
}
