/*
ESUP-portail is a french academic project developed under the GPL (General Public License) augmented according to the following :
A binary or source file developped by ESUP-portail can be used or compiled with products under Apache license.
The official english text of the GPL can be found here : http://www.gnu.org/licenses/gpl.html .
A non official french translation can be found here : http://www.linux-France.org/article/these/gpl.html .
The different kinds of licenses governing the products developed by the Apache foundation can be found here : http://www.apache.org/licenses .
It follows that you can as well as use download contents as well modify and redistribute them provided you respect the GPL terms.
Downloading and using such contents do not provide any guaranty.
Be sure that you have well understood the terms of the license before using the contents it covers.
The ESUP-portail distribution includes the following distributions :
    * UPortal :
      software developed by JA-SIG (Java Architecture - Special Interest Group)
      You can find the license page here : http://mis105.udel.edu/ja-sig/uportal/license.html
    * CAS :
      SSO solution developed by Yale University
      You can find the project page here : http://www.yale.edu/tp/auth
    * Cocoon :
      XML framework distributed by the Apache foundation under Apache license;
      Please find the full text here : http://cocoon.apache.org/2.1/license.html
    * Mod_dav:
      A DAV module for Apache web server
      You can find the project page here : http://www.webdav.org/mod_dav
    * IMP :
      webmail from Horde application framework
      You can find the project page here : http://www.horde.org
    * . To be completed
*/
package org.esupportail.portal.channels.CStockage.channelAction.classic.sharing;

import java.util.Vector;

import org.esupportail.portal.channels.CStockage.config.ChannelConfiguration;
import org.esupportail.portal.channels.CStockage.config.Space;
import org.esupportail.portal.channels.CStockage.exception.AclAccessException;
import org.esupportail.portal.channels.CStockage.exception.AclReadException;
import org.esupportail.portal.channels.CStockage.exception.AclWriteException;
import org.esupportail.portal.channels.CStockage.exception.DataBaseException;
import org.esupportail.portal.channels.CStockage.exception.NotSupportedAclException;
import org.esupportail.portal.channels.CStockage.exception.PropertiesException;
import org.esupportail.portal.channels.CStockage.provider.access.ServerAccess;
import org.esupportail.portal.utils.webdav.acl.EsupPermission;

/**
 * Id: SharingTool.java,v 1.0 13 janv. 2005
 * Copyright (c) 2005 Esup Portail (www.esup-portail.org)
 * Classes: SharingTool
 * Original Author: Yohan Colmant
 * This class is a tool used to manage the sharing actions
 */
public class SharingTool {

	
	/**
	 * The object used to access the database
	 */
	private DataBaseAccess dbAccess;
	

	/**
	 * The sharing directory properties of the current directory
	 */
	private DirectorySharingProperties currentDirectorySharingProperties;
	
	
	
	
	/**
	 * Constructor
	 * @throws PropertiesException
	 */
	public SharingTool() throws PropertiesException {
		dbAccess = new DataBaseAccess();		
	}
	

	
	/**
	 * @return Returns the currentDirectorySharingProperties.
	 */
	public DirectorySharingProperties getCurrentDirectorySharingProperties() {
		return currentDirectorySharingProperties;
	}
	/**
	 * @param currentDirectorySharingProperties The currentDirectorySharingProperties to set.
	 */
	public void setCurrentDirectorySharingProperties(DirectorySharingProperties currentDirectorySharingProperties) {
		this.currentDirectorySharingProperties = currentDirectorySharingProperties;
	}
	
	
	
	
	
	
	
	

	/**
	 * Add sharing properties for the user into the database
	 * @param serverUrl the url of the server
	 * @param path the resource path
	 * @param label the space's label
	 * @param user the target user
	 * @param ownerUserKey the key of the owner
	 * @param xmlSpace the space into an xml format
	 * @throws DataBaseException
	 */
	public void addStorageAvailableSpaceUser(Space currentSpace, String serverUrl, String path, String label, UserForSharing user, String ownerUserKey, String xmlSpace) throws DataBaseException, AclAccessException, AclWriteException, AclReadException, NotSupportedAclException, PropertiesException {
	
		////////////////////////
		// add into the database
		
		// the user
		// System.out.println("User -- "+user.getDisplayName()+" read:"+user.isReading()+" write:"+user.isWriting());
			
		// if we don't have this, we add into database
		dbAccess.addStorageAvailableSpace(serverUrl, path, label, user.getKey(), true, ownerUserKey, xmlSpace, user.isReading(), user.isWriting(), user.isManage());			
		
		
		/////////////////////
		// set the ACL rights
		if (currentSpace.managesAcl()) {			
			ServerAccess access = currentSpace.getServerAccessObject();
			String principal = currentSpace.getAclUserPrefix()+user.getKey();
			if (user.isWriting() && user.isReading()) {
				access.grant(path, principal, EsupPermission.READ);
				access.grant(path, principal, EsupPermission.WRITE);
			}
			else {
				if (user.isReading()) {
					access.grant(path, principal, EsupPermission.READ);
				}
			}
		}
	}
	
	
	
	

	/**
	 * Add sharing properties for the groups into the database
	 * @param serverUrl the url of the server
	 * @param path the resource path
	 * @param label the space's label
	 * @param group the target group
	 * @param ownerUserKey the key of the owner
	 * @param xmlSpace the space into an xml format
	 * @throws DataBaseException
	 */
	public void addStorageAvailableSpaceGroup(Space currentSpace, String serverUrl, String path, String label, GroupForSharing group, String ownerUserKey, String xmlSpace) throws DataBaseException, AclAccessException, AclWriteException, AclReadException, NotSupportedAclException, PropertiesException {
	
		////////////////////////
		// add into the database
		
		// the group
		//System.out.println("Group -- "+group.getDisplayName()+" read:"+group.isReading()+" write:"+group.isWriting());
			
		// if we don't have this, we add into database
		dbAccess.addStorageAvailableSpace(serverUrl, path, label, group.getKey(), false, ownerUserKey, xmlSpace, group.isReading(), group.isWriting(), group.isManage());
		
		
		
		/////////////////////
		// set the ACL rights
		if (currentSpace.managesAcl()) {
			Vector groupHierarchyKey = group.getGroupHierarchy();
			if (groupHierarchyKey!=null) {
				String principal = currentSpace.getAclGroupPrefix();
				if (currentSpace.getAclUportalGroup()!=null && !currentSpace.getAclUportalGroup().trim().equals("")) {
					principal = principal+currentSpace.getAclUportalGroup()+"/";
				}
				for (int j=0; j<groupHierarchyKey.size(); j++) {
					principal = principal + groupHierarchyKey.elementAt(j);
					if (j!=groupHierarchyKey.size()-1)
						principal = principal + "/";
				}
				
				ServerAccess access = currentSpace.getServerAccessObject();
				if (group.isWriting() && group.isReading()) {
					access.grant(path, principal, EsupPermission.READ);
					access.grant(path, principal, EsupPermission.WRITE);
				}
				else {
					if (group.isReading()) {
						access.grant(path, principal, EsupPermission.READ);
					}
				}
			}
		}
		
	}
	
	
	


	
	
	

	/**
	 * Delete sharing properties for users into the database
	 * @param currentSpace the current space
	 * @param user we want to delete preferences
	 * @throws DataBaseException
	 */
	public void delStorageAvailableSpaceForUser(Space currentSpace, UserForSharing user) throws DataBaseException, AclAccessException, AclWriteException, AclReadException, NotSupportedAclException, PropertiesException {

		// delete from the database
		String id = user.getIdOfSharing();
		dbAccess.delStorageUsersPreferences(id);
		dbAccess.delStorageAvailableSpace(id);		
		
		// remove the ACL
		if (currentSpace.managesAcl()) {
			String principal = currentSpace.getAclUserPrefix()+user.getKey();
			currentSpace.getServerAccessObject().revoke(currentSpace.getPath(), principal, EsupPermission.READ);
			currentSpace.getServerAccessObject().revoke(currentSpace.getPath(), principal, EsupPermission.WRITE);
			//TODO 
			currentSpace.getServerAccessObject().revoke(currentSpace.getPath(), principal, EsupPermission.READ_ACL);
			currentSpace.getServerAccessObject().revoke(currentSpace.getPath(), principal, EsupPermission.WRITE_ACL);
		}
	}
	
	
	
	


	/**
	 * Delete sharing properties for groups into the database
	 * @param currentSpace the current space
	 * @param group the group we want to delete sharing
	 * @throws DataBaseException
	 * @throws AclAccessException
	 * @throws AclWriteException
	 * @throws AclReadException
	 * @throws NotSupportedAclException
	 * @throws PropertiesException
	 */
	public void delStorageAvailableSpaceForGroup(Space currentSpace, GroupForSharing group) throws DataBaseException, AclAccessException, AclWriteException, AclReadException, NotSupportedAclException, PropertiesException {

		// delete from the database
		String id = group.getIdOfSharing();		
		dbAccess.delStorageUsersPreferences(id);
		dbAccess.delStorageAvailableSpace(id);
		
		// remove the ACL
		if (currentSpace.managesAcl()) {
			Vector groupHierarchyKey = group.getGroupHierarchy();
			if (groupHierarchyKey!=null) {
				String principal = currentSpace.getAclGroupPrefix();
				if (currentSpace.getAclUportalGroup()!=null && !currentSpace.getAclUportalGroup().trim().equals("")) {
					principal = principal+currentSpace.getAclUportalGroup()+"/";
				}
				for (int j=0; j<groupHierarchyKey.size(); j++) {
					principal = principal + groupHierarchyKey.elementAt(j);
					if (j!=groupHierarchyKey.size()-1)
						principal = principal + "/";
				}
				
				currentSpace.getServerAccessObject().revoke(currentSpace.getPath(), principal, EsupPermission.READ);
				currentSpace.getServerAccessObject().revoke(currentSpace.getPath(), principal, EsupPermission.WRITE);
				//TODO
				currentSpace.getServerAccessObject().revoke(currentSpace.getPath(), principal, EsupPermission.READ_ACL);
				currentSpace.getServerAccessObject().revoke(currentSpace.getPath(), principal, EsupPermission.WRITE_ACL);
			}
		}
		
	}
	
	

	
	

	/**
	 * Get the users for whose the resource is shared
	 * @param serverUrl the url of the server
	 * @param path the path of the resource
	 * @return a vector. Each element is an array of [id, label, targetKey, read:'T'|'F', write:'T'|'F', manage:'T'|'F']
	 * @throws DataBaseException
	 */
	public Vector getTargetsOfAvailableSpaceForUser(String serverUrl, String path) throws DataBaseException {

		// do the request
		Vector result = dbAccess.getTargetsOfAvailableSpace(serverUrl, path, true);
		
		// return the result
		return result;
	}
	
	


	/**
	 * Get the groups for whose the resource is shared
	 * @param serverUrl the url of the server
	 * @param path the path of the resource
	 * @return a vector. Each element is an array of [id, label, targetKey, targetIsUser:'T'|'F', read:'T'|'F', write:'T'|'F', manage:'T'|'F']
	 * @throws DataBaseException
	 */
	public Vector getTargetsOfAvailableSpaceForGroup(String serverUrl, String path) throws DataBaseException {

		// do the request
		Vector result = dbAccess.getTargetsOfAvailableSpace(serverUrl, path, false);
		
		// return the result
		return result;
	}
	
	
	
	
	
	

	/**
	 * Update the label of the sharing properties into the database for users
	 * @param newLabel the new label of the shared spaces
	 * @param serverUrl the url of the server
	 * @param path the resource path
	 * @throws DataBaseException
	 */
	public void updateLabelAvailableSpace(String newLabel, String serverUrl, String path) throws DataBaseException {

		// update the database
		dbAccess.updateLabelAvailableSpace(newLabel, serverUrl, path);

	}
	
	

	
	
	
	

	
	/**
	 * Update the "reading" and "writing" attributes for a specifical sharing for a user
	 * @param currentSpace the current space
	 * @param serverUrl the url of the server
	 * @param path the resource path
	 * @param user the target user
	 * @throws DataBaseException
	 */
	public void updateReadAndWriteAndManageSpaceUser(Space currentSpace, String serverUrl, String path, UserForSharing user)throws DataBaseException, AclAccessException, AclWriteException, AclReadException, NotSupportedAclException, PropertiesException {

		// update the database		
		dbAccess.updateReadAndWriteAndManageSpaceUser(user.isReading(), user.isWriting(), user.isManage(), serverUrl, path, user.getKey(), true);

		// update the ACL
		if (currentSpace.managesAcl()) {
			String principal = currentSpace.getAclUserPrefix()+user.getKey();
			ServerAccess access = currentSpace.getServerAccessObject();
			access.revoke(currentSpace.getPath(), principal, EsupPermission.READ);
			access.revoke(currentSpace.getPath(), principal, EsupPermission.WRITE);
			if (user.isReading() && user.isWriting() && user.isManage()) {
				access.grant(path, principal, EsupPermission.READ);
				access.grant(path, principal, EsupPermission.WRITE);
				// TODO
				access.grant(path, principal, EsupPermission.READ_ACL);
				access.grant(path, principal, EsupPermission.WRITE_ACL);
			}
			else {
				if (user.isWriting() && user.isReading()) {
					access.grant(path, principal, EsupPermission.READ);
					access.grant(path, principal, EsupPermission.WRITE);
				}
				else {
					if (user.isReading()) {
						access.grant(path, principal, EsupPermission.READ);
					}
				}
			}
		}
	}

	
	/**
	 * Update the "reading" and "writing" attributes for a specifical sharing for a group
	 * @param currentSpace the current space
	 * @param serverUrl the url of the server
	 * @param path the resource path
	 * @param group the target group
	 * @throws DataBaseException
	 */
	public void updateReadAndWriteAndManageSpaceGroup(Space currentSpace, String serverUrl, String path, GroupForSharing group)throws DataBaseException, AclAccessException, AclWriteException, AclReadException, NotSupportedAclException, PropertiesException {

		// update the database
		dbAccess.updateReadAndWriteAndManageSpaceUser(group.isReading(), group.isWriting(), group.isManage(), serverUrl, path, group.getKey(), false);

		// update the ACL
		if (currentSpace.managesAcl()) {
			Vector groupHierarchyKey = group.getGroupHierarchy();
			if (groupHierarchyKey!=null) {
				String principal = currentSpace.getAclGroupPrefix();
				if (currentSpace.getAclUportalGroup()!=null && !currentSpace.getAclUportalGroup().trim().equals("")) {
					principal = principal+currentSpace.getAclUportalGroup()+"/";
				}
				for (int j=0; j<groupHierarchyKey.size(); j++) {
					principal = principal + groupHierarchyKey.elementAt(j);
					if (j!=groupHierarchyKey.size()-1)
						principal = principal + "/";
				}
				ServerAccess access = currentSpace.getServerAccessObject();
				access.revoke(currentSpace.getPath(), principal, EsupPermission.READ);
				access.revoke(currentSpace.getPath(), principal, EsupPermission.WRITE);
				if (group.isReading() && group.isWriting() && group.isManage()) {
					access.grant(path, principal, EsupPermission.READ);
					access.grant(path, principal, EsupPermission.WRITE);
					// TODO
					access.grant(path, principal, EsupPermission.READ_ACL);
					access.grant(path, principal, EsupPermission.WRITE_ACL);
				}
				else {
					if (group.isWriting() && group.isReading()) {
						access.grant(path, principal, EsupPermission.READ);
						access.grant(path, principal, EsupPermission.WRITE);
					}
					else {
						if (group.isReading()) {
							access.grant(path, principal, EsupPermission.READ);
						}
					}
				}
			}
		}
		
	}
	
	
	
	
	
	
	

	/**
	 * Get the available spaces for a user or a group
	 * @param targetKey the key of the target
	 * @param targetIsUser if we get spaces for user, the value is true, for a group, the value is false
	 * @param userPortalLogin the login of the user in the portal
	 * @return a vector. Each element is an array of [id, url, path, label, owner, xmlSpace, read:'T'|'F', write:'T'|'F']
	 * @throws PropertiesException
	 * @throws DataBaseException
	 */
	private Vector getAvailableSpacesForTarget(String targetKey, boolean targetIsUser, String userPortalLogin) throws PropertiesException, DataBaseException {

		// update the database
		Vector dbResult = dbAccess.getAvailableSpacesForTarget(targetKey, targetIsUser);

		// we check the result
		Vector result = new Vector();
		
		// for each element
		for (int i=0; i<dbResult.size(); i++) {
			
			String[] line = (String[])dbResult.elementAt(i);

			// the id in database
			String id = line[0];			 

			// the url of the server
			String url = line[1];			 

			// the path
			String path = line[2];			
			
			// the label
			String label = line[3];			

			// the owner
			String owner = line[4];			

			// the space
			String xmlSpace = line[5];			
			SpaceTool tool = new SpaceTool(xmlSpace);
			Space space = tool.getSpace();
			
			// the reading
			String r = line[6];
			boolean read = true;
			if (r.equals("F"))
				read = false;
			
			// the writing
			String w = line[7];
			boolean write = true;
			if (w.equals("F"))
				write = false;

			// the management
			String m = line[8];
			boolean manage = true;
			if (m.equals("F"))
				manage = false;
			
			// set the value into the space
			space.setIdInDatabase(id);
			space.setSharedForTheUser(targetIsUser);
			//space.setKey(url+"/"+path);
			space.setKey("db_"+id);
			space.setUrl(url);
			space.setPath(path);
			space.setLabel(label);
			space.setCanWrite(write);
			space.setOwner(owner);
			space.setTargetOfSharing(targetKey);
			space.setPersonnalSpace(true);
			space.setLogin(userPortalLogin);
			//space.setLoginIsFromConfigurationFile(false);
			if (manage==true) {
				space.setEnableSharing(manage+"");
			}
			
			// the server access class
			String serverType = space.getServerType();
			if (serverType!=null) {
				String serverClass = (String)ChannelConfiguration.getInstance().getServerAccess().get(serverType);
				space.setServerAccessClass(serverClass);
			}
			
			// the channel action type
			String actionType = space.getActionType();
			if (actionType!=null) {
				String actionClass = (String)ChannelConfiguration.getInstance().getChannelActions().get(actionType);
				space.setChannelActionClass(actionClass);				
			}	
			
			// add space to the vector
			result.add(space);
		}
		
		// return the result
		return result;
	}	
	
	

	/**
	 * Get the available spaces for a user
	 * @param targetKey the key of the target
	 * @param userPortalLogin the login of the user in the portal
	 * @return a vector. Each element is an array of [url, path, label, owner, xmlSpace, read:'T'|'F', write:'T'|'F']
	 * @throws DataBaseException
	 */
	public Vector getAvailableSpacesForUser(String targetKey, String userPortalLogin) throws DataBaseException, PropertiesException {
		return getAvailableSpacesForTarget(targetKey, true, userPortalLogin);
	}	


	/**
	 * Get the available spaces for a group
	 * @param targetKey the key of the target
	 * @return a vector. Each element is an array of [url, path, label, owner, xmlSpace, read:'T'|'F', write:'T'|'F']
	 * @throws DataBaseException
	 */
	public Vector getAvailableSpacesForGroup(String targetKey, String userPortalLogin) throws DataBaseException, PropertiesException {
		return getAvailableSpacesForTarget(targetKey, false, userPortalLogin);
	}	
	
	
	
	
	
	
	
	
	

	

	

	/**
	 * Add a user preference into the database from a user
	 * @param id the id of the sharement in the other database
	 * @param userKey the login of the user in the portal
	 * @throws DataBaseException
	 */
	public void addStorageUsersPreferences(String id, String userKey) throws DataBaseException {
		
		// set the database
		dbAccess.addStorageUsersPreferences(id, userKey);
			
	}

	

	
	/**
	 * Get the list of ids of spaces which has already been selected by the current user from users table 
	 * @param target the user or group target key
	 * @return a Vector
	 * @throws DataBaseException
	 */
	public Vector getIdOfSpaceFromPreferencesUser(String target) throws DataBaseException {

		// do the request
		Vector v = dbAccess.getIdOfSpaceFromPreferences(target);
		
		// return the result
		return v;
	}
		


	/**
	 * Delete a user preference from the database
	 * @param id the id of the sharement in the other database
	 * @param userKey the key of the user
	 * @throws DataBaseException
	 */
	public void delStorageUsersPreferencesForUser(String id, String userKey) throws DataBaseException {

		// set the database
		dbAccess.delStorageUsersPreferencesForUser(id, userKey);
		
	}
	
	

	/**
	 * Delete users preferences for a shared space from the database
	 * @param id the id of the sharement in the other database
	 * @throws DataBaseException
	 */
	public void delStorageUsersPreferences(String id) throws DataBaseException {

		// set the database
		dbAccess.delStorageUsersPreferences(id);
		
	}
	
	
	
	


	/**
	 * Delete sharing properties for url and path into the database
	 * @param url the url of the space
	 * @param path the path of the space
	 * @throws DataBaseException
	 */
	public void delStorageAvailableSpaceForSpecifiedPath(String url, String path) throws DataBaseException {

		// get ids of spaces with the specified url and path
		Vector ids = dbAccess.getIdOfSpacesWithPath(url, path);
		
		// for each id, we delete from the 2 tables in database
		for (int i=0; i<ids.size(); i++) {
			String id = (String)ids.elementAt(i);
			
			// delete from the database			
			dbAccess.delStorageUsersPreferences(id);
			dbAccess.delStorageAvailableSpace(id);
		}
			
	}
	



	/**
	 * Update the path of shared spaces
	 * @param url the url of the space
	 * @param oldPath the old path of the space
	 * @param newPath the new path of the space
	 * @return the result of the query
	 * @throws DataBaseException
	 */
	public void updatePathOfSharing(String url, String oldPath, String newPath) throws DataBaseException {

		// get ids of spaces with the specified url and path
		dbAccess.updatePathOfSharing(url, oldPath, newPath);
			
	}
	
	
	
}
