/*
ESUP-portail is a french academic project developed under the GPL (General Public License) augmented according to the following :
A binary or source file developped by ESUP-portail can be used or compiled with products under Apache license.
The official english text of the GPL can be found here : http://www.gnu.org/licenses/gpl.html .
A non official french translation can be found here : http://www.linux-France.org/article/these/gpl.html .
The different kinds of licenses governing the products developed by the Apache foundation can be found here : http://www.apache.org/licenses .
It follows that you can as well as use download contents as well modify and redistribute them provided you respect the GPL terms.
Downloading and using such contents do not provide any guaranty.
Be sure that you have well understood the terms of the license before using the contents it covers.
The ESUP-portail distribution includes the following distributions :
    * UPortal : 
      software developed by JA-SIG (Java Architecture - Special Interest Group)
      You can find the license page here : http://mis105.udel.edu/ja-sig/uportal/license.html
    * CAS :
      SSO solution developed by Yale University
      You can find the project page here : http://www.yale.edu/tp/auth
    * Cocoon :
      XML framework distributed by the Apache foundation under Apache license;
      Please find the full text here : http://cocoon.apache.org/2.1/license.html
    * Mod_dav:
      A DAV module for Apache web server
      You can find the project page here : http://www.webdav.org/mod_dav
    * IMP :
      webmail from Horde application framework
      You can find the project page here : http://www.horde.org
    * . To be completed
*/
package org.esupportail.portal.channels.CStockage.channelAction.injac.metadata;


import java.io.StringReader;
import java.text.DateFormat;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.Vector;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xerces.jaxp.DocumentBuilderFactoryImpl;
import org.esupportail.portal.channels.CStockage.channelAction.injac.metadata.parse.RequiredMetaData;
import org.esupportail.portal.channels.CStockage.config.Space;
import org.esupportail.portal.channels.CStockage.exception.ApplicationException;
import org.esupportail.portal.channels.CStockage.exception.PropertiesException;
import org.esupportail.portal.channels.CStockage.exception.ServerException;
import org.esupportail.portal.channels.CStockage.spacesPersonalization.target.GroupForPersonalization;
import org.esupportail.portal.channels.CStockage.spacesPersonalization.target.UserForPersonalization;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

/**
 * Id: MetaDataManager.java,v 1.0 30 sept. 2004
 * Copyright (c) 2004 Esup Portail (www.esup-portail.org)
 * Classes: MetaDataManager
 * Original Author: Yohan Colmant
 * A tool metadata manager
 */
public class MetaDataManager {


	/**
	 * Logger object
	 */
	protected static final Log log = LogFactory.getLog(MetaDataManager.class);
	
	
	
	
	/**
	 * Get the owner of a document
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @return the type of the resource
	 * @throws ServerException
	 */
	public static String getDocumentOwner(Space currentSpace, String path) throws ServerException {				
		String owner = currentSpace.getServerAccessObject().getProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.DOCUMENT_OWNER);		
		if (owner==null)
			owner = "";

		// log
		if (log.isDebugEnabled()){
			log.debug("getDocumentOwner"+" :: owner = "+owner);
		}
		
		return owner;	
	}
	
	

	/**
	 * Set the owner of a document
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @param value the MD value 
	 * @return true if well done, else false
	 * @throws ServerException
	 */
	public static boolean setUserAsDocumentOwner(Space currentSpace, String path, String value) throws ServerException {				
		
		boolean result = currentSpace.getServerAccessObject().setProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.DOCUMENT_OWNER, value);
		
		// log
		if (log.isDebugEnabled()){
			log.debug("setUserAsDocumentOwner"+" :: result = "+result);
		}
		
		return 	result;			
	}
	
	
	
	/**
	 * Get the directory type (document, space, ... ?)
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @return the type of the resource
	 * @throws ServerException
	 */
	public static String getDirectoryType(Space currentSpace, String path) throws ServerException {				
		
		String property = currentSpace.getServerAccessObject().getProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.INJAC_TYPE);
		
		// log
		if (log.isDebugEnabled()){
			log.debug("getDirectoryType"+" :: property = "+property);
		}
		
		return property;
	}
	
	
	/**
	 * Get the publication date begin
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @return the date, null if no date
	 * @throws ServerException
	 */
	public static Date getPublicationDateBegin(Space currentSpace, String path) throws ServerException {				
		String date = currentSpace.getServerAccessObject().getProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.PUBLICATION_DATE_BEGIN);
		
		Date dateResult = null;
		
		if (date==null || date.equals(""))
			dateResult=null;
		
		try {
			DateFormat df = DateFormat.getDateInstance(DateFormat.SHORT);
			Date d = df.parse(date);
		
			dateResult=d;
		}
		catch(ParseException e) {
			log.error("getPublicationDateBegin"+" :: "+e);
			dateResult=null;
		}
		
		// log
		if (log.isDebugEnabled()){
			log.debug("getPublicationDateBegin"+" :: dateResult = "+dateResult);
		}
		
		return dateResult;
	}
	
	/**
	 * Get the publication date end
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @return the date, null if no date
	 * @throws ServerException
	 */
	public static Date getPublicationDateEnd(Space currentSpace, String path) throws ServerException {				
		String date = currentSpace.getServerAccessObject().getProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.PUBLICATION_DATE_END);
		
		Date dateResult = null;
		
		if (date==null || date.equals(""))
			dateResult = null;
		
		try {
			DateFormat df = DateFormat.getDateInstance(DateFormat.SHORT);
			Date d = df.parse(date);
		
			Calendar cal = Calendar.getInstance();
			cal.setTime(d);
			
			cal.add(Calendar.DATE, 1);
			cal.add(Calendar.SECOND, -1);
			
			dateResult = cal.getTime();
		}
		catch(ParseException e) {
			log.error("getPublicationDateEnd"+" :: "+e);
			dateResult = null;
		}
		
		// log
		if (log.isDebugEnabled()){
			log.debug("getPublicationDateEnd"+" :: dateResult = "+dateResult);
		}
		
		return dateResult;
	}
	
	
	


	/**
	 * Set the publication date begin
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @param date the date value
	 * @return true if well done
	 * @throws ServerException
	 */
	public static boolean setPublicationDateBegin(Space currentSpace, String path, String date) throws ServerException {		
		
		boolean wellDone = currentSpace.getServerAccessObject().setProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.PUBLICATION_DATE_BEGIN, date);
		
		// log
		if (log.isDebugEnabled()){
			log.debug("setPublicationDateBegin"+" :: wellDone = "+wellDone);
		}
		
		return wellDone;
	}
	
	



	/**
	 * Set the publication date end
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @param date the date value
	 * @return true if well done
	 * @throws ServerException
	 */
	public static boolean setPublicationDateEnd(Space currentSpace, String path, String date) throws ServerException {		
		
		boolean wellDone = currentSpace.getServerAccessObject().setProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.PUBLICATION_DATE_END, date);
		
		// log
		if (log.isDebugEnabled()){
			log.debug("setPublicationDateEnd"+" :: wellDone = "+wellDone);
		}
		
		return wellDone;
	}
	
	
	
	/**
	 * Get the refuse message
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @return the type of the resource
	 * @throws ServerException
	 */
	public static String getRejectCause(Space currentSpace, String path) throws ServerException {		
		
		String property = currentSpace.getServerAccessObject().getProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.REJECT_CAUSE);
		
		// log
		if (log.isDebugEnabled()){
			log.debug("getRejectCause"+" :: property = "+property);
		}
		
		return property;
	}
	
	
	


	/**
	 * Set the refuse message
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @param cause the metadata value
	 * @return true if well done
	 * @throws ServerException
	 */
	public static boolean setRejectCause(Space currentSpace, String path, String cause) throws ServerException {		
		
		boolean wellDone = currentSpace.getServerAccessObject().setProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.REJECT_CAUSE, cause);
		
		// log
		if (log.isDebugEnabled()){
			log.debug("setRejectCause"+" :: wellDone = "+wellDone);
		}
		
		return wellDone;
	}
	
	
	
	/**
	 * Get the document state: published, refused, waiting, expired...
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @return the type of the resource
	 * @throws ServerException
	 */
	public static String getDocumentState(Space currentSpace, String path) throws ServerException {
		
		Date currentDate = Calendar.getInstance().getTime();				
		
		String state = currentSpace.getServerAccessObject().getProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.DOCUMENT_STATE);
		
		if (state==null)
			return MetaDataConstants.DOCUMENT_EXPIRED_STATE;
		
		if (state.equals(MetaDataConstants.DOCUMENT_PUBLISHED_STATE) ) {			
			
			Date publicationDateBegin = MetaDataManager.getPublicationDateBegin(currentSpace, path);
			Date publicationDateEnd = MetaDataManager.getPublicationDateEnd(currentSpace, path);
			
			// published
			if (publicationDateBegin==null || publicationDateEnd==null)
				return MetaDataConstants.DOCUMENT_PUBLISHED_STATE;
			
			if ((currentDate.after(publicationDateBegin) && currentDate.before(publicationDateEnd)) || currentDate.equals(publicationDateBegin) || currentDate.equals(publicationDateEnd) )
				return MetaDataConstants.DOCUMENT_PUBLISHED_STATE;
			
			// expired
			else return MetaDataConstants.DOCUMENT_EXPIRED_STATE;
			
		}
		
		// log
		if (log.isDebugEnabled()){
			log.debug("getDocumentState"+" :: state = "+state);
		}
		
		
		return state;
		
	}
	
	
	

	/**
	 * Set the injac-type metadata of the directory specified by its path to "space" value
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @return true if well done
	 * @throws ServerException
	 */
	public static boolean setDirectoryAsSpace(Space currentSpace, String path) throws ServerException {		
		
		boolean wellDone = currentSpace.getServerAccessObject().setProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.INJAC_TYPE, MetaDataConstants.INJAC_TYPE_SPACE);
		
		// log
		if (log.isDebugEnabled()){
			log.debug("setDirectoryAsSpace"+" :: wellDone = "+wellDone);
		}
		
		return wellDone;
	}
	
	
	/**
	 * Set the injac-type metadata of the directory specified by its path to "document" value, and document-state to "waiting"
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @return true if well done
	 * @throws ServerException
	 */
	public static boolean setDirectoryAsWaitingDocument(Space currentSpace, String path) throws ServerException {		
		
		boolean b1 = currentSpace.getServerAccessObject().setProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.INJAC_TYPE, MetaDataConstants.INJAC_TYPE_DOCUMENT);
		boolean b2 = currentSpace.getServerAccessObject().setProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.DOCUMENT_STATE, MetaDataConstants.DOCUMENT_WAITING_STATE);
		
		boolean wellDone = b1 && b2;
		
		// log
		if (log.isDebugEnabled()){
			log.debug("setDirectoryAsWaitingDocument"+" :: wellDone = "+wellDone);
		}
		
		
		return wellDone;		
	}
	
	
	/**
	 * Set the injac-type metadata of the directory specified by its path to "document" value, and document-state to "published"
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @return true if well done
	 * @throws ServerException
	 */
	public static boolean setDirectoryAsPublishedDocument(Space currentSpace, String path) throws ServerException {		
		
		boolean b1 = currentSpace.getServerAccessObject().setProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.INJAC_TYPE, MetaDataConstants.INJAC_TYPE_DOCUMENT);
		boolean b2 = currentSpace.getServerAccessObject().setProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.DOCUMENT_STATE, MetaDataConstants.DOCUMENT_PUBLISHED_STATE);
		
		boolean wellDone = b1 && b2;
		
		// log
		if (log.isDebugEnabled()){
			log.debug("setDirectoryAsPublishedDocument"+" :: wellDone = "+wellDone);
		}
		
		
		return wellDone;		
	}
	

	/**
	 * Set the injac-type metadata of the directory specified by its path to "document" value, and document-state to "refused"
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @return true if well done
	 * @throws ServerException
	 */
	public static boolean setDirectoryAsRejectedDocument(Space currentSpace, String path) throws ServerException {		
		
		boolean b1 = currentSpace.getServerAccessObject().setProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.INJAC_TYPE, MetaDataConstants.INJAC_TYPE_DOCUMENT);
		boolean b2 = currentSpace.getServerAccessObject().setProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.DOCUMENT_STATE, MetaDataConstants.DOCUMENT_REFUSED_STATE);
		
		boolean wellDone = b1 && b2;
		
		// log
		if (log.isDebugEnabled()){
			log.debug("setDirectoryAsRejectedDocument"+" :: wellDone = "+wellDone);
		}
		
		
		return wellDone;		
	}
	
	/**
	 * Set the metadata set by manager on a space
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @param smp object containing all properties
	 * @return true if well done, else false
	 * @throws ServerException
	 * @throws ApplicationException
	 * @throws PropertiesException
	 */
	public static boolean setMetadataOnSpace(Space currentSpace, String path, InjacSpaceManagementProperties smp) throws ServerException, ApplicationException, PropertiesException {		
		
		String desc = smp.getDescription();
		desc = desc.replaceAll("&","&amp;");
		desc = desc.replaceAll("\"","&#148;");
		
		String title = smp.getTitle();
		title = title.replaceAll("&","&amp;");
		title = title.replaceAll("\"","&#148;");
		
		boolean b1 = currentSpace.getServerAccessObject().setProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.DESCRIPTION, desc);
		boolean b2 = true;
		if (!title.trim().equals(""))
			b2 = currentSpace.getServerAccessObject().setProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.TITLE, title);
		
		boolean b3 = currentSpace.getServerAccessObject().setProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.DOCUMENT_INJAC_METADATA, smp.getMetadataProfile());			
		boolean b4 = currentSpace.getServerAccessObject().setProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.RENDERING_TYPE, smp.getSelectedRenderingType());			
		boolean b5 = currentSpace.getServerAccessObject().setProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.WRITERS, smp.getMetadataWritersOnSpace());			
		boolean b6 = currentSpace.getServerAccessObject().setProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.EDITORS, smp.getMetadataEditorsOnSpace());			
		boolean b7 = currentSpace.getServerAccessObject().setProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.MANAGERS, smp.getMetadataManagersOnSpace());
		
		boolean wellDone = b1 && b2 && b3 && b4 && b5 && b6 && b7;
		
		// log
		if (log.isDebugEnabled()){
			log.debug("setMetadataOnSpace"+" :: wellDone = "+wellDone);
		}
		
		
		return wellDone;		
	}
	
	
	
	/**
	 * Set the metadata set by user on a document
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @param metadataToSet object containing all properties
	 * @return true if well done, else false
	 * @throws ServerException 
	 */
	public static boolean setMetadataOnDocument(Space currentSpace, String path, ArrayList metadataToSet) throws ServerException {		
		
		boolean wellDone = true;
		
		for (int i=0; i<metadataToSet.size(); i++) {
			
			RequiredMetaData metadata = (RequiredMetaData)metadataToSet.get(i);
			
			if (metadata.isModifiable()) {
				String metadataName = metadata.getName();
				String metadataValue = metadata.getValue();
				if (metadataValue!=null && !metadataValue.trim().equals("")) {
					metadataValue = metadataValue.replaceAll("&","&amp;");
					metadataValue = metadataValue.replaceAll("\"","&#148;");
					wellDone = wellDone && currentSpace.getServerAccessObject().setProperty(currentSpace.getMetadataNamespace(), path, metadataName, metadataValue);
				}
			}						
		}
		
		// log
		if (log.isDebugEnabled()){
			log.debug("setMetadataOnDocument"+" :: wellDone = "+wellDone);
		}
		
		
		return wellDone;		
	}
	
	
	/**
	 * Return in the MD description
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @return the MD description
	 * @throws ServerException
	 */
	public static String getMetadataDescription(Space currentSpace, String path) throws ServerException {
		
		String property = currentSpace.getServerAccessObject().getProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.DESCRIPTION);
		
		// log
		if (log.isDebugEnabled()){
			log.debug("getMetadataDescription"+" :: property = "+property);
		}
		
		return property;
	}
	
	
	
	/**
	 * Return in the MD title
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @return the MD title
	 * @throws ServerException
	 */
	public static String getMetadataTitle(Space currentSpace, String path) throws ServerException {
		
		String property = currentSpace.getServerAccessObject().getProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.TITLE);
		
		// log
		if (log.isDebugEnabled()){
			log.debug("getMetadataTitle"+" :: property = "+property);
		}
		
		return property;
	}
	
	
	/**
	 * Return the list of MD that the user have to set
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @return the list of MD that the user have to set
	 * @throws ServerException
	 */
	public static String getMetadataSpaceProfile(Space currentSpace, String path) throws ServerException {
		String profile = currentSpace.getServerAccessObject().getProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.DOCUMENT_INJAC_METADATA); 				
		
		// log
		if (log.isDebugEnabled()){
			log.debug("getMetadataSpaceProfile"+" :: profile = "+profile);
		}
		
		return profile;		
	}
	
	/**
	 * Return the rendering type MD that the user have to set
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @return the rendering type MD that the user have to set
	 * @throws ServerException
	 */
	public static String getMetadataRenderingType(Space currentSpace, String path) throws ServerException {						
		
		String property = currentSpace.getServerAccessObject().getProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.RENDERING_TYPE);
		
		// log
		if (log.isDebugEnabled()){
			log.debug("getMetadataRenderingType"+" :: property = "+property);
		}
		
		return property;
	}
	
	
	/**
	 * Parse the Metadata containing users and groups for a role
	 * @param property the property value
	 * @param users users completing in this method
	 * @param groups groups completing in this method
	 * @return true if public user, false else
	 */
	private static boolean parseXmlUser(String property, Vector users, Vector groups) {
		
		// if no property
		if (property == null || property.equals(""))
			return false;
		
		// get the element
		Node rootNode = null;
		try {
			DocumentBuilderFactory dbf = new DocumentBuilderFactoryImpl();
			DocumentBuilder db = dbf.newDocumentBuilder();
			
			InputSource is = new InputSource(new StringReader(property));
			Document doc = db.parse(is);
			rootNode = doc.getDocumentElement();			
		}
		catch(Exception e) {
			log.error("parseXmlUser"+" :: "+e);
		}
		
		
		boolean publicAccess = false;
				
		// search the users and groups		
		if (rootNode!=null) {
			
			// get the "publicAccess" attribute
			NamedNodeMap attributes = rootNode.getAttributes();
			Node publicAttribute = attributes.getNamedItem("publicAccess");
			if (publicAttribute!=null) {
				String value=publicAttribute.getNodeValue();
				if (value.equals("yes"))
					publicAccess = true;				
			}
			
			
			// get the childs
			NodeList list = rootNode.getChildNodes();
			for (int i=0; i<list.getLength(); i++) {				
				Node node = list.item(i);				
				if (node.getNodeType()!=Node.TEXT_NODE) {				
					attributes = node.getAttributes();				
					String displayName = null;
					String key = null;
					String type = null;
					for (int j=0; j<attributes.getLength(); j++) {
						Node attribute = attributes.item(j);
						String name = attribute.getNodeName();
						String value=attribute.getNodeValue();
						
						// check if this attribute is the key, type or display name
						if (name.equals("type"))
							type = value;
						else {
							if (name.equals("key"))
								key = value;
							else {
								if (name.equals("displayName"))
									displayName = value;
							}
						}										
					}
				
					// if we have all the attributes
					if (displayName!=null && key!=null && type!=null) {
						
						// if user
						if (type.equals("user")) {
							UserForPersonalization u = new UserForPersonalization(key, displayName);						
							users.add(u);
						}
					
						// if group
						else {
							if (type.equals("group")) {
								try {
									GroupForPersonalization g = new GroupForPersonalization(key, displayName);								
									groups.add(g);
								}
								catch(ServerException e) {
									log.error("parseXmlUser"+" :: "+e);
								}
							}						
						}
					}
				
				}			
			}
		}
		return publicAccess;
	}
	
	
	
	/**
	 * Return the MD : list of writers on the space
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @param users the users list to set
	 * @param groups the groups list to set
	 * @throws ServerException
	 */
	public static void getMetadataWritersOnSpace(Space currentSpace, String path, Vector users, Vector groups) throws ServerException {
		String prop = currentSpace.getServerAccessObject().getProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.WRITERS);										
		parseXmlUser(prop, users, groups);		
	}
	
	/**
	 * Return the MD : list of editors on the space
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @param users the users list to set
	 * @param groups the groups list to set
	 * @throws ServerException
	 */
	public static void getMetadataEditorsOnSpace(Space currentSpace, String path, Vector users, Vector groups) throws ServerException {
		String prop = currentSpace.getServerAccessObject().getProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.EDITORS);										
		parseXmlUser(prop, users, groups);		
	}
	
	/**
	 * Return the MD : list of managers on the space
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @param users the users list to set
	 * @param groups the groups list to set
	 * @throws ServerException
	 */
	public static void getMetadataManagersOnSpace(Space currentSpace, String path, Vector users, Vector groups) throws ServerException {
		String prop = currentSpace.getServerAccessObject().getProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.MANAGERS);										
		parseXmlUser(prop, users, groups);		
	}
	
	
	
	

	/**
	 * Set the root-file-name metadata of the directory specified by its path
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @param documentName the name of the root document file
	 * @return true if well done
	 * @throws ServerException
	 */
	public static boolean setSpaceRootDocumentName(Space currentSpace, String path, String documentName) throws ServerException {		
		
		boolean wellDone = currentSpace.getServerAccessObject().setProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.ROOT_DOC_NAME, documentName);
			
		// log
		if (log.isDebugEnabled()){
			log.debug("setSpaceRootDocumentName"+" :: wellDone = "+wellDone);
		}
		
		
		return wellDone;		
	}

	
	/**
	 * Return the root document name of a space
	 * @param currentSpace the current space used
	 * @param path the resource path
	 * @return the root document name of a space
	 * @throws ServerException
	 */
	public static String getSpaceRootDocumentName(Space currentSpace, String path) throws ServerException {						
		
		String property = currentSpace.getServerAccessObject().getProperty(currentSpace.getMetadataNamespace(), path, MetaDataConstants.ROOT_DOC_NAME);
		
		// log
		if (log.isDebugEnabled()){
			log.debug("getSpaceRootDocumentName"+" :: property = "+property);
		}
		
		return property;
	}
	
}
