/*
ESUP-portail is a french academic project developed under the GPL (General Public License) augmented according to the following :
A binary or source file developped by ESUP-portail can be used or compiled with products under Apache license.
The official english text of the GPL can be found here : http://www.gnu.org/licenses/gpl.html .
A non official french translation can be found here : http://www.linux-France.org/article/these/gpl.html .
The different kinds of licenses governing the products developed by the Apache foundation can be found here : http://www.apache.org/licenses .
It follows that you can as well as use download contents as well modify and redistribute them provided you respect the GPL terms.
Downloading and using such contents do not provide any guaranty.
Be sure that you have well understood the terms of the license before using the contents it covers.
The ESUP-portail distribution includes the following distributions :
    * UPortal :
      software developed by JA-SIG (Java Architecture - Special Interest Group)
      You can find the license page here : http://mis105.udel.edu/ja-sig/uportal/license.html
    * CAS :
      SSO solution developed by Yale University
      You can find the project page here : http://www.yale.edu/tp/auth
    * Cocoon :
      XML framework distributed by the Apache foundation under Apache license;
      Please find the full text here : http://cocoon.apache.org/2.1/license.html
    * Mod_dav:
      A DAV module for Apache web server
      You can find the project page here : http://www.webdav.org/mod_dav
    * IMP :
      webmail from Horde application framework
      You can find the project page here : http://www.horde.org
    * . To be completed
*/

package org.esupportail.portal.channels.CWebdav.provider;

import java.net.MalformedURLException;
import java.util.Vector;

import org.esupportail.portal.channels.CWebdav.provider.ChannelWebdavResource;
import org.esupportail.portal.channels.CWebdav.exception.ApplicationException;
import org.esupportail.portal.channels.CWebdav.exception.BadFormatException;
import org.esupportail.portal.channels.CWebdav.exception.CopyException;
import org.esupportail.portal.channels.CWebdav.exception.CreateDirectoryException;
import org.esupportail.portal.channels.CWebdav.exception.DeleteException;
import org.esupportail.portal.channels.CWebdav.exception.DownloadException;
import org.esupportail.portal.channels.CWebdav.exception.EmptyFileOnUploadException;
import org.esupportail.portal.channels.CWebdav.exception.MoveException;
import org.esupportail.portal.channels.CWebdav.exception.NotAuthorizedDeleteException;
import org.esupportail.portal.channels.CWebdav.exception.NotAuthorizedException;
import org.esupportail.portal.channels.CWebdav.exception.NotAuthorizedNewDirException;
import org.esupportail.portal.channels.CWebdav.exception.NotAuthorizedRenameException;
import org.esupportail.portal.channels.CWebdav.exception.NotAuthorizedUploadException;
import org.esupportail.portal.channels.CWebdav.exception.NotExistsResourceException;
import org.esupportail.portal.channels.CWebdav.exception.OverQuotaException;
import org.esupportail.portal.channels.CWebdav.exception.PasteDeletedResourceException;
import org.esupportail.portal.channels.CWebdav.exception.PasteInChildDirectoryException;
import org.esupportail.portal.channels.CWebdav.exception.PasteNotAuthorizedResourceException;
import org.esupportail.portal.channels.CWebdav.exception.PasteWithSameNameException;
import org.esupportail.portal.channels.CWebdav.exception.RenameException;
import org.esupportail.portal.channels.CWebdav.exception.ServerException;
import org.esupportail.portal.channels.CWebdav.exception.StillExistsException;
import org.esupportail.portal.channels.CWebdav.exception.UploadException;
import org.jasig.portal.MultipartDataSource;


/**
 * Id: DavAccess.java,v 1.0 14 avr. 2004<br/>
 * Copyright (c) 2004 Esup Portail (www.esup-portail.org)<br/>
 * Classes: DavAccess<br/>
 * Original Author: Yohan Colmant<br/>
 * Interface for the dav access<br/>
 */
public abstract class DavAccess {
	
    
	/**
	 * Give the user login
	 * @return the user login
	 */
	public abstract String getLogin();
	
	/**
	 * Give the user password
	 * @return the user password
	 */
	public abstract String getPassword();
	
	/**
	 * Give the uri webdav access
	 * @return the uri webdav access
	 */
	public abstract String getUri();
	
	/**
     * This method initialize the parameters. You have to call this method after calling connect method.
     * @param uri webdav uri
     * @param path access path
     * @param login connexion login
     * @param password connexion password
     * @throws MalformedURLException
     */
	public abstract void init(String uri, String path, String login, String password) throws MalformedURLException;
	
	/**
     * This method initialize the parameters. You have to call this method after calling connect method.
     * @param uri webdav uri
     * @param login connexion login
     * @param password connexion password
     * @throws MalformedURLException
     */
	public abstract void init(String uri, String login, String password) throws MalformedURLException;
	
	
	/**
	 * Return the hierarchy path oh the server. for example, if the server's url is http://URL:PORT/slide, the currentHierPath is "/slide"
	 * @return a String
	 * @throws MalformedURLException
	 */
	public abstract String getServerCurrentHierPath() throws MalformedURLException;
	
	/**
	 * Connect this object to the server	 
     * @throws ServerException
	 */
	public abstract void connect() throws ServerException;
	
	/**
	 * Disconnection of the server
	 * @throws ServerException
	 */
	public abstract void disconnect() throws ServerException;
	
	/**
	 * List all ressources for a given path
	 * @param fullPath directory path on the dav server
	 * @return ChannelWebdavResource array
	 * @throws NotExistsResourceException
	 * @throws ApplicationException
	 * @throws NotAuthorizedException
	 */	
	public abstract ChannelWebdavResource[] ls(String fullPath) throws NotExistsResourceException, ApplicationException, NotAuthorizedException;
	
	/**
	 * List all ressources for a given path
	 * @param path directory path on the dav server
	 * @param targetDirectory the directory name to list
	 * @return ChannelWebdavResource array
	 * @throws NotExistsResourceException
	 * @throws ApplicationException
	 * @throws NotAuthorizedException
	 */	
	public abstract ChannelWebdavResource[] ls(String path, String targetDirectory) throws NotExistsResourceException, ApplicationException, NotAuthorizedException;	
	
	/**
	 * Upload a file to the path given in argument
	 * @param inputFile the input file
	 * @param path the path
	 * @return boolean true if no problem, else false
	 * @throws ApplicationException
	 * @throws ServerException
	 * @throws NotExistsResourceException
	 * @throws StillExistsException
	 * @throws BadFormatException
	 * @throws UploadException
	 * @throws EmptyFileOnUploadException
	 * @throws OverQuotaException
	 * @throws NotAuthorizedDeleteException
	 * @throws NotAuthorizedUploadException
	 * @throws NotAuthorizedException
	 */	
	public abstract boolean upload(MultipartDataSource inputFile, String path)throws ApplicationException, ServerException, NotExistsResourceException, StillExistsException, BadFormatException, UploadException, EmptyFileOnUploadException,OverQuotaException, NotAuthorizedDeleteException, NotAuthorizedUploadException, NotAuthorizedException;
	
	/**
	 * Delete the resource given in argument
	 * @param file file/directory to delete
	 * @param path path where is file/directory
	 * @return boolean true if well done else false
	 * @throws ApplicationException
	 * @throws DeleteException
	 * @throws NotAuthorizedDeleteException
	 * @throws NotExistsResourceException
	 * @throws NotAuthorizedException
	 */
	public abstract boolean delete(String file, String path) throws ApplicationException, DeleteException, NotAuthorizedDeleteException, NotExistsResourceException, NotAuthorizedException ;
	
	/**
	 * Rename the resource
	 * @param oldName file/directory to rename
	 * @param newName new name of the file/directory
	 * @param path path where is file/directory
	 * @return boolean true if well done else false
	 * @throws StillExistsException
	 * @throws BadFormatException
	 * @throws ApplicationException
	 * @throws RenameException
	 * @throws NotAuthorizedRenameException
	 * @throws NotExistsResourceException
	 * @throws NotAuthorizedException
	 */
	public abstract boolean rename(String oldName, String newName, String path) throws StillExistsException, BadFormatException, ApplicationException, RenameException, NotAuthorizedRenameException, NotExistsResourceException, NotAuthorizedException;
	
	/**
	 * Create the directory given in argument
	 * @param name file/directory to delete
	 * @param path path where is file/directory
	 * @return boolean true if well done else false
	 * @throws ApplicationException
	 * @throws CreateDirectoryException
	 * @throws StillExistsException
	 * @throws BadFormatException
	 * @throws NotAuthorizedNewDirException
	 * @throws NotExistsResourceException
	 * @throws NotAuthorizedException
	 */
	abstract public boolean createDir (String name, String path) throws ApplicationException, CreateDirectoryException, StillExistsException, BadFormatException, NotAuthorizedNewDirException, NotExistsResourceException, NotAuthorizedException;

	/**
	 * Check if we can paste here
	 * @param clipboardPath the path where is the clipboad
	 * @param clipboard the containt of the clipboard
	 * @param curentDirPath the path of the curent dir
	 * @return true if ok excpetion else
	 * @throws ApplicationException
	 * @throws PasteInChildDirectoryException
	 * @throws PasteWithSameNameException
	 * @throws PasteDeletedResourceException
	 * @throws NotAuthorizedException
	 */
	abstract public boolean canPaste(String clipboardPath, Vector clipboard, String curentDirPath) throws ApplicationException, PasteInChildDirectoryException, PasteWithSameNameException, PasteDeletedResourceException, NotAuthorizedException;
	
	/**
	 * Copy method
	 * @param resource resource to copy
	 * @param fromPath path where is the resource
	 * @param toPath path to copy the resource
	 * @return boolean true if well done else false
	 * @throws CopyException
	 * @throws ApplicationException
	 * @throws OverQuotaException
	 * @throws NotAuthorizedException
	 * @throws NotExistsResourceException
	 * @throws PasteNotAuthorizedResourceException
	 * @throws NotAuthorizedUploadException
	 */
	abstract public boolean copy(String resource, String fromPath, String toPath) throws CopyException, ApplicationException,OverQuotaException, NotAuthorizedException, NotExistsResourceException, PasteNotAuthorizedResourceException, NotAuthorizedUploadException;
	
	/**
	 * Move method
	 * @param resource resource to move
	 * @param fromPath path where is the resource
	 * @param toPath path to move the resource
	 * @return boolean true if well done else false
	 * @throws ApplicationException
	 * @throws MoveException
	 * @throws NotAuthorizedException
	 * @throws NotExistsResourceException
	 * @throws PasteNotAuthorizedResourceException
	 * @throws NotAuthorizedUploadException
	 */
	abstract public boolean move(String resource, String fromPath, String toPath) throws ApplicationException, MoveException, NotAuthorizedException, NotExistsResourceException, PasteNotAuthorizedResourceException, NotAuthorizedUploadException;
	
	/**
	 * Return the resource associated to the file to download
	 * @param name the file name
	 * @param path the path where to find the file
	 * @return the ChannelWebdavResource
	 * @throws DownloadException
	 */
	abstract public ChannelWebdavResource dowload(String name,String path) throws DownloadException;
	
	
	
	
	/************************
	 * Webdav control methods
	 ************************/
	
	
	
	/**
	 * Check is the resource exists
	 * @param fullPath path to find the file/directory
	 * @return true if file/directory exists else false
	 * @throws ApplicationException
	 * @throws NotAuthorizedException
	 */
	public abstract boolean exists(String fullPath) throws ApplicationException, NotAuthorizedException;
	
	/**
	 * Check is the resource exists
	 * @param resourceName name of the file/directory to verify
	 * @param path path to find the file/directory
	 * @return true if file/directory exists else false
	 * @throws ApplicationException
	 * @throws NotAuthorizedException
	 */
	public abstract boolean exists(String resourceName, String path) throws ApplicationException, NotAuthorizedException;
	
	/**
	 * Check if we are authorized to read this resource
	 * @param resourceName name of the file/directory to verify
	 * @param path path to find the file/directory
	 * @return true if authorized else false
	 * @throws ApplicationException
	 */
	public abstract boolean canRead(String resourceName, String path) throws ApplicationException;
	
	/**
	 * Check if we are authorized to read this resource
	 * @param path path to find the file/directory
	 * @return true if authorized else false
	 * @throws ApplicationException
	 */
	public abstract boolean canRead(String path) throws ApplicationException;
	
	/**
	 * Check if the resource named name is a directory in the path given
	 * @param name the name of the resource
	 * @param path the path
	 * @return true if directory, false else
	 * @throws ApplicationException
	 * @throws NotAuthorizedException
	 * @throws NotExistsResourceException
	 */
	public abstract boolean isDirectory(String name, String path) throws ApplicationException, NotAuthorizedException, NotExistsResourceException;

	/**
	 * Check if the directory named dir is empty or not. We suppose that dir has been checked as a directory before
	 * @param dir the name of the directory
	 * @param path the path
	 * @return true if empty, false else
	 * @throws ApplicationException
	 * @throws NotExistsResourceException
	 * @throws NotAuthorizedException
	 */
	public abstract boolean isEmpty (String dir, String path) throws ApplicationException, NotExistsResourceException, NotAuthorizedException;	
	
	
	/**
	 * Get a resource's property specified by the path
	 * @param path the resource path
	 * @param propertyName the property name
	 * @return a vector containing the properties
	 * @throws ServerException
	 */
	
	/*********************
	 * Properties methods
	 *********************/
	
	
	/**
	 * Get a resource's property specified by the path
	 * @param path the resource path
	 * @param propertyName the property name
	 * @return a vector containing the properties
	 * @throws ServerException
	 */
	public abstract Vector getProperty(String path, String propertyName) throws ServerException;

	/**
	 * Get a resource's property specified by the path
	 * @param path the resource path
	 * @param properties a vector containing the properties name
	 * @return a vector containing the properties
	 * @throws ServerException
	 */
	public abstract Vector getProperty(String path, Vector properties) throws ServerException;

	/**
	 * Set the property of resource
	 * @param path the resource path
	 * @param propertyName the name of the property to set
	 * @param propertyValue the value of the property to set
	 * @return true if well done
	 * @throws ServerException
	 */	
	public abstract boolean setProperty(String path, String propertyName, String propertyValue) throws ServerException;

	
}
