/*
ESUP-portail is a french academic project developed under the GPL (General Public License) augmented according to the following :
A binary or source file developped by ESUP-portail can be used or compiled with products under Apache license.
The official english text of the GPL can be found here : http://www.gnu.org/licenses/gpl.html .
A non official french translation can be found here : http://www.linux-France.org/article/these/gpl.html .
The different kinds of licenses governing the products developed by the Apache foundation can be found here : http://www.apache.org/licenses .
It follows that you can as well as use download contents as well modify and redistribute them provided you respect the GPL terms.
Downloading and using such contents do not provide any guaranty.
Be sure that you have well understood the terms of the license before using the contents it covers.
The ESUP-portail distribution includes the following distributions :
    * UPortal :
      software developed by JA-SIG (Java Architecture - Special Interest Group)
      You can find the license page here : http://mis105.udel.edu/ja-sig/uportal/license.html
    * CAS :
      SSO solution developed by Yale University
      You can find the project page here : http://www.yale.edu/tp/auth
    * Cocoon :
      XML framework distributed by the Apache foundation under Apache license;
      Please find the full text here : http://cocoon.apache.org/2.1/license.html
    * Mod_dav:
      A DAV module for Apache web server
      You can find the project page here : http://www.webdav.org/mod_dav
    * IMP :
      webmail from Horde application framework
      You can find the project page here : http://www.horde.org
    * . To be completed
*/

package org.esupportail.portal.channels.CWebdav.provider;
 
import java.io.IOException; 
import java.net.MalformedURLException;
import java.util.Enumeration;
import java.util.Vector; 
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpURL;
import org.apache.commons.httpclient.URIException;
import org.apache.webdav.lib.BaseProperty;
import org.apache.webdav.lib.PropertyName;
import org.apache.webdav.lib.WebdavResource;
import org.apache.webdav.lib.methods.DepthSupport;
import org.apache.webdav.lib.methods.PropFindMethod;
import org.esupportail.portal.channels.CWebdav.provider.ChannelWebdavResource;
import org.esupportail.portal.channels.CWebdav.exception.ApplicationException;
import org.esupportail.portal.channels.CWebdav.exception.BadFormatException;
import org.esupportail.portal.channels.CWebdav.exception.CopyException;
import org.esupportail.portal.channels.CWebdav.exception.CreateDirectoryException;
import org.esupportail.portal.channels.CWebdav.exception.DeleteException;
import org.esupportail.portal.channels.CWebdav.exception.DownloadException;
import org.esupportail.portal.channels.CWebdav.exception.EmptyFileOnUploadException;
import org.esupportail.portal.channels.CWebdav.exception.MoveException;
import org.esupportail.portal.channels.CWebdav.exception.NotAuthorizedDeleteException;
import org.esupportail.portal.channels.CWebdav.exception.NotAuthorizedException;
import org.esupportail.portal.channels.CWebdav.exception.NotAuthorizedNewDirException;
import org.esupportail.portal.channels.CWebdav.exception.NotAuthorizedRenameException;
import org.esupportail.portal.channels.CWebdav.exception.NotAuthorizedUploadException;
import org.esupportail.portal.channels.CWebdav.exception.NotExistsResourceException;
import org.esupportail.portal.channels.CWebdav.exception.OverQuotaException;
import org.esupportail.portal.channels.CWebdav.exception.PasteDeletedResourceException;
import org.esupportail.portal.channels.CWebdav.exception.PasteInChildDirectoryException;
import org.esupportail.portal.channels.CWebdav.exception.PasteNotAuthorizedResourceException;
import org.esupportail.portal.channels.CWebdav.exception.PasteWithSameNameException;
import org.esupportail.portal.channels.CWebdav.exception.RenameException;
import org.esupportail.portal.channels.CWebdav.exception.ServerException;
import org.esupportail.portal.channels.CWebdav.exception.StillExistsException;
import org.esupportail.portal.channels.CWebdav.exception.UploadException;
import org.jasig.portal.MultipartDataSource;
import org.jasig.portal.services.LogService;
import org.w3c.dom.Node;

/**
 * Id: DavAccessImpl.java,v 1.0 14 avr. 2004<br/>
 * Copyright (c) 2004 Esup Portail (www.esup-portail.org)<br/>
 * Classes: DavAccessImpl<br/>
 * Original Author: Yohan Colmant<br/>
 * Implements the DavAccess methods<br/>
 */
public class DavAccessImpl extends DavAccess{
	
	/**
	 * The dav access uri
	 */
	private String uri;
	
	/**
	 * The user login
	 */
	private String login;
	
	/**
	 * The password for the dav acccess
	 */
	private String password;
	
	/**
	 * The Http url used to access the webdav
	 */
	private HttpURL httpURL;	
	
	/** 
	 * The WebDAV ressource 
	 */
	private WebdavResource resource;
	
	
    
    
	
	/**
     * Default constructor
     */
	public DavAccessImpl() {		
	}
	
	
	/**
	 * Give the user login
	 * @return the user login
	 */
	public String getLogin() {
		return this.login;
	}
	
	/**
	 * Give the user password
	 * @return the user password
	 */
	public String getPassword() {
		return this.password;
	}
	
	/**
	 * Give the uri webdav access
	 * @return the uri webdav access
	 */
	public String getUri() {
		return this.uri;
	}
	
	/**
     * This method initialize the parameters. You have to call this method after calling connect method.
     * @param uri webdav uri
     * @param path access path
     * @param login connexion login
     * @param password connexion password
     * @throws MalformedURLException
     */
	public void init(String uri, String path, String login, String password) throws MalformedURLException {	
		this.uri = uri;	
		
		if (this.uri.charAt(this.uri.length()-1)!='/')
			this.uri = this.uri+"/";
		
		if (!path.startsWith("/")) {
			this.uri = this.uri+path;
		}
		else {
			if (path.length()!=1)
				this.uri = this.uri+(path.substring(1, path.length()-1));
		}
		
		if (this.uri.charAt(this.uri.length()-1)!='/')
			this.uri = this.uri+"/";
				
		this.login = login;
		this.password = password;
		try {
			httpURL = new HttpURL(this.uri);
			httpURL.setUserinfo(this.login, this.password);
		}
		catch(URIException e) {			
			LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::init "+e);
            throw new MalformedURLException();
		}
	}
	
	
	
	

	/**
     * This method initialize the parameters. You have to call this method after calling connect method.
     * @param uri webdav uri
     * @param login connexion login
     * @param password connexion password
     * @throws MalformedURLException
     */
	public void init(String uri, String login, String password) throws MalformedURLException {	
		this.uri = uri;	
		
		if (this.uri.charAt(this.uri.length()-1)!='/')
			this.uri = this.uri+"/";
				
		this.login = login;
		this.password = password;
		try {
			httpURL = new HttpURL(this.uri);
			httpURL.setUserinfo(this.login, this.password);
		}
		catch(URIException e) {			
			LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::init "+e);
            throw new MalformedURLException();
		}
	}
	
	
	
	/**
	 * Return the hierarchy path oh the server. for example, if the server's url is http://URL:PORT/slide, the currentHierPath is "/slide"
	 * @return a String
	 * @throws MalformedURLException
	 */
	public String getServerCurrentHierPath() throws MalformedURLException {
		try {
			return httpURL.getCurrentHierPath();
		}
		catch(URIException e) {			
			LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::getServerCurrentHierPath "+e);
            throw new MalformedURLException();
		}
	}
	
	
	
	/**
	 * Connect this object to the server	 
     * @throws ServerException
	 */
	public void connect() throws ServerException {		
		try {
			// connection
			if (this.resource == null) {
				this.resource = new WebdavResource(httpURL);
				this.resource.setDebug(0);
			}
			else this.resource.setHttpURL(httpURL);
		}
		catch (HttpException ex) {		
        	LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::connect "+ex);
            throw new ServerException();
        }
		catch (IOException ex) {  			
        	LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::connect "+ex);
            throw new ServerException();
        }		
    }
	
	 
	/**
	 * Disconnection of the server
	 * @throws ServerException
	 */
	public void disconnect() throws ServerException {
		try {
			this.resource.close();			
		}
		catch (IOException ex) {  
        	LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::disconnect "+ex);
            throw new ServerException();
        }
    }
	
	
	
	/**
	 * List all ressources for a given path
	 * @param fullPath directory path on the dav server
	 * @return ChannelWebdavResource array
	 * @throws NotExistsResourceException
	 * @throws ApplicationException
	 * @throws NotAuthorizedException
	 */	
	public ChannelWebdavResource[] ls(String fullPath) throws NotExistsResourceException, ApplicationException, NotAuthorizedException {		
		
		//System.out.println("fullPath: "+fullPath);
		
		if (this.resource == null)
			throw new ApplicationException();
		else {
			try {
				
				// we are in a directory, if this path doesn't finish by a "/", we add it
				if(!fullPath.endsWith(("/"))){
					fullPath=fullPath+"/";
				}
				
				if (exists(fullPath)) {					
					this.resource.setPath(fullPath);								
				}
				else {						
					throw new NotExistsResourceException();
				}				
				
				Vector ressources = new Vector();
				WebdavResource[] liste = this.resource.listWebdavResources();
				if (this.resource.getStatusCode()==404)
					throw new NotExistsResourceException();
								
				int length = liste.length;				
				
				// we transform the WebdavResource array into a ChannelWebdavResource array				
				ChannelWebdavResource[] channelList = new ChannelWebdavResource[length];
											
				for (int i=0; i<length; i++) {
					WebdavResource res = liste[i];					
					channelList[i] = new ChannelWebdavResource( res.getName(), res.getGetContentLength(), res.getGetContentType(), res.getGetLastModified(), res.isCollection());
				}
							
				return channelList;
			}
			catch(HttpException e) {						
				LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::ls:"+e);
				throw new NotAuthorizedException();
			}
			catch(IOException e) {
				LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::ls:"+e);
				throw new ApplicationException();
			}			
			}       	
	}
	
	
	
		
	/**
	 * List all ressources for a given path
	 * @param path directory path on the dav server
	 * @param targetDirectory the directory name to list
	 * @return ChannelWebdavResource array
	 * @throws NotExistsResourceException
	 * @throws ApplicationException
	 * @throws NotAuthorizedException
	 */	
	public ChannelWebdavResource[] ls(String path, String targetDirectory) throws NotExistsResourceException, ApplicationException, NotAuthorizedException {		
		
		//System.out.println("path: "+path+"   targetDirectory: "+targetDirectory);
		
		if (this.resource == null)
			throw new ApplicationException();
		else {
			try {
				String fullPath = path+targetDirectory;

				// we are in a directory, if this path doesn't finish by a "/", we add it
				if(!fullPath.endsWith(("/"))){
					fullPath=fullPath+"/";
				}
				
				
				//System.out.println("fullPath: "+fullPath);
				if (exists(fullPath)) {					
					this.resource.setPath(fullPath);								
				}
				else {											
					throw new NotExistsResourceException();
				}
				
				Vector ressources = new Vector();
				WebdavResource[] liste = this.resource.listWebdavResources();
				if (this.resource.getStatusCode()==404)
					throw new NotExistsResourceException();
			
				int length = liste.length;
				
				// we transform the WebdavResource array into a ChannelWebdavResource array
				ChannelWebdavResource[] channelList = new ChannelWebdavResource[length];
				for (int i=0; i<length; i++) {
					WebdavResource res = liste[i];
					channelList[i] = new ChannelWebdavResource( res.getName(), res.getGetContentLength(), res.getGetContentType(), res.getGetLastModified(), res.isCollection());
				}
				
			
				return channelList;
			}
			catch(HttpException e) {						
				LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::ls:"+e);
				throw new NotAuthorizedException();
			}
			catch(IOException e) {
				LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::ls:"+e);
				throw new ApplicationException();
			}			
			}       	
	}
	
	
	/**
	 * Upload a file to the path given in argument
	 * @param inputFile the input file
	 * @param path the path
	 * @return boolean true if no problem, else false
	 * @throws ApplicationException
	 * @throws ServerException
	 * @throws NotExistsResourceException
	 * @throws StillExistsException
	 * @throws BadFormatException
	 * @throws UploadException
	 * @throws EmptyFileOnUploadException
	 * @throws OverQuotaException
	 * @throws NotAuthorizedDeleteException
	 * @throws NotAuthorizedUploadException
	 * @throws NotAuthorizedException
	 */
	public boolean upload(MultipartDataSource inputFile, String path) throws ApplicationException, ServerException, NotExistsResourceException, StillExistsException, BadFormatException, UploadException, EmptyFileOnUploadException,OverQuotaException, NotAuthorizedDeleteException, NotAuthorizedUploadException, NotAuthorizedException{ 

		try {			
			
			// get the file name		      		      
			String fileName = inputFile.getName();
			
			// the file to upload to the webdav			           					
            String cible = path+fileName;                                            
            
            // check if the file still exists 
            boolean exist = false;                       
            exist = exists(cible);
            
            // we check if a directory has this name
            if(!cible.endsWith(("/"))){            	
            	exist = exist || exists(cible+"/");
            }
            
            if ( exist ) {
            	LogService.log(LogService.INFO, "CWebdav::DavAccessImpl::upload : file already exist");
            	throw new StillExistsException();
            }
            else {       
            	// check if the filename contains fobiden characters
            	boolean wellFormed = DavControl.isWellFormed(fileName);
            	if ( !wellFormed ) {
                	LogService.log(LogService.INFO, "CWebdav::DavAccessImpl::upload : name not well formed");
                	throw new BadFormatException();
            	}
            	// we can send
            	else {            		            		
            		
            		boolean answer = this.resource.putMethod(cible, inputFile.getInputStream());
            		int httpCode = this.resource.getStatusCode();
            		
            		// if a multistatus response
            		if (httpCode == 207 || httpCode == 403)
            			throw new NotAuthorizedUploadException();
            		
            		if (httpCode==507)throw new OverQuotaException(httpCode);
            		
            		if ((httpCode>=400) && (httpCode<600) ) answer=false;
            		
            		if (answer) {
            			
            			// we check if the uploaded file is empty
            			if (fileIsEmptyOnServer(fileName, path)) {
            				
            				// we delete the file
            				delete(fileName, path);
            				
            				// we throw a new Exception
            				throw new EmptyFileOnUploadException();
            			}
            			else {                  				
            				return true;
            			}            			
            		}
            		else {
                    	//LogService.log(LogService.ERROR, "DavAccessImpl::upload ");
                    	LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::upload http code : "+ httpCode);
            			throw new UploadException(httpCode);                			
            		}     
            	}
            }
        }
		catch (HttpException ex) {
        	LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::upload "+ex);
            throw new ServerException();
        }
		catch (IOException ex) {  
        	LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::upload "+ex);
            throw new ServerException();
        }
        catch (DeleteException ex) {
        	LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::upload "+ex);
            throw new UploadException();
        } 
	}
	
	
	/**
	 * Check if a file is empty on the server
	 * @param fileName the name of the file to check
	 * @param path the path where is the file on the server
	 * @return true if the file is empty on the server, false else
	 * @throws ApplicationException
	 */
	private boolean fileIsEmptyOnServer(String fileName, String path) throws ApplicationException {
		
		try {
			String pathAndName = path+fileName;
			
			// check if this is a valid resource
			if (exists(pathAndName)) {
				this.resource.setPath(pathAndName);								
			}
			else {											
				throw new NotExistsResourceException();
			}
			
			if (this.resource.exists()) {
				int length = (int)resource.getGetContentLength();				
				if (length == 0)
					return true;
				else return false;
			}
			else return false;
		}
		catch (Exception e) {			
			LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::fileIsEmptyOnServer:"+e);
			throw new ApplicationException();
		}		   
	}
	
	
	/**
	 * Delete the resource given in argument
	 * @param file file/directory to delete
	 * @param path path where is file/directory
	 * @return boolean true if well done else false
	 * @throws ApplicationException
	 * @throws DeleteException
	 * @throws NotAuthorizedDeleteException
	 * @throws NotExistsResourceException
	 * @throws NotAuthorizedException
	 */
	public boolean delete(String file, String path) throws ApplicationException, DeleteException, NotAuthorizedDeleteException, NotExistsResourceException, NotAuthorizedException {
		try {			 
            String cible = path+file;
			
            // check if is a directory
            boolean isDir = isDirectory(file, path);
            if (isDir)
            	cible = cible+"/";            
            
            boolean answer = this.resource.deleteMethod(cible);
    		int httpCode = this.resource.getStatusCode();
    		
    		// if a multistatus response
    		if (httpCode == 207 || httpCode == 403)
    			throw new NotAuthorizedDeleteException();
    		
    		if ((httpCode>=400) && (httpCode<600) ) answer=false;
    		if (answer) { 
                return true;
            } else {
            	LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::delete http code : "+ httpCode);
    			throw new DeleteException(httpCode);
            }
        }
		catch (HttpException ex) {
         	LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::delete : "+ex);
        	throw new NotAuthorizedDeleteException();
        } 
		catch (IOException ex) {
        	LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::delete : "+ex);
        	throw new ApplicationException();
        }                   
	}
	
	/**
	 * Rename the resource
	 * @param oldName file/directory to rename
	 * @param newName new name of the file/directory
	 * @param path path where is file/directory
	 * @return boolean true if well done else false
	 * @throws StillExistsException
	 * @throws BadFormatException
	 * @throws ApplicationException
	 * @throws RenameException
	 * @throws NotAuthorizedRenameException
	 * @throws NotExistsResourceException
	 * @throws NotAuthorizedException
	 */
	public boolean rename(String oldName, String newName, String path) throws StillExistsException, BadFormatException, ApplicationException, RenameException, NotAuthorizedRenameException, NotExistsResourceException, NotAuthorizedException {
		try {
			//check if file exist on dav Server
			String fullPath = path+newName;			
            boolean exist = exists(fullPath);
            
            if(!fullPath.endsWith(("/")) && !exist) {
            	fullPath=fullPath+"/";
            	exist = exist || exists(fullPath);
            }
            
            if ( exist ) {
            	LogService.log(LogService.INFO, "CWebdav::DavAccessImpl::rename : file already exist");
            	throw new StillExistsException();            	
            }
            else {          
            	// check if the filename contains fobiden characters
            	boolean wellFormed = DavControl.isWellFormed(newName);
            	if ( !wellFormed ) { 
            		LogService.log(LogService.INFO, "CWebdav::DavAccessImpl::rename : name not well formed");
                	throw new BadFormatException();
            	}
            	// we can rename
            	else {            		
            		String cible = path+oldName;
			
            		// check if is a directory
            		boolean isDir = isDirectory(oldName, path);
            		if (isDir)
            			cible = cible+"/"; 
            		
            		if (exists(cible)) {
    					this.resource.setPath(cible);								
    				}
    				else {											
    					throw new NotExistsResourceException();
    				}
            		
            		// we rename            
            		cible = path+newName;
            		if (isDir)
            			cible = cible+"/";
            		
            		boolean answer = this.resource.moveMethod(cible);
             		int httpCode = this.resource.getStatusCode();
             		
             		// if a multistatus response
            		if (httpCode == 207 || httpCode == 403)
            			throw new NotAuthorizedRenameException();
             		
             		if ((httpCode>=400) && (httpCode<600) ) answer=false;
             		if (answer) { 
            			return true;
             		}
            		else{
            			LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::rename http code : "+ httpCode);
            			throw new RenameException(httpCode);
            		}
            	}            
            }
		}
        catch (HttpException ex) {
        	LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::rename : "+ex);
        	throw new NotAuthorizedRenameException();
        }
        catch (IOException ex) {
        	LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::rename : "+ex);
        	throw new RenameException();
        }               
	}
	
	
	/**
	 * Create the directory given in argument
	 * @param name file/directory to delete
	 * @param path path where is file/directory
	 * @return boolean true if well done else false
	 * @throws ApplicationException
	 * @throws CreateDirectoryException
	 * @throws StillExistsException
	 * @throws BadFormatException
	 * @throws NotAuthorizedNewDirException
	 * @throws NotExistsResourceException
	 * @throws NotAuthorizedException
	 */
	public boolean createDir (String name, String path) throws ApplicationException, CreateDirectoryException, StillExistsException, BadFormatException, NotAuthorizedNewDirException, NotExistsResourceException, NotAuthorizedException{
		try {
			//look if a file with the same name exist
			String fullPath = path+name;			
            boolean exist = exists(fullPath);
            
            if(!fullPath.endsWith(("/")) && !exist) {
            	fullPath=fullPath+"/";
            	exist = exist || exists(fullPath);
            }
            
            if ( exist ) {
            	LogService.log(LogService.INFO, "CWebdav::DavAccessImpl::createDir : file already exist");
            	throw new StillExistsException();           	
            }
            else {          
            	// check if the file is well-formed
            	boolean wellFormed = DavControl.isWellFormed(name);
            	if ( !wellFormed ) {
            		LogService.log(LogService.INFO, "CWebdav::DavAccessImpl::createDir : name not well formed");
                	throw new BadFormatException();
            	}
            	// we can create the directory
            	else {            		
            		if (exists(path)) {
    					this.resource.setPath(path);								
    				}
    				else {											
    					throw new NotExistsResourceException();
    				}
            		
               		boolean answer = this.resource.mkcolMethod(path+name+"/");
               		int httpCode = this.resource.getStatusCode();
               		
               		// if a multistatus response
            		if (httpCode == 207 || httpCode == 403)
            			throw new NotAuthorizedNewDirException();
               		
            		if ((httpCode>=400) && (httpCode<600) ) answer=false;
            		if (answer)
            			return true;
            		else {
                    	LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::createDir : code http : "+httpCode);
                    	throw new CreateDirectoryException(httpCode);
            		}
            	}
            }
		}
        catch (HttpException ex) {
        	LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::createDir : "+ex);
        	throw new NotAuthorizedNewDirException();
        }
        catch (IOException ex) {
         	LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::createDir : "+ex);
        	throw new CreateDirectoryException();
        }          
	}
	
	
	/**
	 * Check if we can paste here
	 * @param clipboardPath the path where is the clipboad
	 * @param clipboard the containt of the clipboard
	 * @param curentDirPath the path of the curent dir
	 * @return true if ok excpetion else
	 * @throws ApplicationException
	 * @throws PasteInChildDirectoryException
	 * @throws PasteWithSameNameException
	 * @throws PasteDeletedResourceException
	 * @throws NotAuthorizedException
	 */
	public boolean canPaste(String clipboardPath, Vector clipboard, String curentDirPath) throws ApplicationException, PasteInChildDirectoryException, PasteWithSameNameException, PasteDeletedResourceException, NotAuthorizedException {
		
		// we check if there is a resource with the same name in the curent dir
		for (int i=0; i<clipboard.size(); i++) {
			String resourceName = (String)clipboard.elementAt(i);
			
			String fullPath = curentDirPath+resourceName;			
            boolean exist = exists(fullPath);
            
            if(!fullPath.endsWith(("/")) && !exist) {
            	fullPath=fullPath+"/";
            	exist = exist || exists(fullPath);
            }
			
			if (exist)
				throw new PasteWithSameNameException();			
		}
		
		// we check if we try to paste a resource in its child
		for (int i=0; i<clipboard.size(); i++) {
			String resourceName = (String)clipboard.elementAt(i);
			if (curentDirPath.startsWith(clipboardPath+resourceName+"/")) {				
				throw new PasteInChildDirectoryException();
			}					
		}
		
		// we check if one of the selected resource has been deleted
		for (int i=0; i<clipboard.size(); i++) {
			String resourceName = (String)clipboard.elementAt(i);			
			
			String fullPath = clipboardPath+resourceName;
			
			boolean exist = exists(fullPath);
            
            if(!fullPath.endsWith(("/")) && !exist) {
            	fullPath=fullPath+"/";
            	exist = exist || exists(fullPath);
            }
			
			
			if (!exist)
				throw new PasteDeletedResourceException();					
		}
		
		return true;
	}
	
	
	/**
	 * Copy method
	 * @param resource resource to copy
	 * @param fromPath path where is the resource
	 * @param toPath path to copy the resource
	 * @return boolean true if well done else false
	 * @throws CopyException
	 * @throws ApplicationException
	 * @throws OverQuotaException
	 * @throws NotAuthorizedException
	 * @throws NotExistsResourceException
	 * @throws PasteNotAuthorizedResourceException
	 * @throws NotAuthorizedUploadException
	 */
	public boolean copy(String resource, String fromPath, String toPath) throws CopyException, ApplicationException,OverQuotaException, PasteNotAuthorizedResourceException, NotExistsResourceException, PasteNotAuthorizedResourceException, NotAuthorizedException, NotAuthorizedUploadException {
		try {						
			
			// check if resource is a directory   
			String realResource = resource;
			boolean isDir = isDirectory(resource, fromPath);
            if (isDir)
            	realResource = realResource+"/"; 			
			
			// absolute path of the resource
			String absoluteFromPath = fromPath+realResource;
			
			// absolute path to copy the resource
			String absoluteToPath = toPath;
			if (!toPath.endsWith("/"))
				absoluteToPath += "/";
			absoluteToPath += realResource;						
            
			// fix the path
			if (exists(absoluteFromPath)) {
				this.resource.setPath(absoluteFromPath);								
			}
			else {											
				throw new NotExistsResourceException();
			}            
            
            
            boolean answer = this.resource.copyMethod(absoluteToPath);
       		int httpCode = this.resource.getStatusCode();
       		
       		// if a multistatus response
    		if (httpCode == 207 || httpCode == 500)
    			throw new PasteNotAuthorizedResourceException();
       		
    		// if we try to put a document in a not authorized directory
    		if (httpCode == 403)
    			throw new NotAuthorizedUploadException();
       		
    		
       		if (httpCode==507)throw new OverQuotaException(httpCode);
    		
    		if ((httpCode>=400) && (httpCode<600) ) answer=false;
    		
    		if (answer)
    			return true;
    		else {
            	LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::copy : code http : "+httpCode);
            	throw new CopyException(httpCode);
    		}
        }
        catch (HttpException ex) {
        	LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::copy : "+ex);
        	throw new PasteNotAuthorizedResourceException();
        }
        catch (IOException ex) {
        	LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::copy : "+ex);
        	throw new CopyException();
        }         
	}
	
	
	
	
	// MODIFIEE
	/**
	 * Move method
	 * @param resource resource to move
	 * @param fromPath path where is the resource
	 * @param toPath path to move the resource
	 * @return boolean true if well done else false
	 * @throws ApplicationException
	 * @throws MoveException
	 * @throws NotAuthorizedException
	 * @throws NotExistsResourceException
	 * @throws PasteNotAuthorizedResourceException
	 * @throws NotAuthorizedUploadException
	 */
	public boolean move(String resource, String fromPath, String toPath) throws ApplicationException, MoveException, PasteNotAuthorizedResourceException, NotExistsResourceException, PasteNotAuthorizedResourceException, NotAuthorizedException, NotAuthorizedUploadException {
		try {						
			
			// check if resource is a directory   
			String realResource = resource;
			boolean isDir = isDirectory(resource, fromPath);
            if (isDir)
            	realResource = realResource+"/"; 			
			
			// absolute path of the resource
			String absoluteFromPath = fromPath+realResource;
			
			// absolute path to copy the resource
			String absoluteToPath = toPath;
			if (!toPath.endsWith("/"))
				absoluteToPath += "/";
			absoluteToPath += realResource;
            
			// fix the path
			if (exists(absoluteFromPath)) {
				this.resource.setPath(absoluteFromPath);								
			}
			else {											
				throw new NotExistsResourceException();
			} 

            boolean answer = this.resource.moveMethod(absoluteToPath);
       		int httpCode = this.resource.getStatusCode();
       		
       		// if a multistatus response
    		if (httpCode == 207 || httpCode == 500)
    			throw new PasteNotAuthorizedResourceException();
       		
    		// if we try to put a document in a not authorized directory
    		if (httpCode == 403)
    			throw new NotAuthorizedUploadException();
       		    		
    		
    		if ((httpCode>=400) && (httpCode<600) ) answer=false;
    		if (answer)
    			return true;
    		else {
            	LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::move : code http : "+httpCode);
            	throw new MoveException(httpCode);
    		}
    		
        }
        catch (HttpException ex) {
         	LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::move : "+ex);
        	throw new PasteNotAuthorizedResourceException();
        }
        catch (IOException ex) {
        	LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::move : "+ex);
        	throw new MoveException();
        }               
	}
	
	
	
	/**
	 * Return the resource associated to the file to download
	 * @param name the file name
	 * @param path the path where to find the file
	 * @return the ChannelWebdavResource
	 * @throws DownloadException
	 */
	public ChannelWebdavResource dowload(String name,String path) throws DownloadException {
		
		try {
			this.init(this.uri,this.login,this.password);
		} 
		catch (MalformedURLException e2) {
			LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::setFileToDownload MalformedURLException: "+e2);
		}
		
		try {				        					
			WebdavResource fileToDownloadResource = new WebdavResource(httpURL);
			fileToDownloadResource.setPath(path+name);
			
			return new ChannelWebdavResource( fileToDownloadResource.getName(), fileToDownloadResource.getGetContentLength(), fileToDownloadResource.getGetContentType(), fileToDownloadResource.getGetLastModified(), fileToDownloadResource.isCollection(), fileToDownloadResource.getMethodData());						
		}
		catch(HttpException e){			
	     	LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::setFileToDownload HttpException: "+e);
	     	throw new DownloadException();
		}
		catch(IOException e){			
	     	LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::setFileToDownload IOException: "+e);
	     	throw new DownloadException();
		}
		
		
	}
	
	
	
	
	
	/************************
	 * Webdav control methods
	 ************************/
	
	/**
	 * Check is the resource exists
	 * @param resourceName name of the file/directory to verify
	 * @param path path to find the file/directory
	 * @return true if file/directory exists else false
	 * @throws ApplicationException
	 * @throws NotAuthorizedException
	 */
	
	/*public boolean exists(String resourceName, String path) throws ApplicationException, NotAuthorizedException {				
			String pathAndName = path+resourceName;			
			return exists(pathAndName);
	}*/
	
	
	/**
	 * Check is the resource exists
	 * @param fullPath path to find the file/directory
	 * @return true if file/directory exists else false
	 * @throws ApplicationException
	 * @throws NotAuthorizedException
	 */
	/*public boolean exists(String fullPath) throws ApplicationException, NotAuthorizedException {
			
			try {
				// check if this is a valid resource
				try {
					
					this.resource.setPath(fullPath);	
					
					if (this.resource.exists()) {						
						return true;
					}
				}
				catch(HttpException e) {					
				}
				
				// try with a "/"				
				this.resource.setPath(fullPath+"/");
				
				if (this.resource.exists()) {					
					return true;
				}
				else {
					return false;
				}			
			}
			catch (HttpException e) {					
				LogService.log(LogService.DEBUG, "CWebdav::DavAccessImpl::isExist:"+e);
				return false;
				//throw new NotAuthorizedException();
			}
			catch (IOException e) {					
				LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::isExist:"+e);
				throw new ApplicationException();
			}			
	}*/
		
	
	
	/**
	 * Check is the resource exists
	 * @param resourceName name of the file/directory to verify
	 * @param path path to find the file/directory
	 * @return true if file/directory exists else false
	 * @throws ApplicationException
	 * @throws NotAuthorizedException
	 */
	
	public boolean exists(String resourceName, String path) throws ApplicationException, NotAuthorizedException {				
			String pathAndName = path+resourceName;			
			return exists(pathAndName);
	}
	
	
	/**
	 * Check is the resource exists
	 * @param fullPath path to find the file/directory
	 * @return true if file/directory exists else false
	 * @throws ApplicationException
	 * @throws NotAuthorizedException
	 */
	public boolean exists(String fullPath) throws ApplicationException, NotAuthorizedException {
			
			try {
							
				this.resource.setPath(fullPath);
				
				if (this.resource.exists()) {					
					return true;
				}
				else {
					return false;
				}			
			}
			catch (HttpException e) {					
				LogService.log(LogService.DEBUG, "CWebdav::DavAccessImpl::isExist:"+e);
				return false;
			}
			catch (IOException e) {					
				LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::isExist:"+e);
				throw new ApplicationException();
			}			
	}
	
	
	
	
	
	/**
	 * Check if we are authorized to read this resource
	 * @param resourceName name of the file/directory to verify
	 * @param path path to find the file/directory
	 * @return true if authorized else false
	 * @throws ApplicationException
	 */
	public boolean canRead(String resourceName, String path) throws ApplicationException {
		
		try {
			
			String fullPath = path+resourceName;
			boolean exists = exists(fullPath);
			 if(!fullPath.endsWith(("/")) && !exists){
                fullPath=fullPath+"/";
                exists = exists || exists(fullPath);
            }
						
			if (exists)
				return true;
			else return false;
		}
		catch (NotAuthorizedException e) {			
			LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::isAuthorized:"+e);
			return false;
		}
		
	}
	
	
	/**
	 * Check if we are authorized to read this resource
	 * @param path path to find the file/directory
	 * @return true if authorized else false
	 * @throws ApplicationException
	 */
	public boolean canRead(String path) throws ApplicationException {
		
		try {
			String fullPath = path;			
			boolean exists = exists(fullPath);
			 if(!fullPath.endsWith(("/")) && !exists){
                fullPath=fullPath+"/";
                exists = exists || exists(fullPath);
            }
			 
			if (exists)
				return true;
			else return false;
		}
		catch (NotAuthorizedException e) {			
			LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::isAuthorized:"+e);
			return false;
		}
		
	}
	
	/**
	 * Check if the resource named name is a directory in the path given
	 * @param name the name of the resource
	 * @param path the path
	 * @return true if directory, false else
	 * @throws ApplicationException
	 * @throws NotAuthorizedException
	 * @throws NotExistsResourceException
	 */
	public boolean isDirectory(String name, String path) throws ApplicationException, NotAuthorizedException, NotExistsResourceException {
		
		try {
			// the full path used without "/"
			String fullPath = path+name;			
			
			if (exists(fullPath)) {
				this.resource.setPath(fullPath);								
			}
			else {	
				
				if(!fullPath.endsWith(("/"))){
					fullPath=fullPath+"/";
				}
				
				if (exists(fullPath)) {
					this.resource.setPath(fullPath);								
				}
				else {				
					throw new NotExistsResourceException();		
				}
			}
			
			
			
			if (this.resource.isCollection()) {
				return true;
			}
			else {
				return false;				
			}				
		}				
		catch (HttpException e) {			
			LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::isDirectory:"+e);
			throw new NotAuthorizedException();
		}
		catch (IOException e) {			
			LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::isDirectory:"+e);
			throw new ApplicationException();
		}			
	}
	
	
	
	/**
	 * Check if the directory named dir is empty or not. We suppose that dir has been checked as a directory before
	 * @param dir the name of the directory
	 * @param path the path
	 * @return true if empty, false else
	 * @throws ApplicationException
	 * @throws NotExistsResourceException
	 * @throws NotAuthorizedException
	 */
	public boolean isEmpty (String dir, String path) throws ApplicationException, NotExistsResourceException, NotAuthorizedException {		
				
		try {			
			String pathToList;
			
			// we construct the path
			if (!dir.endsWith("/"))
				pathToList = path+dir+"/";
			else pathToList = path+dir;
			
			// we count the number of resources
			if (ls(pathToList).length > 0){
				return false;
			}
			else {
				return true;
			}//end if							
		}
		catch (ApplicationException e) {
			LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::isEmpty:"+e);
			throw new ApplicationException();
		}		
	}	
	
	
	
	
	
	
	/*********************
	 * Properties methods
	 *********************/
	
	
	/**
	 * Get a resource's property specified by the path
	 * @param path the resource path
	 * @param propertyName the property name
	 * @return a vector containing the properties
	 * @throws ServerException
	 */
	public Vector getProperty(String path, String propertyName) throws ServerException {
		Vector properties = new Vector(1);
		properties.addElement(propertyName);		
	   
	    return getProperty(path, properties);
	}

	
	/**
	 * Get a resource's property specified by the path
	 * @param path the resource path
	 * @param properties a vector containing the properties name
	 * @return a vector containing the properties
	 * @throws ServerException
	 */
	public Vector getProperty(String path, Vector properties) throws ServerException {
		try {
			PropFindMethod method = new PropFindMethod(path, DepthSupport.DEPTH_INFINITY, properties.elements());
			this.resource.retrieveSessionInstance().executeMethod(method);
			Enumeration props = method.getResponseProperties(path);
			
			Vector res = new Vector();
			while (props.hasMoreElements()) {
				Object obj = props.nextElement();			
				BaseProperty p = (BaseProperty)obj;				
				Node node = p.getElement().getFirstChild();
				String s = null;
				if (node!=null)
					s = node.getNodeValue();
				res.addElement(s);
			}	    
			return res;
		}
		catch(IOException e) {
			LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::getProperty:"+e);
			throw new ServerException();
		}
	}

	/**
	 * Set the property of resource
	 * @param path the resource path
	 * @param propertyName the name of the property to set
	 * @param propertyValue the value of the property to set
	 * @return true if well done
	 * @throws ServerException
	 */	
	public boolean setProperty(String path, String propertyName, String propertyValue) throws ServerException {
		try {
			return this.resource.proppatchMethod(path, new PropertyName("DAV:", propertyName), propertyValue, true);
		}
		catch(HttpException e) {
			LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::setProperty:"+e);
			throw new ServerException();
		}
		catch(IOException e) {
			LogService.log(LogService.ERROR, "CWebdav::DavAccessImpl::setProperty:"+e);
			throw new ServerException();
		}
	}

	
	
	
	
	
}
