/*
 * Copyright 1999-2002,2004-2005 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.cocoon.ojb.components;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.thread.ThreadSafe;

import org.apache.ojb.broker.util.configuration.Configuration;
import org.apache.ojb.broker.util.configuration.ConfigurationException;
import org.apache.ojb.broker.util.logging.Logger;

/**
 * OJB logger implementation delegating to the Avalon logger
 *
 * @author <a href="mailto:vgritsenko@apache.org">Vadim Gritsenko</a>
 * @version $Id: LoggerImpl.java 154169 2005-02-17 16:58:11Z vgritsenko $
 */
public class LoggerImpl extends AbstractLogEnabled
                                 implements Component, ThreadSafe, Initializable,
                                            Logger {

    /**
     * Root logger for all OJB loggers
     */
    private static org.apache.avalon.framework.logger.Logger LOGGER;

    private final String name;
    private transient int level;
    private transient org.apache.avalon.framework.logger.Logger logger;

    /**
     * Constructor used by Container
     */
    public LoggerImpl() {
        this.name = null;
    }

    /**
     * Constructor used by OJB to create a logger instance
     */
    public LoggerImpl(String name) {
        this.name = name.startsWith("org.apache.ojb.")? name.substring(15): name;
    }

    /**
     * Set root logger instance which will be used by OJB
     */
    public void initialize() {
        LOGGER = getLogger();
    }

    protected int getLevel() {
        if (logger == null) {
            this.logger = LOGGER.getChildLogger(this.name);
            if (this.logger.isDebugEnabled()) this.level = DEBUG;
            else if (this.logger.isInfoEnabled()) this.level = INFO;
            else if (this.logger.isWarnEnabled()) this.level = WARN;
            else if (this.logger.isErrorEnabled()) this.level = ERROR;
            else this.level = FATAL;
        }
        return level;
    }

    public String getName() {
        return name;
    }

    public void debug(Object message) {
        if (DEBUG >= getLevel()) {
            logger.debug(toString(message));
        }
    }

    public void debug(Object message, Throwable t) {
        if (DEBUG >= getLevel()) {
            logger.debug(toString(message), t);
        }
    }

    public void safeDebug(String message, Object obj) {
        if (DEBUG >= getLevel()) {
            logger.debug(message + " : " + toString(obj));
        }
    }

    public void safeDebug(String message, Object obj, Throwable t) {
        if (DEBUG >= getLevel()) {
            logger.debug(message + " : " + toString(obj), t);
        }
    }

    public void info(Object message) {
        if (INFO >= getLevel()) {
            logger.info(toString(message));
        }
    }

    public void info(Object message, Throwable t) {
        if (INFO >= getLevel()) {
            logger.info(toString(message), t);
        }
    }

    public void safeInfo(String message, Object obj) {
        if (INFO >= getLevel()) {
            logger.info(message + " : " + toString(obj));
        }
    }

    public void safeInfo(String message, Object obj, Throwable t) {
        if (INFO >= getLevel()) {
            logger.info(message + " : " + toString(obj), t);
        }
    }

    public void warn(Object message) {
        if (WARN >= getLevel()) {
            logger.warn(toString(message));
        }
    }

    public void warn(Object message, Throwable t) {
        if (WARN >= getLevel()) {
            logger.warn(toString(message), t);
        }
    }

    public void safeWarn(String message, Object obj) {
        if (WARN >= getLevel()) {
            logger.warn(message + " : " + toString(obj));
        }
    }

    public void safeWarn(String message, Object obj, Throwable t) {
        if (WARN >= getLevel()) {
            logger.warn(message + " : " + toString(obj), t);
        }
    }

    public void error(Object message) {
        if (ERROR >= getLevel()) {
            logger.fatalError(toString(message));
        }
    }

    public void error(Object message, Throwable t) {
        if (ERROR >= getLevel()) {
            logger.fatalError(toString(message), t);
        }
    }

    public void safeError(String message, Object obj) {
        if (ERROR >= getLevel()) {
            logger.error(message + " : " + toString(obj));
        }
    }

    public void safeError(String message, Object obj, Throwable t) {
        if (ERROR >= getLevel()) {
            logger.error(message + " : " + toString(obj), t);
        }
    }

    public void fatal(Object message) {
        if (FATAL >= getLevel()) {
            logger.fatalError(toString(message));
        }
    }

    public void fatal(Object message, Throwable t) {
        if (FATAL >= getLevel()) {
            logger.fatalError(toString(message), t);
        }
    }

    public void safeFatal(String message, Object obj) {
        if (FATAL >= getLevel()) {
            logger.fatalError(message + " : " + toString(obj));
        }
    }

    public void safeFatal(String message, Object obj, Throwable t) {
        if (FATAL >= getLevel()) {
            logger.fatalError(message + " : " + toString(obj), t);
        }
    }

    public boolean isDebugEnabled() {
        return isEnabledFor(DEBUG);
    }

    public boolean isEnabledFor(int priority) {
        return priority >= getLevel();
    }

    /*
     * @see org.apache.ojb.broker.util.configuration.Configurable#configure(Configuration)
     */
    public void configure(Configuration config) throws ConfigurationException {
    }

    private String toString(Object obj) {
        if (obj != null) {
            try {
                return obj.toString();
            } catch (Throwable throwable) {
                return "BAD toString() impl for " + obj.getClass().getName();
            }
        }

        return "null";
    }
}
