/*
 * LIUS - Lucene Index Update and Search
 * http://sourceforge.net/projects/lius/
 *
 * Copyright (c) 2004, Laval University Library.  All rights reserved.
 *
 * This program is a free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package ca.ulaval.bibl.lius.index;

/**
 * <p>Titre : Lius (Lucene Index, Update and Search)</p>
 * <p>Description : Application Java permettant d'indexer du XML, PDF, HTML, JSP, ASP, PHP, Word, Excel et des Objets Java</p>
 * <p>Copyright : Copyright (c) 2003 Rida Benjelloun</p>
 * @author Rida Benjelloun
 * @e-mail rbenjelloun@hotmail.com
 *         rida.benjelloun@bibl.ulaval.ca
 * @version 0.0.1
 * @date 09-08-2003
 */

import java.util.Collection;

import org.apache.lucene.document.Document;

import ca.ulaval.bibl.lius.Lucene.LuceneActions;
import ca.ulaval.bibl.lius.config.LiusConfig;
import ca.ulaval.bibl.lius.config.LiusConfigBuilder;

/**
 * @author Rida Benjelloun
 */

public abstract  class Indexer {

    /**
     * Mthode retournant un objet de type Lucene document  partir du fichier 
     * indexer et du fichier de configuration de Lius
     * <br/><br/>
     * Method that returns a Lucene Document object from the file to index and
     * the Lius configuration file.
     */
  public Document createLuceneDocument(String file,
                                       String liusXmlConfigFilePath) {
    LiusConfig lc = LiusConfigBuilder.getSingletonInstance().getLiusConfig(
        liusXmlConfigFilePath);
    Document doc = createLuceneDocument(file, lc);
    return doc;
  }

  /**
   * Mthode retournant un objet de type Lucene document  partir du fichier
   *  indexer et du fichier de configuration de Lius exprim sous forme
   * d'objet de type LiusConfig.
   * <br/><br/>
   * Method that returns a Lucene Document object from a file to index and
   * the Lius Configuration as a LiusConfig object.
   */
  public abstract Document createLuceneDocument(String file, LiusConfig lc) ;
      //throws LiusException;

  /**
   * Mthode retournant un objet de type Lucene document  partir du fichier 
   * indexer et d'une collection d'objets de type LiusField. Chaque objet LiusField
   * contient de l'information sur le nom du champs Lucene, le type, etc.
   * <br/><br/>
   * Method that retursn Lucene Document object from a file to index and a Collection
   * of LiusField objects. Each LiusField object contains information about the name
   * of the Lucene field, the type, etc.
   *
   *
   */
  public Document createLuceneDocument(String file, Collection liusFields) {
    LuceneActions la = new LuceneActions();
    Collection coll = getPopulatedCollection(file, liusFields);
    Document doc = la.populateLuceneDoc(coll);
    return doc;
  }

  /**
   * Permet de rcuprer les champs de Lius  partir du fichier de configuration
   * pour effectuer l'indexation.
   * <br/><br/>
   * Get Lius fields from the configuration file for indexation.
   */
  public abstract Collection getLiusFields(LiusConfig lc);

  /**
   * Mthode retournant un objet de type Lucene document  partir du fichier 
   * indexer et d'une collection d'objets de type LiusField. Chaque objet
   * LiusField contient de l'information sur le nom du champs Lucene, le type,
   * etc.
   * <br/><br/>
   * Method that return a Lucene object from the configuration file and a collection
   * of LiusField objects. Each LiusField object contains information about the Lucene
   * field, the type, etc.
   */
  public abstract Collection getPopulatedCollection(Object file, Collection liusFields);

  public abstract Collection getPopulatedCollection(Object file, String liusConfig);

  public abstract Collection getPopulatedCollection(Object file, LiusConfig liusConfig);

}