/*
 * LIUS - Lucene Index Update and Search
 * http://sourceforge.net/projects/lius/
 *
 * Copyright (c) 2004, Laval University Library.  All rights reserved.
 *
 * This program is a free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package ca.ulaval.bibl.lius.index.Excel;

import java.io.*;
import jxl.*;

import org.apache.lucene.document.Document;

import ca.ulaval.bibl.lius.Lucene.LuceneActions;
import ca.ulaval.bibl.lius.config.*;
import ca.ulaval.bibl.lius.index.Indexer;

import java.util.Collection;
import java.util.ArrayList;
import java.util.Iterator;
import ca.ulaval.bibl.lius.config.LiusConfig;



/**
 * Classe permettant d'indexer des fichiers Excel.
 * <br/><br/>
 * Class for indexing Excel documents.
 * @author Rida Benjelloun (rida.benjelloun@bibl.ulaval.ca)
 * */
public class ExcelIndexer
    extends Indexer {

  public Object parse(Object file) {
    String content = "";
    StringBuffer sb = new StringBuffer();
    try {
      Workbook workbook = Workbook.getWorkbook(new File( (String) file));
      Sheet[] sheets = workbook.getSheets();
      for (int i = 0; i < sheets.length; i++) {
        Sheet sheet = sheets[i];
        int nbCol = sheet.getColumns();
        for (int j = 0; j < nbCol; j++) {
          Cell[] cells = sheet.getColumn(j);
          for (int k = 0; k < cells.length; k++) {
            sb.append(cells[k].getContents() + " ");
          }
        }
      }
    }
    catch (IOException e) {
      e.printStackTrace();
    }
    catch (jxl.read.biff.BiffException e) {
      e.printStackTrace();
    }
    content = sb.toString();
    return content;
  }


  /**
   * Mthode retournant un objet de type Lucene document  partir du fichier
   *  indexer et du fichier de configuration de Lius exprim sous forme
   * d'objet de type LiusConfig.
   * <br/><br/>
   * Method that returns a Lucene document object from a file to index and
   * the Lius Configuration as a LiusConfig object.
   */
  public Document createLuceneDocument(String file, LiusConfig lc) {
    Document doc = createLuceneDocument(file, lc.getExcelFields());
    return doc;
  }

  /**
   * Mthode retournant un objet de type Lucene document  partir du fichier 
   * indexer et d'une collection d'objets de type LiusField. Chaque objet
   * LiusField contient de l'information sur le nom du champs Lucene, le type,
   * etc.
   * <br/><br/>
   * Method that returns a Lucene object from the configuration file and a collection
   * of LiusField objects. Each LiusField object contains information about the Lucene
   * field, the type, etc.
   */
  public Collection getPopulatedCollection(Object file, Collection liusFields) {
    LuceneActions la = new LuceneActions();
    Collection coll = new ArrayList();
    Iterator it = liusFields.iterator();
    while (it.hasNext()) {
      LiusField lf = (LiusField) it.next();
      if (lf.getGet() != null) {
        if (lf.getGet().equalsIgnoreCase("content")) {
          String text = (String) parse(file);
          lf.setValue(text);
          coll.add(lf);
        }
      }
    }
    return coll;
  }

  /**
   * Permet de rcuprer les champs de Lius  partir du fichier de configuration
   * pour effectuer l'indexation.
   * <br/><br/>
   * Gets Lius fiels from the configuration file for indexation.
   */
  public Collection getLiusFields(LiusConfig lc) {
    return lc.getExcelFields();
  }

  public Collection getPopulatedCollection(Object file, String liusConfig) {
    LiusConfig lc = LiusConfigBuilder.getSingletonInstance().getLiusConfig(
        liusConfig);
    return getPopulatedCollection(file, lc);
  }

  public Collection getPopulatedCollection(Object file, LiusConfig lc) {
    return getPopulatedCollection(file, lc.getExcelFields());
  }
}