/*
 * LIUS - Lucene Index Update and Search
 * http://sourceforge.net/projects/lius/
 *
 * Copyright (c) 2004, Laval University Library.  All rights reserved.
 *
 * This program is a free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package ca.ulaval.bibl.lius.index.MixteIndexing;


import org.apache.lucene.document.Document;

import ca.ulaval.bibl.lius.Exception.LiusException;
import ca.ulaval.bibl.lius.Lucene.LuceneActions;
import ca.ulaval.bibl.lius.config.LiusConfig;
import ca.ulaval.bibl.lius.config.LiusConfigBuilder;
import ca.ulaval.bibl.lius.index.Indexer;
import ca.ulaval.bibl.lius.index.IndexerFactory;
import java.util.Collection;
import java.util.List;
import java.util.ArrayList;
import java.io.File;
import ca.ulaval.bibl.lius.config.LiusField;
import java.util.Iterator;
import org.apache.commons.beanutils.BeanUtils;
import java.lang.reflect.*;

/**
 * Classe permettant d'effectuer une indexation mixte. Cette
 * indexation permet d'integrer dans le mme "Lucene Document"
 * des mta-donnes dans format XML et le texte integral dans
 * un fichier PDF Word etc.
 * <br/><br/>
 * Class for mixed indexation. This indexation allows for integrating in the
 * same Lucene document XML metadata and full text from a PDF file, Word file, etc.
 * @author Rida Benjelloun (rida.benjelloun@bibl.ulaval.ca)
 */
public class MixteIndexer
    extends Indexer {
  private LiusConfig lcP = null;

  public Object parse(Object file) {
    File f = new File( (String) file);
    String[] lf = null;
    if (f.isDirectory()) {
      lf = f.list();
    }
    return lf;
  }

  /**
   * Mthode retournant un objet de type Lucene document  partir du fichier 
   * indexer et du fichier de configuration de Lius.
   * <br/><br/>
   * Method that returns a Lucene Document object from the file to index and
   * the Lius configuration file.
   */
  public Document createLuceneDocument(String file,
                                       String liusXmlConfigFilePath) {
    LiusConfig lc = LiusConfigBuilder.getSingletonInstance().
        getLiusConfig(liusXmlConfigFilePath);
    lcP = lc;
    Document doc = createLuceneDocument(file, lc);
    return doc;
  }

  /**
   * Mthode retournant un objet de type Lucene document  partir du fichier
   *  indexer et du fichier de configuration de Lius exprim sous forme
   * d'objet de type LiusConfig.
   * <br/><br/>
   * Method that returns a Lucene document object from a file to index and
   * the Lius Configuration as a LiusConfig object.
   */
  public Document createLuceneDocument(String file, LiusConfig lc) {
    lcP = lc;
    Document doc = createLuceneDocument(file, lc.getMixteIndexingElements());
    return doc;
  }

  /**
   * Mthode retournant un objet de type Lucene document  partir du fichier
   *  indexer et d'une collection d'objets de type LiusField. Chaque
   * objet LiusField contient de l'information sur le nom du champs Lucene,
   * le type, etc.
   * <br/><br/>
   * Method that returns a Lucene document object from the file to index
   * and a collection of LiusField objects. Each LiusField object contains
   * information about the name of the Lucene Field, the type, etc.
   */
  public Document createLuceneDocument(String file,
                                       Collection mixteIndexingElements) {
    List populatedList = (List) getPopulatedCollection(file,
        mixteIndexingElements);
   Document doc = LuceneActions.getSingletonInstance().populateLuceneDoc(populatedList);
    return doc;
  }

  /**
   * Retourne une collection contenant les champs avec les valeurs 
   * indexer comme par exemple: le texte integrale, titre etc.
   * <br/><br/>
   * Returns a collection containing the fieds with the values to index,
   * like : full text, title, etc.
   */
  public Collection getPopulatedCollection(Object file,
                                           Collection mixteIndexingElements) {

    String sep = System.getProperty("file.separator");
    String[] lf = (String[]) parse((String)file);
    List populatedList = new ArrayList();

    for (int i = 0; i < lf.length; i++) {
      Indexer indexer = null;
      String fileName = lf[i];
      File fileNameF = new File(fileName);
      String filePath = (String)file + sep + fileName;
      try {
        indexer = IndexerFactory.getIndexer(filePath,
                                            (List) mixteIndexingElements);
        if (indexer != null) {
          if (indexer.getLiusFields(lcP).size() > 0) {
            Collection populCollFile = indexer.getPopulatedCollection(filePath,
                indexer.getLiusFields(lcP));
            Iterator it = populCollFile.iterator();
            while (it.hasNext()) {
              LiusField f = new LiusField();
              f = (LiusField) it.next();
              LiusField newLF = new LiusField();
              try {
                BeanUtils.copyProperties(newLF, f);
              }
              catch (InvocationTargetException ex) {
                ex.printStackTrace();
              }
              catch (IllegalAccessException ex) {
                ex.printStackTrace();
              }
              populatedList.add(newLF);
             }
          }

        }
      }
      catch (LiusException e) {
        e.printStackTrace();
      }
    }
    return populatedList;
  }

  /**
   * Mthode retournant un objet de type Lucene document  partir du fichier 
   * indexer et d'une collection d'objets de type LiusField. Chaque objet
   * LiusField contient de l'information sur le nom du champs Lucene, le type,
   * etc.
   * <br/><br/>
   * Method that return a Lucene object from the configuration file and a collection
   * of LiusField objects. Each LiusField object contains information about the Lucene
   * field, the type, etc.
   */
  public Collection getLiusFields(LiusConfig lc) {
    return null;
  }

  public Collection getPopulatedCollection(Object file, String liusConfig) {
     LiusConfig lc = LiusConfigBuilder.getSingletonInstance().getLiusConfig(
         liusConfig);
     return getPopulatedCollection(file, lc);
   }

   public Collection getPopulatedCollection(Object file, LiusConfig lc) {
     return getPopulatedCollection(file, lc.getMixteIndexingElements());
   }

}
