/**
 * Copyright  2004 ESUP-portail consortium
 * Authors : Nathalie.Vauquier@univ-valenciennes.fr
 * This program is free software; you can redistribute 
 * it and/or modify it under the terms of the GNU 
 * General Public License as published by the Free 
 * Software Foundation augmented according to the
 * possibility of using it with programs developped under 
 * the Apache licence published by the Apache Software Foundation.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 *  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 *  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY 
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, 
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE 
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 * See the license terms site for more details : 
 * http://www.esup-portail.org/license.html
 */

package fr.univvalenciennes.publi.WebServices.adminPubli.GestionCategories.Categories;

import java.util.Vector;

import org.apache.log4j.Logger;

import fr.univvalenciennes.publi.WebServices.adminPubli.GestionProfils.Profils.*;


/**
 * Description :<br/>
 * Encapsule les donnes concernant une entre dans CATEGORIES
 * qui stocke les catgories de la base:<br/>
 * CATEGORIES(uidcat,caption,niveau,summary,gifUrl,needsValidation,mailValidation)
 * @version $Id : Categories.java, V1.0, 09 avril 2004<br/>
 * Copyright (c) 2004 Esup Portail (www.esup-portail.org)<br/>
 * Classe(s) : Categories<br/>
 */
public class Categories implements java.io.Serializable {
	static Logger log;

	//	Champs de la table CATEGORIES
	private String 	uidCat 	= null;
	private String 	caption = null;
	private String 	summary = null;
	private String 	gifUrl 	= null;

	private boolean existe = false; // vrai si la catgorie est dans la BD

	public Categories() {
		log = Logger.getLogger(this.getClass().getPackage().getName());
	}

	/**
	 * Demande le chargement de la catgorie  partir de la base
	 * @param uidCat	uid de la catgorie
	 * @param sqlCat	Objet SqlCategories regroupant les accs  la table CATEGORIES
	 * @throws Exception
	 */
	public Categories(String uidCat) throws Exception {
		log = Logger.getLogger(this.getClass().getPackage().getName());
		setUidCat(uidCat);
		try {
			existe = SqlCategories.loadCategorie(this);
		} catch (Exception e) {
			log.error(
				"Categories : Problme de chargement de la catgorie : " + e);
		}
	}
	
	/**
	 * Demande le chargement de la catgorie  partir de la base
	 * puis affecte les valeurs passes en paramtres
	 * @param uidCat	uid de la catgorie
	 * @param caption	valeur  affecter au champ caption
	 * @param summary	valeur  affecter au champ summary
	 * @param gifUrl	valeur  affecter au champ gifurl
	 * @param sqlCat	Objet SqlCategories regroupant les accs  la table CATEGORIES
	 * @throws Exception
	 */
	public Categories(
		String uidCat,
		String caption,
		String summary,
		String gifUrl)
		throws Exception {
		this(uidCat);

		setCaption(caption);
		setSummary(summary);
		setGifUrl(gifUrl);
	}

	/**
	 * Renvoie le nombre de categories filles de la catgorie courante
	 * dans la table ORGACATEGORIES
	 * @return nombre de categories filles de la catgorie courante
	 * @throws Exception
	 */
	public long getNbFilles() throws Exception {
		try {
			return SqlOrgaCategories.nbCategoriesFilles(this);
		} catch (Exception e) {
			log.error(
				"Categories : Problme getNbFilles() : " + e.getMessage());
			throw (Exception)e;
		}
	}
	
	public Vector getCategoriesFilles() throws Exception {
		return SqlOrgaCategories.getFillesUidcats(this);
	}

	/**
	 * Renvoie sous forme de Vector toutes les categories descendantes
	 * de la catgorie courante (filles, petites-filles, ...)
	 * @return Vector des uidcats des catgories descendantes de la catgorie
	 * @throws Exception
	 */
	public Vector getDescendanceUidcats() throws Exception {
		Vector v = null;
		try {
			v = SqlOrgaCategories.getFillesUidcats(this);
		} catch (Exception e) {
			log.error(
				"Categories : Problme getDescendanceUidCats() : " + e.getMessage());
			throw (Exception)e;
		} finally {
			return v;
		}
	}

	/**
	 * Renvoie sous forme de Vector toutes les categories ascendantes
	 * de la catgorie courante (mres, grand-mres, ...)
	 * @return Vector des uidcats des catgories ascendantes de la catgorie
	 * @throws Exception
	 */
	public Vector getAscendanceUidcats() throws Exception {
		Vector v=null;
		try {
			v = SqlOrgaCategories.getParentesUidcats(this);
		} catch (Exception e) {
			log.error(
				"Categories : Problme getAscendanceUidcats() : " + e.getMessage());
			throw (Exception)e;
		} finally {
			return v;
		}
	}

	
	/**
	 * Renvoie sous forme de Vector tous les profils attachs 
	 *  la catgorie courante
	 * @return Vector des uidprof des profils attachs
	 * @throws SQLException
	 * @throws ClassNotFoundException
	 */
	public Vector getProfilsAttaches() throws Exception {
		Vector v = null;
		try {
			v = SqlCatsProfils.getProfilsAttaches(this);
		} catch (Exception e) {
			log.error(
				"Categories : Problme getProfilsAttaches() : " + e.getMessage());
			throw (Exception)e;
		} finally {
			return v;
		}
		
	}
	
	/**
	 * Publie la catgorie en tant que canal RSS
	 * @param owner		valeur  affecter au champ owner
	 * @throws Exception
	 */
	public void publishAsRss(String owner)
		throws Exception {
		try {
			if (existe) {
				CanauxRss canal =
					new CanauxRss(
						getUidCat(),
						getCaption(),
						owner);
				if (isPublishedAsRss())
					SqlCanauxRss.modifCanauxRss(canal);
				else
					SqlCanauxRss.createCanauxRss(canal);
			}
		} catch (Exception e) {
			log.error(
				"Categories : Problme publishAsRss : " + e.getMessage());
			throw (Exception)e;
		}
	}

	/**
	 * Dtermine si la catgorie est un canal RSS
	 * @return true si la catgorie est dans la table CANAUXRSS
	 * @throws Exception
	 */
	public boolean isPublishedAsRss() throws Exception {
		boolean b=false;
		try {
			b = SqlCanauxRss.isPublishedAsRss(this);
		} catch (Exception e) {
			log.error(
				"Categories : Problme isPublishedAsRss() : " + e.getMessage());
			throw (Exception)e;
		} finally {
			return b;
		}
	}

	/**
	 * Dpublie la catgorie en tant que canal RSS
	 * @throws Exception
	 */
	public void unpublishAsRss() throws Exception {
		try {
			if (isPublishedAsRss()) {
				CanauxRss can = new CanauxRss(getUidCat());
				can.delete();
			}
		} catch (Exception e) {
			log.error(
				"Categories : Problme unpublishAsRss() : " + e.getMessage());
			throw (Exception)e;
		}
	}

	public void create() throws Exception{
		SqlCategories.createCategories(this);
	}
	
	public void modif() throws Exception {
		SqlCategories.modifCategories(this);
	}
	
	public static Vector getAllAvailableGifs() throws Exception {
		return SqlCategories.getAllAvailableGifs();
	}
	
	public static Vector getAllCategories(Users user) throws Exception {
		return SqlCategories.getAllCategories(user);
	}
	
	/**
	 * Supprime la catgorie de la base :
	 * Si c'est un canal RSS, la dpublie
	 * Si des news y sont attaches, les supprime
	 * Supprime les liens avec les autres catgories
	 * Enfin, supprime la catgorie
	 * @throws Exception
	 */
	public void delete() throws Exception {
		try {
			if (isExiste()) {
				// si la catgorie est un canal RSS, la dpublier
				unpublishAsRss();
			
				// supprimer les liens o la catgorie est mre
				SqlOrgaCategories.deleteChildCatLinks(this);
			
				// supprimer les liens o la catgorie est fille
				SqlOrgaCategories.deleteParentCatLinks(this);
				
				// si la catgories est attache  des profils, supprimer ces liens
				SqlCatsProfils.deleteCatsLinks(this);
				
				// supprimer la catgorie
				SqlCategories.deleteCategories(this);
			
			}
		}  catch (Exception e) {
			log.error(
				"Categories : Problme delete() : " + e.getMessage());
			throw (Exception)e;
		}
	}

	public String getCaption() {
		return caption;
	}

	public String getGifUrl() {
		return gifUrl;
	}

	public String getSummary() {
		return summary;
	}

	public String getUidCat() {
		return uidCat;
	}

	public void setCaption(String string) {
		if (string.length() > 80)
			string = string.substring(0, 80);
		caption = string;
	}

	public void setGifUrl(String string) {
		if (string.length() > 500)
			string = string.substring(0, 500);
		gifUrl = string;
	}

	public void setSummary(String string) {
		if (string.length() > 500)
			string = string.substring(0, 500);
		summary = string;
	}

	public void setUidCat(String string) {
		if (isExiste())
			return;
		// on ne peut pas changer l'uidCat d'une catgorie existante
		uidCat = string;
	}

	public boolean isExiste() {
		return existe;
	}

	public void setExiste(boolean b) {
		existe = b;
	}
}