/**
 * Copyright  2004 ESUP-portail consortium
 * Authors : Nathalie.Vauquier@univ-valenciennes.fr
 * This program is free software; you can redistribute 
 * it and/or modify it under the terms of the GNU 
 * General Public License as published by the Free 
 * Software Foundation augmented according to the
 * possibility of using it with programs developped under 
 * the Apache licence published by the Apache Software Foundation.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 *  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 *  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY 
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, 
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE 
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 * See the license terms site for more details : 
 * http://www.esup-portail.org/license.html
 */

package fr.univvalenciennes.publi.WebServices.adminPubli.GestionCategories.Categories;

import java.sql.*;
import java.util.Vector;

import org.apache.log4j.Logger;

import fr.univvalenciennes.publi.utils.db.CachedConnectionsManager;
import fr.univvalenciennes.publi.utils.properties.*;

/**
 * Description :<br/>
 * Classe regroupant les accs  la base de donnes
 * concernant la table ORGACATEGORIES qui stocke les liens 
 * de descendance entre les catgories :
 * ORGACATEGORIES(uidcat,uiddesccat,ordre)
 * @version $Id : SqlCategories.java, V1.0, 09 avril 2004<br/>
 * Copyright (c) 2004 Esup Portail (www.esup-portail.org)<br/>
 * Classe(s) : SqlOrgaCategories<br/>
 */
public class SqlOrgaCategories {
	static String pool = "applinews";
	static Logger log = Logger.getLogger("administration");

	public SqlOrgaCategories() {
	}

	private static Connection getConnection() throws Exception {
		return CachedConnectionsManager.checkOut(pool);
	}

	/**
	 * Ajoute le lien pass en paramtre dans la base
	 * @param lien Lien  ajouter  la base
	 * @throws SQLException
	 * @throws ClassNotFoundException
	 */
	public static void createOrgaCategories(OrgaCategories lien)
		throws Exception {
		Connection conn = null;
		PreparedStatement statement	= null;
		try {
			if (lien.getOrdre() > 0) {
				conn = getConnection();
				new PropertiesManager("requetesAdmin.properties");
				String sqlString = PropertiesManager.getProperty("organisationcanaux.creation");
				statement = conn.prepareStatement(sqlString);
				statement.setString(1, lien.getCategorieMere().getUidCat());
				statement.setString(2, lien.getCategorieFille().getUidCat());
				statement.setLong(3, lien.getOrdre());
				statement.executeUpdate();
				statement.close();
				lien.setExiste(true);
			} else {
				if (lien.getOrdre() <= 0)
					log.debug(
						"SqlOrgaCategories - lien non valide : ordre = "
							+ lien.getOrdre());
			}

		} catch (Exception e) {
			log.error("SqlOrgaCategories - createOrgaCategories : " + e);
			throw e;
		} finally {
			if (statement != null) statement.close();
			CachedConnectionsManager.checkIn(conn);
		}
	}

	/**
	 * Modifie le lien pass en paramtre dans la base
	 * @param lien	Lien  modifier
	 * @throws SQLException
	 * @throws ClassNotFoundException
	 */
	public static void modifOrgaCategories(OrgaCategories lien)
		throws Exception {

		Connection conn = null;
		PreparedStatement statement = null;
		try {
			if (lien.isExiste()) {
				conn = getConnection();
				new PropertiesManager("requetesAdmin.properties");
				String sqlString = PropertiesManager.getProperty("organisationcanaux.modification");

				statement = conn.prepareStatement(sqlString);

				statement.setLong(1, lien.getOrdre());
				statement.setString(2, lien.getCategorieMere().getUidCat());
				statement.setString(3, lien.getCategorieFille().getUidCat());
				statement.executeUpdate();
				statement.close();
			}
		} catch (Exception e) {
			log.error("SqlOrgaCategories - modifOrgaCategories : " + e);
			throw e;
		} finally {
			if (statement != null) statement.close();
			CachedConnectionsManager.checkIn(conn);
		}
	}

	/**
	 * Supprime de la base le lien pass en paramtre
	 * @param lien	Lien  supprimer
	 * @throws SQLException
	 * @throws ClassNotFoundException
	 */
	public static void deleteOrgaCategories(OrgaCategories lien)
		throws Exception {
		Connection conn = null;
		PreparedStatement stmt = null;
		String sqlString;

		try {
			conn = getConnection();
			new PropertiesManager("requetesAdmin.properties");
			sqlString = PropertiesManager.getProperty("organisationcanaux.suppression");
			stmt = conn.prepareStatement(sqlString);
			stmt.setString(1,lien.getCategorieMere().getUidCat());
			stmt.setString(2,lien.getCategorieFille().getUidCat());
			stmt.executeUpdate();
			stmt.close();
		} catch (Exception e) {
			log.error("SqlOrgaCategories - deleteOrgaCategories : " + e);
			throw e;
		} finally {
			if (stmt != null) stmt.close();
			CachedConnectionsManager.checkIn(conn);
		}
	}

	/**
	 * Charge le lien s'il est dans la base
	 * On peut charger un lien dont on connait la catgorie Mre et la catgorie Fille
	 * o un lien dont on connait la catgorie Mre et l'ordre
	 * @param lien	Lien  charger
	 * @return true si le lien est dans la base
	 * @throws SQLException
	 */
	public static boolean loadOrgaCategories(OrgaCategories lien) throws Exception {
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet rst = null;
		String sqlString;
		try {
			if (lien.getOrdre() == 0)
			// cas o on charge un lien entre 2 catgories connues
			try {
				conn = getConnection();
				new PropertiesManager("requetesAdmin.properties");
				sqlString = PropertiesManager.getProperty("organisationcanaux.chargement1");
				stmt = conn.prepareStatement(sqlString);
				stmt.setString(1,lien.getCategorieMere().getUidCat());
				stmt.setString(2,lien.getCategorieFille().getUidCat());
				rst = stmt.executeQuery();

				if (rst.next()) {
					lien.setOrdre(rst.getInt("ordre"));
					stmt.close();
					return true;
				} else {
					stmt.close();
					return false;
				}

			} catch (SQLException e) {
				log.error(
					"SqlOrgaCategories - loadOrgaCategories : "	+ e);
				throw (Exception)e;
			} else
			// cas o on charge un lien dont on connait la catgorie mre et l'ordre
				try {
					conn = getConnection();
					new PropertiesManager("requetesAdmin.properties");
					sqlString = PropertiesManager.getProperty("organisationcanaux.chargement2");
					stmt = conn.prepareStatement(sqlString);
					stmt.setString(1,lien.getCategorieMere().getUidCat());
					stmt.setLong(2,lien.getOrdre());

					rst = stmt.executeQuery();
					if (rst.next()) {
						Categories catFille =
							new Categories(rst.getString("uiddesccat"));
						lien.setCategorieFille(catFille);
						stmt.close();
						return true;
					} else {
						stmt.close();
						return false;
					}
	
				} catch (SQLException e) {
					log.error(
						"SqlOrgaCategories - loadOrgaCategories : "	+ e);
					throw (Exception)e;
				}
		} finally {
			if (rst != null) rst.close();
			if (stmt != null) stmt.close();
			CachedConnectionsManager.checkIn(conn);
		}
	}

	/**
	 * Renvoie le nombre de catgories filles d'une catgorie
	 * @param cat	Catgorie
	 * @return nombre de catgories filles de la categorie
	 * @throws SQLException
	 */
	public static long nbCategoriesFilles(Categories cat) throws Exception {
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet rst = null;
		String sqlString;

		try {
			conn = getConnection();
			new PropertiesManager("requetesAdmin.properties");
			sqlString = PropertiesManager.getProperty("organisationcanaux.nbCanauxFils");
			stmt = conn.prepareStatement(sqlString);
			stmt.setString(1,cat.getUidCat());
			rst = stmt.executeQuery();

			if (rst.next()){
				long n = rst.getLong(1);
				stmt.close();
				return n;				
			}
			else {
				stmt.close();
				return 0;
			}				
		} catch (Exception e) {
			log.error(
				"SqlOrgaCategories - nbCategoriesFilles : "
					+ e);
			throw e;
		} finally {
			if (rst != null) rst.close();
			if (stmt != null) stmt.close();
			CachedConnectionsManager.checkIn(conn);
		}
	}

	/**
	 * Renvoie les categories directement parentes d'une categorie
	 * @param uidCat	uid de la catgorie
	 * @return Vector des uidcat des categories directement parentes de la categorie
	 * @throws Exception
	 */
	public static Vector getParentesUidcats(Categories cat) throws Exception {
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet rst = null;
		String sqlString;
		Vector vParentes = new Vector(0);

		try {
			conn = getConnection();
			new PropertiesManager("requetesAdmin.properties");
			sqlString = PropertiesManager.getProperty("organisationcanaux.canauxPeres");
			stmt = conn.prepareStatement(sqlString);
			stmt.setString(1,cat.getUidCat());
			rst = stmt.executeQuery();

			while (rst.next())
				vParentes.addElement(rst.getString("uidcat"));
			
			stmt.close();
			return vParentes;
		} catch (Exception e) {
			log.error(
				"SqlOrgaCategories - getParentesUidcats : " + e);
			throw e;
		} finally {
			if (rst != null) rst.close();
			if (stmt != null) stmt.close();
			CachedConnectionsManager.checkIn(conn);
		}
	}
	
	/**
	 * Renvoie les categories directement filles d'une categorie
	 * @param uidCat uid de la catgorie
	 * @return Vector des uidcat des categories directement filles de la categorie
	 * @throws Exception
	 */
	public static Vector getFillesUidcats(Categories cat) throws Exception {
		return getFillesUidcats(cat,0);
	}
	
	/**
	 * Renvoie les categories directement filles d'une catgorie avec un ordre
	 * suprieur  celui pass en paramtre
	 * @param uidCat	uid de la catgorie
	 * @param ordre		ordre minimal
	 * @return Vector des uidcat des categories directement filles de la categorie d'ordre
	 * suprieur  celui pass en paramtre
	 * @throws Exception
	 */
	public static Vector getFillesUidcats(Categories cat, long ordre) throws Exception {
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet rst = null;
		String sqlString;
		Vector vFilles = new Vector(0);

		try {
			conn = getConnection();
			new PropertiesManager("requetesAdmin.properties");
			sqlString = PropertiesManager.getProperty("organisationcanaux.canauxFils");
			stmt = conn.prepareStatement(sqlString);
			stmt.setString(1,cat.getUidCat());
			stmt.setLong(2,ordre);
			rst = stmt.executeQuery();

			while (rst.next()) {
				vFilles.addElement(rst.getString("uiddesccat"));
			}
			stmt.close();
			return vFilles;
		} catch (Exception e) {
			log.error(
				"SqlOrgaCategories - getFillesUidcats : " + e);
			throw e;
		} finally {
			if (rst != null) rst.close();
			if (stmt != null) stmt.close();
			CachedConnectionsManager.checkIn(conn);
		}
	}

	/**
	 * Supprime de ORGACATEGORIES les liens o la catgorie passe en paramtre 
	 * est mre
	 * @param cat Catgorie dont on veut supprimer les liens
	 */
	public static void deleteChildCatLinks(Categories cat) throws Exception {
		Connection conn = null;
		PreparedStatement stmt  = null;
		PreparedStatement stmt2 = null;
		ResultSet rst  = null;
		ResultSet rst2 = null;

		try {
			String sqlString;
			conn = getConnection();
			new PropertiesManager("requetesAdmin.properties");
			sqlString = PropertiesManager.getProperty("organisationcanaux.tousLesFils");
			stmt = conn.prepareStatement(sqlString);
			stmt.setString(1,cat.getUidCat());

			// on rcupre les uidcat des filles de la catgorie
			rst = stmt.executeQuery();
			
			while (rst.next()) {
				String uidcat=rst.getString("uiddesccat");
				Categories catF = new Categories(uidcat);
				
				// on supprime le lien
				OrgaCategories lien = new OrgaCategories(cat.getUidCat(),catF.getUidCat());
				lien.delete();
				
				// on va supprimer les catgories qui ne seront plus rattaches
				//  aucune catgorie aprs la suppression du lien
				String sqlString2 = PropertiesManager.getProperty("organisationcanaux.canauxPeres");
				stmt2 = conn.prepareStatement(sqlString2);
				stmt2.setString(1,catF.getUidCat());
				rst2=stmt2.executeQuery();
				if(rst2 == null) {
					log.error("**** SqlOrgaCategories - deleteChildCatLinks() :: "+catF.getUidCat()+" est orphelin");
					// catF est orpheline, on supprime la catgorie
					catF.delete();
				}
			}
		} catch (Exception e) {
			log.error("SqlOrgaCategories - deleteChildCatLinks : " + e);
			throw e;
		} finally {
			if (rst != null) rst.close();
			if (rst2 != null) rst2.close();
			if (stmt != null) stmt.close();
			if (stmt2 != null) stmt2.close();
			CachedConnectionsManager.checkIn(conn);
		}
	}

	/**
	 * Supprime de ORGACATEGORIES les liens o la catgorie passe en paramtre 
	 * est fille
	 * @param cat	Catgorie dont on veut supprimer les liens
	 */
	public static void deleteParentCatLinks(Categories cat) throws Exception {
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet rst = null;

		try {
			String sqlString;
			conn = getConnection();
			new PropertiesManager("requetesAdmin.properties");
			sqlString = PropertiesManager.getProperty("organisationcanaux.canauxPeres");
			stmt = conn.prepareStatement(sqlString);
			stmt.setString(1,cat.getUidCat());
			rst = stmt.executeQuery();
			
			while(rst.next()) {
				OrgaCategories lien = new OrgaCategories(rst.getString("uidcat"), cat.getUidCat());
				lien.delete();
			}

		} catch (Exception e) {
			log.error("SqlOrgaCategories - deleteParentCatLinks : " + e);
			throw e;
		} finally {
			if (rst != null) rst.close();
			if (stmt != null) stmt.close();
			CachedConnectionsManager.checkIn(conn);
		}
	}

}

