package fr.univvalenciennes.publi.utils.db;

import java.sql.*;
import java.util.Vector;

import fr.univvalenciennes.publi.utils.properties.PropertiesManager;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
/**
 * @author Nathalie Vauquier
 *
 * Classe qui se charge d'allouer les connexions du pool et de 
 * fermer les connexions
 */
public class CachedConnectionsManager {
	private static boolean verbose = true;
	private static Log log =  LogFactory.getLog("SqlUsers");
	private static int numberConnections = 0;	// nombre de connexions ouvertes
	private static Vector cachedConnections = new Vector();	// Vector des connexions ouvertes
	private static Thread monitor = null;	// Thread qui s'occupe de supprimer les connexions non utilises
	
	private static boolean init = false;	// permet de savoir si les valeurs suivantes ont t initialises
	private static long MAX_IDLE ;			// delai d'inactivite avant fermeture d'une connexion
	private static long FREQ;				// frequence de verification du temps d'inactivite des connexions
	private static long MAX_CON;			// nombre de connexions maximal dans le pool
	
	/**
	 * Alloue une connection  partir du pool pass en paramtre
	 * @param baseName
	 * @return
	 */
	synchronized public static Connection checkOut(String baseName) throws Exception {
		boolean found = false;
		CachedConnection cached = null;
		
		if(!init) {
			if(verbose) {
				log.info("** Initialisation du CachedConnectionsManager **");
				log.info("CachedConnectionsManager : fichier properties : "+baseName + ".properties");
			}
			new PropertiesManager(baseName + ".properties");//directory,baseName + ".properties");
			MAX_IDLE = Long.parseLong(PropertiesManager.getProperty("pool.max_idle"));
			FREQ = Long.parseLong(PropertiesManager.getProperty("pool.frequence"));
			MAX_CON = Long.parseLong(PropertiesManager.getProperty("pool.max_con"));
			verbose = PropertiesManager.getProperty("pool.verbose").equals("true");
			init = true;
		}
		
		if(verbose) {
			log.info("Il y a " +
				Integer.toString(numberConnections) +
				" connexions dans le cache");
			log.info("Recherche d'une connexion en attente ...");
		}
		for (int i=0;!found&& i<numberConnections;i++) {
			if(verbose) {
				log.info("Connexion " + Integer.toString(i));
			}
			cached = (CachedConnection)cachedConnections.get(i);
			if(!cached.isInUse()&&cached.getBaseName().equals(baseName)) {
				if(verbose) {
					log.info("connexion trouvee " + Integer.toString(i) + 
						" pour " + baseName);
				}
				found = true;
			}
		}
		
		if(found){
			cached.setInUse(true);
		} else {
			if(numberConnections<MAX_CON) {
				if(verbose) {
					log.info("Il n'y a pas de connexion libre");
					log.info("Creation d'une nouvelle connexion pour " + baseName);
				}
				cached = new CachedConnection(CachedConnectionCreator.getConnection(baseName,verbose),true,baseName);
				cachedConnections.add(cached);
				numberConnections++;
			} else {
				throw new CachedConnectionException();
			}
		}
		
		if (monitor==null) {
			monitor = new Thread(
							new Runnable(){
								public void run(){
									while(numberConnections>0){
										runMonitor();
									}
									monitor=null;
								}
							}
					);
			monitor.setDaemon(true);
			monitor.start();
		}
		return cached.getConn();
	}
	
	/**
	 * Remet la connexion passe en paramtre dans le cache
	 * @param c
	 */
	synchronized public static void checkIn(Connection c){
		boolean found = false;
		boolean closed = false;
		CachedConnection cached = null;
		Connection conn = null;
		int i = 0;
		
		if(verbose) {
			log.info("Recherche de la connexion a mettre en cache");
		}
		for(i=0;!found && i<numberConnections;i++) {
			if(verbose){
				log.info("Connexion " + Integer.toString(i));
			}
			cached = (CachedConnection) cachedConnections.get(i);
			conn = cached.getConn();
			if (conn == c) {
				if(verbose) {
					log.info("Connexion trouvee " + Integer.toString(i));
				}
				found = true;
			}
		}
		
		if (found) {
			try {
				closed = (conn==null || conn.isClosed());
			} catch (SQLException e) {
				closed=true;
			}
			
			if (!closed)
				cached.setInUse(false);
			else {
				cachedConnections.remove(i);
				numberConnections--;
			}
		} else if (verbose) {
			log.info("La connexion n'a pas ete trouvee !!!");
		}
	}
	
	/**
	 * Mthode appele par le thread monitor et charge 
	 * de fermer les connexions aprs la dure d'inactivit 
	 * spcifie par MAX_IDLE 
	 */
	synchronized private static void checkUse() {
		CachedConnection cached = null;
		Connection conn = null;
		int i = 0;
		long now = System.currentTimeMillis();
		long then = 0;


		for (i=numberConnections-1;i>-1;i--) {
			if (verbose) {
				log.info("Verification par monitor de l'utilisation de " +
					"la connexion " + Integer.toString(i));
			}
			cached = (CachedConnection) cachedConnections.get(i);
			if (!cached.isInUse()){
				then = cached.getLastUsed();
				if(verbose) {
					log.info("La connexion est inactive depuis " +
						(now-then) + " millis");
				}
				if ((now-then) >= MAX_IDLE) {
					if(verbose) {
						log.info("La connexion " + Integer.toString(i) + 
							" a depasse le temps d'inactivite : elle va etre supprimee");
					}
					conn = cached.getConn();
					try {
						conn.close();
					} catch (Exception e) {
						if (verbose) {
							log.info("CachedConnectionManager - Problme lors de la fermeture de la connexion :");
							log.info(e);
						}
					}
					cachedConnections.remove(i);
					numberConnections--;
				}
			}
		}
	}
	
	/**
	 * Ferme les connexions et dort pendant 5 minutes
	 */
	private static void runMonitor() {
		checkUse();
		if(numberConnections > 0) {
			if(verbose) {
				log.info("Le monitor se met en mode sleep");
			}
			try {
				monitor.sleep(FREQ);
			} catch (Exception e) {
				if(verbose) {
					log.info("Le mode sleep du monitor a t interrompu");
				}
			}
		}
	}
	
	/* 
	 * Ferme toutes les connexions  la fermeture du conteneur de servlets
	 */
	public void finalize() throws Throwable {
		CachedConnection cached = null;
		for (int i=0;i<numberConnections;i++){
			cached = (CachedConnection) cachedConnections.get(i);
			if (cached.getConn() != null) {
				if (verbose) {
					log.info("Fermeture de la connexion " + Integer.toString(i));
				}
				try {
					cached.getConn().close();
				} catch (Exception e) {
					if (verbose) {
						log.info("CachedConnectionManager - Problme lors de la fermeture de la connexion "+i+":");
						log.info(e);
					}
				}
			}
		}
		numberConnections = 0;
	}
	
	/**
	 * @param b
	 */
	public static void setVerbose(boolean b) {
		verbose = b;
	}

}