/*
 * HomedirCreation.java,v 1.0 29 juin 2004
 * Copyright (c) 2004 Esup Portail (www.esup-portail.org)
 * Classes: HomedirCreation
 * Original Author: Thomas BELLEMBOIS
 */
/**
 * @author : Thomas BELLEMBOIS
 * date : 29 juin 2004
 */

package org.esupportail.portal.utils.injac.homeDirCreation;

import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpURL;
import org.apache.commons.httpclient.URIException;
import org.esupportail.portal.utils.injac.homeDirCreation.config.Config;
import org.esupportail.portal.utils.injac.homeDirCreation.exception.ConfigException;
import org.esupportail.portal.utils.injac.homeDirCreation.exception.WrongPathException;
import org.esupportail.portal.utils.injac.tools.ACLManager;

import com.novell.ldap.*;
import org.apache.log4j.*;
import org.apache.webdav.lib.WebdavResource;
import org.apache.webdav.lib.methods.DepthSupport;
import org.apache.webdav.lib.methods.PropFindMethod;

import java.io.IOException;
import java.util.*;

/**
 * THE FILE config.xml configures this application, read it with attention !<br/>
 * 
 * Creates homedirs for users of the WebDAV server<br/>
 * HomedirCreation -help<br/>
 * HomedirCreation -user [username]<br/>Create homedir of the user [username]
 * HomedirCreation -all<br/>Create a series of homedir for users registered in the LDAP directory specified in the config.xml file
 * 
 */
public class HomedirCreation {
	
	private HttpURL httpUrl;
	private WebdavResource webDavResource;
	
	private String ldapUrl;
	private String ldapPort;
	private String ldapBaseDn;
	private String ldapRequestFilter;
	private String ldapUserUidAttribut;
	private String ldapMaxResults;
	
	private String webdavServerUrl;
	private String webdavServerLogin;
	private String webdavServerPassword;
	private String webdavServerUsersRole;
	
	private String userDirectoryRootDirectory;
	private String roleDirectoryRootDirectory;
	
	private ACLManager aclManager;
	
	private String directoriesStructureRootDirectory;
	private String directoriesStructureHashType;
	
	private Logger logger = Logger.getLogger("HomedirCreation");
	
	/**
	 * Get a ressource property
	 * @param webdavResource
	 * @param path
	 * @param properties
	 * @return Property Enumeration
	 * @throws URIException
	 * @throws IOException
	 */
	private Enumeration getProperty(WebdavResource webdavResource, String path, Vector properties) throws URIException, IOException {
		PropFindMethod method = new PropFindMethod(path, DepthSupport.DEPTH_INFINITY, properties.elements());
		webdavResource.retrieveSessionInstance().executeMethod(method);
		return method.getResponseProperties(path);
	}// getProperty
	
	/**
	 * @return String Vector of registered user in the LDAP directory specified in the config.xml file
	 */
	private Vector getLdapRegisteredUsers(){
		
		Vector ldapRegisteredUsers = new Vector();
		////////////////////////////////////////////////////////
		//
		// Variables used to send requests to the LDAP directory
		//
		int msLimit=0; 		 								// The maximum time in milliseconds to wait for results
		int serverTimeLimit=0; 								// The maximum time in seconds that the server should spend returning search results
		int dereference=LDAPSearchConstraints.DEREF_NEVER; 	// Specifies when aliases should be dereferenced
		int maxResults=Integer.parseInt(ldapMaxResults);	// The maximum number of search results to return for a search request
		boolean doReferrals=false; 				// Determines whether to automatically follow referrals or not
		int batchSize=0; 	 	 				// The number of results to return in a batch
		LDAPReferralHandler handler=null; 		// The custom authentication handler called when LDAPConnection needs to authenticate, typically on following a referral
		int hop_limit=0; 				 		// The maximum number of referrals to follow in a sequence during automatic referral following
		int scope = LDAPConnection.SCOPE_SUB;   // The scope of the entries to search
		String[] attrs = new String[1]; 		// The names of attributes to retrieve
		attrs[0]=ldapUserUidAttribut;
		boolean typesOnly = false;				// If true, returns the names but not the values of the attributes found, if false, returns the names and values for attributes found
		
		try{
			// Connecting to the LDAP directory
			LDAPConnection ldapConnection = new LDAPConnection();
			ldapConnection.connect(ldapUrl, Integer.parseInt(ldapPort));
			if(ldapConnection.isConnected()){logger.info("Connected to "+ldapUrl);}
			
			// Getting all the entries in the LDAP directory
			LDAPSearchConstraints LdapSearchConstraints = new LDAPSearchConstraints(msLimit, serverTimeLimit, dereference, maxResults, doReferrals, batchSize, handler, hop_limit); 	           
			LDAPSearchResults ldapSearchResults = ldapConnection.search(ldapBaseDn, scope, ldapRequestFilter, attrs, typesOnly, LdapSearchConstraints); 
			
			while (ldapSearchResults.hasMore())
			{
				LDAPEntry ldapEntry=ldapSearchResults.next();
				LDAPAttributeSet ldapAttributeSet = ldapEntry.getAttributeSet();
				String uid = ldapAttributeSet.getAttribute(ldapUserUidAttribut).getStringValue();
				
				ldapRegisteredUsers.add(uid);
				if(logger.isDebugEnabled()){logger.debug("User "+uid+" retrieved");}
			}// while
			
			// Disconnecting from the LDAP directory
			ldapConnection.disconnect();
			if(!ldapConnection.isConnected()){logger.info("Disconnected from "+ldapUrl);}
			
		}catch(LDAPException ldapException){logger.info("HomedirCreation::getLdapRegisteredUsers::"+ldapException);}
		return ldapRegisteredUsers;
		
	}// getLdapRegisteredUsers
	
	/**
	 * Initialize all the variables and connect to the WebDAV server
	 */
	private void init(){
		try{
			// Getting all the entries of the configuration file
			ldapUrl = Config.getInstance().getLdapUrl();     					 // LDAP directory name
			ldapPort = Config.getInstance().getLdapPort();                       // LDAP server port
			ldapBaseDn = Config.getInstance().getLdapBaseDn();                   // LDAP server suffix
			ldapRequestFilter = Config.getInstance().getLdapRequestFilter();     // LDAP request filter
			ldapUserUidAttribut = Config.getInstance().getLdapUserUidAttribut(); // The attribut of users UIDs in the LDAP directory
			ldapMaxResults = Config.getInstance().getLdapMaxResults();			 // Max number of results returned by the LDAP directory
			
			webdavServerUrl = Config.getInstance().getWebdavServerUrl();		    // WebDav server URL
			webdavServerLogin = Config.getInstance().getWebdavServerLogin();	    // WebDav server login
			webdavServerPassword = Config.getInstance().getWebdavServerPassword();  // WebDav server password
			webdavServerUsersRole = Config.getInstance().getWebdavServerUsersRole();// Role assigned to users
			
			userDirectoryRootDirectory = Config.getInstance().getUserDirectoryRootDirectory(); // Root directory to create homedirs on the WebDav server
			roleDirectoryRootDirectory = Config.getInstance().getRoleDirectoryRootDirectory(); // Root directory to create roles on the WebDav server
						
			aclManager = new ACLManager(webdavServerUrl+"/", webdavServerLogin, webdavServerPassword);
			
			// Connecting to the Webdav server
			httpUrl = new HttpURL(webdavServerUrl+"/");
			httpUrl.setUserinfo(webdavServerLogin, webdavServerPassword);
			String context = httpUrl.getCurrentHierPath();
			webDavResource = new WebdavResource(httpUrl);
			logger.info("Connected to "+webDavResource.getPath());
			
			// TODO explain why we add "context" to directoriesStructureRootDirectory
			// context == /slide for example
			directoriesStructureRootDirectory = context+"/"+Config.getInstance().getDirectoriesStructureRootDirectory(); // Root directory to create the directories on the WebDav server
			directoriesStructureHashType = Config.getInstance().getDirectoriesStructureHashType(); // Hash type to create the directories on the WebDav server	
		}
		catch(ConfigException configException){logger.info("HomedirCreation::init::"+configException);}
		catch (HttpException httpException){logger.info("HomedirCreation::init::"+httpException);}
		catch (IOException ioException){logger.info("HomedirCreation::init::"+ioException);}
	}// init
	
	/**
	 * Verify if the WebDAV ressource exits
	 * @param webDavResource
	 * @param path syntax: /{context}/path - exemple: /slide/files/myRessource
	 * @return true if the path exists, else false
	 */
	private boolean pathExist(WebdavResource webDavResource, String path){
		try {
			webDavResource.setPath(path);
		} catch (HttpException httpException) {
			return false;
		} catch (IOException ioException) {
			logger.info("HomedirCreation::pathExist::"+ioException);
		}
		return true;
	}// isExist
	
	/**
	 * Create and protect a user homedir
	 * @param webDavResource
	 * @param user
	 * @throws WrongPathException
	 */
	private void createHomedir(WebdavResource webDavResource, String user) throws WrongPathException{
		try{
			String userFirstSubdirectory = new String();
			String userSecondSubdirectory = new String();
			int userLength = user.length();
			
			// Creating the homedir path depending on the hash method
			if(directoriesStructureHashType.equals("with_hash")){
				if(logger.isDebugEnabled()){logger.debug("Hash method : with_hash");}
				userFirstSubdirectory = user.charAt(0)+"";
				userSecondSubdirectory = user.charAt(0)+""+user.charAt(1)+"";
			}// if
			else if(directoriesStructureHashType.equals("with_hash_reverse")){
				if(logger.isDebugEnabled()){logger.debug("Hash method : with_hash_reverse");}
				userFirstSubdirectory = user.charAt(userLength-1)+"";
				userSecondSubdirectory = user.charAt(userLength-1)+""+user.charAt(userLength-2)+"";
			}//else
			
			// Verifying if the directoriesStructureRootDirectory exits - must be created by the administrator
			if (!pathExist(webDavResource,directoriesStructureRootDirectory)){
				throw new WrongPathException("HomedirCreation", "createHomedir", directoriesStructureRootDirectory);
			}// if
			
			// Creating the homedir (and the full path if not created)
			if (pathExist(webDavResource,directoriesStructureRootDirectory+"/"+userFirstSubdirectory)){
				if(logger.isDebugEnabled()){logger.debug(directoriesStructureRootDirectory+"/"+userFirstSubdirectory+"/"+" exists");}
				if (pathExist(webDavResource,directoriesStructureRootDirectory+"/"+userFirstSubdirectory+"/"+userSecondSubdirectory)){
					if(logger.isDebugEnabled()){logger.debug(directoriesStructureRootDirectory+"/"+userFirstSubdirectory+"/"+userSecondSubdirectory+"/"+" exists");}
					if(pathExist(webDavResource,directoriesStructureRootDirectory+"/"+userFirstSubdirectory+"/"+userSecondSubdirectory+"/"+user)){
						logger.info("The homedir of "+user+" already exists !!!");
					}// if
					else{
						if(logger.isDebugEnabled()){logger.debug("Creation of "+directoriesStructureRootDirectory+"/"+userFirstSubdirectory+"/"+userSecondSubdirectory+"/"+user);}
						webDavResource.mkcolMethod(directoriesStructureRootDirectory+"/"+userFirstSubdirectory+"/"+userSecondSubdirectory+"/"+user);
					}// else
				}// if
				else{
					if(logger.isDebugEnabled()){logger.debug("Creation of "+directoriesStructureRootDirectory+"/"+userFirstSubdirectory+"/"+userSecondSubdirectory);}	
					webDavResource.mkcolMethod(directoriesStructureRootDirectory+"/"+userFirstSubdirectory+"/"+userSecondSubdirectory);
					if(logger.isDebugEnabled()){logger.debug("Creation of "+directoriesStructureRootDirectory+"/"+userFirstSubdirectory+"/"+userSecondSubdirectory+"/"+user);}	
					webDavResource.mkcolMethod(directoriesStructureRootDirectory+"/"+userFirstSubdirectory+"/"+userSecondSubdirectory+"/"+user);
				}// else
			}// if
			else{
				if(logger.isDebugEnabled()){logger.debug("Creation of "+directoriesStructureRootDirectory+"/"+userFirstSubdirectory);}	
				webDavResource.mkcolMethod(directoriesStructureRootDirectory+"/"+userFirstSubdirectory);
				if(logger.isDebugEnabled()){logger.debug("Creation of "+directoriesStructureRootDirectory+"/"+userFirstSubdirectory+"/"+userSecondSubdirectory);}
				webDavResource.mkcolMethod(directoriesStructureRootDirectory+"/"+userFirstSubdirectory+"/"+userSecondSubdirectory);
				if(logger.isDebugEnabled()){logger.debug("Creation of "+directoriesStructureRootDirectory+"/"+userFirstSubdirectory+"/"+userSecondSubdirectory+"/"+user);}
				webDavResource.mkcolMethod(directoriesStructureRootDirectory+"/"+userFirstSubdirectory+"/"+userSecondSubdirectory+"/"+user);
			}// else
			
			// Protecting the homedir with an ACL
			// deny on /u /u/us /u/us/user for all
			// grant on /u/us/user for user
			// TODO
			//aclManager.deny(directoriesStructureRootDirectory+"/"+userFirstSubdirectory, "/"+roleDirectoryRootDirectory+"/"+webdavServerUsersRole, "read");
			//aclManager.deny(directoriesStructureRootDirectory+"/"+userFirstSubdirectory, "/"+roleDirectoryRootDirectory+"/"+webdavServerUsersRole, "write");
			//aclManager.deny(directoriesStructureRootDirectory+"/"+userFirstSubdirectory+"/"+userSecondSubdirectory, "/"+roleDirectoryRootDirectory+"/"+webdavServerUsersRole, "read");
			//aclManager.deny(directoriesStructureRootDirectory+"/"+userFirstSubdirectory+"/"+userSecondSubdirectory, "/"+roleDirectoryRootDirectory+"/"+webdavServerUsersRole, "write");
			
			aclManager.grant(directoriesStructureRootDirectory+"/"+userFirstSubdirectory+"/"+userSecondSubdirectory+"/"+user, "/"+userDirectoryRootDirectory+"/"+user, "read");
			aclManager.grant(directoriesStructureRootDirectory+"/"+userFirstSubdirectory+"/"+userSecondSubdirectory+"/"+user, "/"+userDirectoryRootDirectory+"/"+user, "write");
			aclManager.deny(directoriesStructureRootDirectory+"/"+userFirstSubdirectory+"/"+userSecondSubdirectory+"/"+user, "/"+roleDirectoryRootDirectory+"/"+webdavServerUsersRole, "read");
			aclManager.deny(directoriesStructureRootDirectory+"/"+userFirstSubdirectory+"/"+userSecondSubdirectory+"/"+user, "/"+roleDirectoryRootDirectory+"/"+webdavServerUsersRole, "write");
		}
		catch(HttpException httpException){logger.info("HomedirCreation::createHomedir::"+httpException);}
		catch(IOException ioException){logger.info("HomedirCreation::createHomedir::"+ioException);}
				
	}// createHomedir
	
	public static void main(String[] args) {
		
		HomedirCreation homedirCreation = new HomedirCreation();
		
		///////////////////////////
		//
		// Verifying the parameters
		//
		if(args.length==0){
			homedirCreation.logger.error("Parameter needed");
			System.exit(0);
		}
		if(args.length>2){
			homedirCreation.logger.error("Too many arguments");
			System.exit(0);
		}
		if(args[0].equals("-help")){
			homedirCreation.logger.info("HomedirCreation -all|-user <username>");
			System.exit(0);
		}
		if(!args[0].equals("-all") && !args[0].equals("-user")){
			homedirCreation.logger.error("Wrong first parameter");
			System.exit(0);
		}
		if(args[0].equals("-user") && !(args.length==2)){
			homedirCreation.logger.error("Wrong username");
			System.exit(0);
		}	
		
		// Initializing variables - connecting to the WebDAV server
		homedirCreation.init();
		
		homedirCreation.logger.info("Starting process...");
		
		try{			
			if(args[0].equals("-all")){
			// Creating homedir of every registered user (in the LDAP directory)
				Vector registeredUsers = homedirCreation.getLdapRegisteredUsers();
				String currentUser;
				
				for (Enumeration enum = registeredUsers.elements(); enum.hasMoreElements();){
					currentUser = (String)enum.nextElement();
					homedirCreation.createHomedir(homedirCreation.webDavResource, currentUser);
				}// for
				
			}// -all
			else{
			// Creating homedir of one user
				homedirCreation.createHomedir(homedirCreation.webDavResource, args[1]);	
			}// -user
			
			// Disconnecting from the Webdav server
			homedirCreation.webDavResource.close();
			homedirCreation.logger.info("Disconnected from "+homedirCreation.webDavResource.getPath());
		}
		catch (URIException e) {homedirCreation.logger.info("HomedirCreation::main::"+e);}
		catch (NullPointerException e1) {homedirCreation.logger.info("HomedirCreation::main::"+e1);}
		catch (IOException e2){homedirCreation.logger.info("HomedirCreation::main::"+e2);}
		catch (WrongPathException e3){homedirCreation.logger.info("HomedirCreation::main::"+e3);}
		homedirCreation.logger.info("Process finished...");
		
	}// main
	
}// HomedirCreation
