/* ====================================================================
 *
 * mod_auth_esup.c
 * ce module derive du module mod_auth_anon livre avec la distribution
 * apache
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    nor may "Apache" appear in their name, without prior written
 *    permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * Portions of this software are based upon public domain software
 * originally written at the National Center for Supercomputing Applications,
 * University of Illinois, Urbana-Champaign.
 */

/*
 * http_auth: authentication
 * 
 * Rob McCool & Brian Behlendorf.
 * 
 * Adapted to Apache by rst.
 *
 * Version 0.5 May 1996
 *
 * Modified by Dirk.vanGulik@jrc.it to
 * 
 * Modified bye Vincent.Mathieu@univ-nancy2.fr (2003-12-11) to Esup logins
 *
 * Adapted to allow anonymous logins, just like with Anon-FTP, when
 * one gives the magic user name 'anonymous' and ones email address
 * as the password.
 *
 * Just add the following tokes to your <directory> setup:
 * 
 * Esup                    magic-userid [magic-userid]...
 *
 * Esup_MustGivePassword        [ on | off ] default = on
 * Esup_Password                password
 * Esup_Authoritative           [ on | off ] default = off
 *
 * The magic user id is something like 'root', it is NOT case sensitive. 
 *  They a le unhautorized users
 * 
 * The MustGivePassword flag can be used to force users to enter something
 * in the password field. Default is on.
 *
 * 
 * The Esup_Password is a password for all users
 */

#include "apr_strings.h"

#define APR_WANT_STRFUNC
#include "apr_want.h"

#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "http_request.h"
#include "http_protocol.h"

typedef struct esup_auth {
  char *userid;   // contient un des comptes a ne pas tester (Esup toto)
    struct esup_auth *next;
} esup_auth;

typedef struct {
    esup_auth *esup_auth_userids;
    char * esup_auth_password;
    int esup_auth_mustpassword;
    int esup_auth_authoritative;
} esup_auth_config_rec;

static void *create_esup_auth_dir_config(apr_pool_t *p, char *d)
{
    esup_auth_config_rec *conf = apr_palloc(p, sizeof(*conf));

    /* just to illustrate the defaults really. */
    conf->esup_auth_userids = NULL;

    conf->esup_auth_password = NULL;
    conf->esup_auth_mustpassword = 1;
    conf->esup_auth_authoritative = 0;
    return conf;
}

static const char *esup_set_string_slots(cmd_parms *cmd,
                                         void *my_config, const char *arg)
{
    esup_auth_config_rec *conf = my_config;
    esup_auth *first;

    if (!(*arg))
        return "user string cannot be empty";

    /* squeeze in a record */
    first = conf->esup_auth_userids;

    if (!(conf->esup_auth_userids = apr_palloc(cmd->pool, sizeof(esup_auth))) ||
       !(conf->esup_auth_userids->userid = apr_pstrdup(cmd->pool, arg)))
             return "Failed to claim memory for a userid...";

    /* and repair the next */
    conf->esup_auth_userids->next = first;

    return NULL;
}

static const command_rec esup_auth_cmds[] =
{
    AP_INIT_ITERATE("Esup", esup_set_string_slots, NULL, OR_AUTHCFG, 
     "a space-separated list of unautorized user IDs"),
    AP_INIT_TAKE1("Esup_Password", ap_set_string_slot,
     (void *)APR_OFFSETOF(esup_auth_config_rec, esup_auth_password), OR_AUTHCFG,
		  "Password to use to validate the users"),
    AP_INIT_FLAG("Esup_MustGivePassword", ap_set_flag_slot,
     (void *)APR_OFFSETOF(esup_auth_config_rec, esup_auth_mustpassword),
     OR_AUTHCFG, "Limited to 'on' or 'off'"),
    AP_INIT_FLAG("Esup_Authoritative", ap_set_flag_slot,
     (void *)APR_OFFSETOF(esup_auth_config_rec, esup_auth_authoritative),
     OR_AUTHCFG, "Limited to 'on' or 'off'"),
    {NULL}
};

module AP_MODULE_DECLARE_DATA auth_esup_module;

static int esup_authenticate_basic_user(request_rec *r)
{
    esup_auth_config_rec *conf = ap_get_module_config(r->per_dir_config,
                                                      &auth_esup_module);
    const char *sent_pw;
    int res;

    if ((res = ap_get_basic_auth_pw(r, &sent_pw))) {
        return res;
    }

    if (!r->user[0]) {
       ap_log_rerror(APLOG_MARK, APLOG_ERR, APR_SUCCESS, r,
                        "Esup: empty user not allowed");
            ap_note_basic_auth_failure(r);
            return HTTP_UNAUTHORIZED;
    }

    /* is it the magic one
     */

    esup_auth *p = conf->esup_auth_userids;

    while (p != NULL) {
      if (strcasecmp(r->user, p->userid) == 0) {
        if (conf->esup_auth_authoritative) {
            ap_log_rerror(APLOG_MARK, APLOG_ERR, APR_SUCCESS, r,
                        "Esup: Authoritative, user <%s> not accepted", r->user);
            ap_note_basic_auth_failure(r);
            return HTTP_UNAUTHORIZED;
        }
        return DECLINED;
      }
      p = p->next;
    }
    if (!conf->esup_auth_mustpassword)
         return OK;
    if  ((strlen(sent_pw) && conf->esup_auth_password == NULL))
	 return OK;
    if  (strcmp(sent_pw, conf->esup_auth_password) == 0) {
        return OK;
    }
    if (conf->esup_auth_authoritative) {
       ap_log_rerror(APLOG_MARK, APLOG_ERR, APR_SUCCESS, r,
                        "Esup: Authoritative, Passwd <%s> not accepted",
                        (sent_pw && *sent_pw)  ? "****" : "\'none\'");
       ap_note_basic_auth_failure(r);
       return HTTP_UNAUTHORIZED;
    }
    return DECLINED;
}

static int check_esup_access(request_rec *r)
{
    esup_auth_config_rec *conf = ap_get_module_config(r->per_dir_config,
                                                 &auth_esup_module);
    char *user = r->user;
    int m = r->method_number;
    int method_restricted = 0;
    register int x;
    const char *t, *w;
    const apr_array_header_t *reqs_arr = ap_requires(r);
    require_line *reqs;

    /* BUG FIX: tadc, 11-Nov-1995.  If there is no "requires" directive, 
     * then any user will do.
     */
    if (!reqs_arr) {
        return OK;
    }
    reqs = (require_line *)reqs_arr->elts;

    for (x = 0; x < reqs_arr->nelts; x++) {

        if (!(reqs[x].method_mask & (AP_METHOD_BIT << m))) {
            continue;
        }

        method_restricted = 1;

        t = reqs[x].requirement;
        w = ap_getword_white(r->pool, &t);
        if (!strcmp(w, "valid-user")) {
            return OK;
        }
        if (!strcmp(w, "user")) {
            while (t[0]) {
                w = ap_getword_conf(r->pool, &t);
                if (!strcmp(user, w)) {
                    return OK;
                }
            }
        }
        else if (conf->esup_auth_authoritative) {
            /* if we aren't authoritative, any require directive could be
             * valid even if we don't grok it.  However, if we are 
             * authoritative, we can warn the user they did something wrong.
             * That something could be a missing "AuthAuthoritative off", but
             * more likely is a typo in the require directive.
             */
            ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r,
                          "access to %s failed, reason: unknown require "
                          "directive:\"%s\"", r->uri, reqs[x].requirement);
        }
    }

    if (!method_restricted) {
        return OK;
    }

    if (!(conf->esup_auth_authoritative)) {
        return DECLINED;
    }

    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r,
                  "access to %s failed, reason: user %s not allowed access",
                  r->uri, user);
        
    ap_note_basic_auth_failure(r);
    return HTTP_UNAUTHORIZED;
}

static void register_hooks(apr_pool_t *p)
{
    ap_hook_check_user_id(esup_authenticate_basic_user,NULL,NULL,APR_HOOK_MIDDLE);
    ap_hook_auth_checker(check_esup_access,NULL,NULL,APR_HOOK_MIDDLE);
}

module AP_MODULE_DECLARE_DATA auth_esup_module =
{
    STANDARD20_MODULE_STUFF,
    create_esup_auth_dir_config,    /* dir config creater */
    NULL,                           /* dir merger ensure strictness */
    NULL,                           /* server config */
    NULL,                           /* merge server config */
    esup_auth_cmds,                 /* command apr_table_t */
    register_hooks                  /* register hooks */
};
