#!/usr/bin/perl
# syncDavLdap.pl
#
# Version 1.0 du 08/04/2004
#
# cet utilitaire synchonise l'espace du serveur webDav V1 d'esup-portail
# avec les entrees LDAP de l'etablissement
#
# Le module perl Net::LDAP doit etre installe
#
# pour chaque compte valide LDAP (ex : login 'etudiant') :
#   . S'assure que le repertoire e/et/etudiant existe, et contient un fichier
#     .htaccess
#   . sinon, creation du repertoire et du fichier, qui va contenir
#            "require user etudiant"
#
# Eventuellement, supprime les comptes webDav (en fait, les repertoires associes)
# qui ne correspondent plus a des comptes LDAP valides
#
# ATTENTION : ce script doit etre execute a partir du compte unix sous lequel
#             fonctionne le serveur WebDav
#
# il recoit les parametres suivants :

# -path : obligatoire. Le repertoire a partir duquel les comptes webDav sont crees
# -ldap.serv : obligatoire. le serveur ldap (ex : ldap.univ.fr, ou ldap.univ.fr:392 si autre port TCP)
# -ldap.basedn : obligatoire. le base DN de bind LDAP
# -ldap.binddn : optionnel. utilise en cas de bind non anonyme
# - ldap.bindpasswd : optionnel. utilise en cas de bind non anonyme
# -ldap.filtre : optionnel. Contient un filtre a appliquer pour determiner les entrees LDAP concernee
#           par defaut : (objectclass=supannPerson)
# -ldap.attrib : optionnel. l'attribut correspondant au compte webDav. Par defaut : uid
# - hash : optionnel. 2 valeurs possibles : normal ou reverse. 
#                     Ces valeurs correspondent au 'hashage' du mod_userdir-esup
#                     par défaut : normal
# -mode :   optionnel. 4 valeurs possibles :
#                    . test : se contente de lister les divergences
#                    . create : cree les comptes webdav qui manqueraient
#                    . delete : supprime les comptes webdav qui ne sont plus dans LDAP
#                    . sync : synchronise les comptes LDAP et webDav . En fait, c'est create + delete
#           par defaut : test
#
# Un fichier de compte-rendu est genere : syncDavLdap.txt
#
# un code erreur different de zero est retourne en cas d'erreurs ou d'anomalies
#

use strict;

use Net::LDAP;
use File::Find;

use vars qw($timeout $nbTraite %loginLDAP %comptesWebdav $codeRetour);
use vars qw($hash $path); 

$SIG{'ALRM'}  = 'ArretTimeout';

{
  my ($mesg, $connLDAP, $servLDAP, $portLDAP, $bindDN, $bindPasswd, $basednLDAP);
  my ($filtreLDAP, $attribLDAP, $mode, $laDate);
  $timeout = 120;      # delai d'execution en secondes, avant erreur, pour chaque entree
  my $nomFicRes = "syncDavLdap.txt";
  $filtreLDAP = "(objectclass=supannPerson)";
  $attribLDAP = "uid";
  $mode = "test";
  $hash = "normal";
  alarm($timeout);
#  ------ recuperation des parametres --------
  for (my $ind = 0; $ind < scalar @ARGV; $ind++) 
  {
    if ($ARGV[$ind] eq "-path")
    {
      $ind++;
      $path = $ARGV[$ind];
    }
    elsif ($ARGV[$ind] eq "-ldap.serv")
    {
      $ind++;
      ($servLDAP, $portLDAP) = split(":", $ARGV[$ind]);
      $portLDAP = 389 if ($portLDAP eq "");
    }
    elsif ($ARGV[$ind] eq "-ldap.basedn")
    {
      $ind++;
      $basednLDAP = $ARGV[$ind];
    }
    elsif ($ARGV[$ind] eq "-ldap.binddn")
    {
      $ind++;
      $bindDN = $ARGV[$ind];
    }
    elsif ($ARGV[$ind] eq "-ldap.bindpasswd")
    {
      $ind++;
      $bindPasswd = $ARGV[$ind];
    }
    elsif ($ARGV[$ind] eq "-ldap.filtre")
    {
      $ind++;
      $filtreLDAP = $ARGV[$ind];
    }
    elsif ($ARGV[$ind] eq "-ldap.attrib")
    {
      $ind++;
      $attribLDAP = $ARGV[$ind];
    }
    elsif ($ARGV[$ind] eq "-hash")
    {
      $ind++;
      $hash = $ARGV[$ind];
    }
    elsif ($ARGV[$ind] eq "-mode")
    {
      $ind++;
      $mode = $ARGV[$ind];
    }
  }
  &ErrSyntaxe() if (($path eq "") || ($servLDAP eq "") || ($basednLDAP eq ""));
  &ErrSyntaxe() if (($mode ne "test") && ($mode ne "create") && ($mode ne "delete") && ($mode ne "sync"));
  &ErrSyntaxe() if (($hash ne "normal") && ($hash ne "reverse"));
  if (! open(FICRES, ">$nomFicRes"))
  {
    print FICRES "impossible de creer le fichier $nomFicRes";
    exit 1;
  }
  my $laDate = &GetDate();
  print FICRES "************************************************************************\n";
  print FICRES "\t*** controle synchro LDAP <-> webDav, le $laDate ***\n\n";
  print FICRES " \t\t\tmode $mode\n";
  if (! ($connLDAP = new Net::LDAP($servLDAP, port => $portLDAP )))
  {
    print FICRES "ERREUR. connexion LDAP refusee sur $servLDAP:$portLDAP\n" ;
    exit 1;
  }
  if ($bindDN eq "")
  {
    $mesg = $connLDAP->bind (version => 3 );
    if ($mesg->is_error)
    {
      print FICRES "ERREUR. bind anonyme LDAP refuse sur $servLDAP:$portLDAP\n" ;
      exit 1;
    }
  }
  else
  {    
    $mesg = $connLDAP->bind ( dn => $bindDN, password  => $bindPasswd, version => 3 );
    if ($mesg->is_error)
    {
      print FICRES "ERREUR. bind LDAP refuse sur $servLDAP:$portLDAP pour binddnLDAP\n" ;
      exit 1;
    }
  }
  print FICRES "\nLECTURE DES COMPTES LDAP ...\n";
  my $nbLDAP = &recupLoginLDAP(\$connLDAP, $basednLDAP, $filtreLDAP, $attribLDAP);
  print FICRES "   nombre de comptes LDAP lus : $nbLDAP\n";
  $connLDAP->unbind;
  alarm(0);
  print FICRES "\nLECTURE DES COMPTES WEBDAV depuis $path ...\n";
  my $nbWebdav = &recupComptesWebdav($path);
  print FICRES "   nombre de comptes webDav lus : $nbWebdav\n";

  print FICRES "\n\nRECHERCHE DES COMPTES WEBDAV QUI NE SONT PAS DANS LDAP\n";
  my $nbDavEnTrop = 0;
  foreach my $userDav (sort keys %comptesWebdav)
  {
     if (! (defined $loginLDAP{$userDav}))
     {
	$nbDavEnTrop++;
        if (($mode eq "delete") || ($mode eq "sync"))
        {
            my $repUser = &calculeRepUser($userDav);
	    &deleteDav($repUser);
        }
        else
        {
	  print FICRES "   $userDav\n";
        } 
     }
  }
  print FICRES "  nombre de comptes webDav en trop : $nbDavEnTrop\n";
  print FICRES "\n\nRECHERCHE DES COMPTES WEBDAV MANQUANT\n";
  my $nbDavEnMoins = 0;
  foreach my $userLDAP (sort keys %loginLDAP)
  {
     if (! (defined $comptesWebdav{$userLDAP}))
     {
	$nbDavEnMoins++;
        if (($mode eq "create") || ($mode eq "sync"))
        {
            my $repUser = &calculeRepUser($userLDAP);
	    &createDav($repUser, $userLDAP);
        }
        else
        {
	  print FICRES "   $userLDAP\n";
        } 
     }
  }


  print FICRES "  nombre de comptes webDav manquant : $nbDavEnMoins\n";

  print FICRES "\n\n\t*** Fin de traitement de $servLDAP:$portLDAP, le $laDate ***\n\n";
  print FICRES "************************************************************************\n";
  exit $codeRetour;
}

# ------------------------------------------------------------------
# calcul du nom de repertoire d'un user
# param 0 : le compte concerne
# ------------------------------------------------------------------
sub calculeRepUser
{
    my ($first, $second);
    my $user = shift;
    if ($ hash eq "normal")
    {
	$first = substr($user, 0, 1);
        $second = substr($user, 0, 2);
    }
    else
    {
	my $len = length($user);
        $first = substr($user, $len - 1, 1);
        $second = $first . substr($user, $len - 2, 1);
    }
    return "$path/$first/$second/$user";
}

# ------------------------------------------------------------------
# creation d'un compte webDav
# param 0 : la directory de l'utilisateur a creer
# param 1 : le login du compte
# ------------------------------------------------------------------
sub createDav
{
    my $repUser = shift;
    my $user = shift;
    if (! mkdir($repUser))
    {
	print FICRES "  ERREUR : $repUser : creation impossible\n";
        $codeRetour = 31;
        return 0;
    }
    my $htaccess = "$repUser/.htaccess";
    if (! (open(FICACCESS, ">$htaccess")))
    {
	print FICRES "  ERREUR : $htaccess : creation impossible\n";
        $codeRetour = 32;
        return 0;
    }
    print FICACCESS "require user $user\n";
    close FICACCESS;
    print FICRES "  $repUser : creation du repertoire et du .htaccess OK\n";
}

# ------------------------------------------------------------------
# suppression d'un compte webDav
# param 0 : la directory de l'utilisateur a supprimer
# ------------------------------------------------------------------
sub deleteDav
{
    my $repUser = shift;
    if (system("/bin/rm -r $repUser 2>/dev/null") != 0)
    {
	print FICRES "  ERREUR : $repUser : suppression impossible\n";
        $codeRetour = 21;
    }
    else
    {
	print FICRES "  $repUser : suppression OK\n";
    }
}

# ------------------------------------------------------------------
# recuperation des comptes webDav
# param 0 : la directory de base dees comptes webDav
# ------------------------------------------------------------------
sub recupComptesWebdav
{
   my ($rep1, $rep2, $user);
   my $path = shift;
   my $nbre = 0;
   if (! opendir(REP, $path))
   {
      print FICRES "probleme ouverture du repertoire $path\n";
      exit 5;
   }      
   while (defined ($rep1 = readdir REP))
   {
      next if $rep1 =~ /^\./;
      if (! opendir(REP1, "$path/$rep1"))
      {
        print FICRES "probleme ouverture du repertoire $path/$rep1\n";
        exit 5;
      }
      while (defined ($rep2 = readdir REP1))
      {
         next if $rep2 =~ /^\./;
         if (! opendir(REPUSER, "$path/$rep1/$rep2"))
         {
           print FICRES "probleme ouverture du repertoire $path/$rep1/$rep2\n";
           exit 5;
         } 
#         print "$path/$rep1/$rep2\n";
         while (defined ($user = readdir REPUSER))
         {
            next if $user =~ /^\./;
            $comptesWebdav{$user} = 1;
            $nbre++;
            my $repUser = calculeRepUser($user);
            if ($repUser ne "$path/$rep1/$rep2/$user")
            {
              print FICRES "  BIZARRE : le chemin $path/$rep1/$rep2/$user ne correspond pas au hashage choisi\n";
              $codeRetour = 11;
	    }
            if (! (-e "$path/$rep1/$rep2/$user/.htaccess"))
            {
              print FICRES "  ERREUR : pas de fichier .htaccess dans $path/$rep1/$rep2/$user\n";
              $codeRetour = 10;
	    }
	 }
         closedir(REPUSER);
      }
      closedir(REP2);
   }
   closedir(REP);
   return $nbre;
}

# ------------------------------------------------------------------
# recuperation des logins LDAP qui doivent avoir des comptes webDav
# param 0 : pointeur sur handle de connexion LDAP
# param 1 : le base DN
# param 3 : le filtre LDAP de recherche des entrees concernees
# param 4 : l'attribut LDAP qui sert de reference au compte webDav
# resultat dans un tableau associatif : %loginLDAP
# ------------------------------------------------------------------
sub recupLoginLDAP
{
    my ($Entry);
  my $connLDAP = shift;
  my $basednLDAP = shift;
  my $filtreLDAP = shift;
  my $attribLDAP = shift;
  my $mesg = $$connLDAP->search(base => $basednLDAP, scope => "sub", filter => $filtreLDAP,
                             attrs => [ "$attribLDAP"] );
  if ($mesg->is_error)
  {
      my $Err = $mesg->code;
      my $ErrLDAP = $mesg->error;
      print FICRES "Erreur LDAP lors de la recup des logins LDAP. $Err, $ErrLDAP\n";
      exit 2;
  }
  if ($mesg->count == 0)
  {
      print FICRES "Bizarre : aucune entree LDAP retournee lors de la requete\n";
      exit 2;
  }
  foreach $Entry ($mesg->all_entries)
  {
    my $attrib;
    $attrib = $Entry->get_value($attribLDAP) if defined $attribLDAP;
    $loginLDAP{$attrib}= 1;
  }
  return $mesg->count;
}

# ----------------------------------------------------------------------------
# --            Fonction GetDate                                            --
# retourne la date du jour en format JJ/MM/AAAA                              -
# ----------------------------------------------------------------------------
sub GetDate
{
    my ($dateE, $an);
    my ($sec, $min, $heure, $mjour, $mois, $annee, $sjour, $ajour, $isdst) = localtime(time);
    $mois++;
    $an = 1900 + $annee;
    $mois = "0" . $mois if (length($mois) == 1);
    $mjour = "0" . $mjour if (length($mjour) == 1);
    $heure = "0" . $heure if (length($heure) == 1);
    $min = "0" . $min if (length($min) == 1);
    $dateE = "$mjour/$mois/$an - $heure:$min";
    return $dateE;
}

# ----------------------------------------------------------------------------
# --            Fonction ErrSyntaxe                                         --
# ----------------------------------------------------------------------------
sub ErrSyntaxe
{
    my $mess = shift;
    print "Erreur de syntaxe $mess\n\n";
    print "3 parametres obligatoires :\n";
    print "  . -path : le repertoire a partir duquel les comptes webDav son crees\n";
    print "  . -ldap.serv : le serveur LDAP. si port <> 389, preciser celui-ci en fin de chaine. separateur = ':'\n";
    print "  . -ldap.basedn : le 'base DN' de l'operation\n\n";
    print "et 5 parametres facultatifs\n";
    print "  . -ldap.binddn : le compte de bind, si non anonyme\n";
    print "  . -ldap.bindpasswd : le mot de passe de bind, si non anonyme\n";
    print "  . -ldap.filtre. filtre LDAP a appliquer. Par defaut : (objectclass=supannPerson)\n";
    print "  . -ldap.attrib. Attribut correspondant au compte LDAP. Par defaut : uid\n";
    print "  . -hash. 2 valeurs possibles : normal ou reverse. par defaut : normal\n";
    print "  . -mode. 4 valeurs possibles, par defaut = test\n";
    print "         test : liste les divergences\n";
    print "         create : cree les comptes webDav manquants\n";
    print "         delete : supprime les comptes webDav qui ne sont plus dans LDAP\n";
    print "         sync : synchronise. create + delete\n\n";
    print "Un exemple :\n";
    print "syncDavLdap.pl -ldap.serv ldap.univ.fr:390 -ldap.basedn \"dc=univ,dc=fr\" -mode create\n\n";
    exit 1;
}

# ----------------------------------------------------------------------------
# --            Fonction ArreTimeout                                        --
# --  cette fonction est appelee si le delai de timeout est depasse         -- 
# ----------------------------------------------------------------------------
sub ArretTimeout
{
    my ($LaDate);
    $LaDate = GetDate();
    print FICERR "*********************************************************************\n";
    print FICERR "*   arret anormal de la procedure sur timeout                       *\n";
    print FICERR "*   arret a $LaDate                                     *\n";
    print FICERR "*   $nbTraite entrees parcourues au total                           *\n";
    print FICERR "*********************************************************************\n\n";
    exit 1;
}
