/*
 * $Header$
 * $Revision: 208541 $
 * $Date: 2005-02-25 17:57:55 +0100 $
 *
 * ====================================================================
 *
 * Copyright 2004 The Apache Software Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.apache.slide.webdav.event;

import java.util.EventListener;

import org.apache.slide.event.VetoException;

/**
 * WebDAV event listener that listens to some detailed events.
 */
public interface DetailedWebdavListener extends EventListener {

    /**
     * Called before a single resoure is copied while a WebDAV MOVE request.
     * 
     * <li>event source: the {@link org.apache.slide.webdav.method.MoveMethod}
     * <li>event sourceUri: the old resource URI
     * <li>event destinationUri: the new resource URI
     */
    public void moveBeforeCopy(DetailedWebdavEvent event) throws VetoException;

    /**
     * Called after a single resoure is copied while a WebDAV MOVE request.
     * 
     * <li>event source: the {@link org.apache.slide.webdav.method.MoveMethod}
     * <li>event sourceUri: the old resource URI
     * <li>event destinationUri: the new resource URI
     */
    public void moveAfterCopy(DetailedWebdavEvent event) throws VetoException;

    /**
     * Called before a single resource is deleted while a WebDAV MOVE request.
     * 
     * <li>event source: the {@link org.apache.slide.webdav.method.MoveMethod}
     * <li>event sourceUri: <code>null</code>
     * <li>event destinationUri: the new resource URI
     */
    public void moveBeforeDelete(DetailedWebdavEvent event)
            throws VetoException;

    /**
     * Called after a single resource is deleted while a WebDAV MOVE request.
     * 
     * <li>event source: the {@link org.apache.slide.webdav.method.MoveMethod}
     * <li>event sourceUri: <code>null</code>
     * <li>event destinationUri: the new resource URI
     */
    public void moveAfterDelete(DetailedWebdavEvent event) throws VetoException;

    /**
     * Called before a single resource is copied while a WebDAV COPY request.
     * 
     * <li>event source: the {@link org.apache.slide.webdav.method.CopyMethod}
     * <li>event sourceUri: the old resource URI
     * <li>event destinationUri: the new resource URI
     */
    public void copyBeforeCopy(DetailedWebdavEvent event) throws VetoException;

    /**
     * Called after a single resource is copied while a WebDAV COPY request.
     * 
     * <li>event source: the {@link org.apache.slide.webdav.method.CopyMethod}
     * <li>event sourceUri: the old resource URI
     * <li>event destinationUri: the new resource URI
     */
    public void copyAfterCopy(DetailedWebdavEvent event) throws VetoException;

    /**
     * Called before a single resource is deleted while a WebDAV COPY request.
     * 
     * <li>event source: the {@link org.apache.slide.webdav.method.CopyMethod}
     * <li>event sourceUri: <code>null</code>
     * <li>event destinationUri: the new resource URI
     */
    public void copyBeforeDelete(DetailedWebdavEvent event)
            throws VetoException;

    /**
     * Called after a single resource is deleted while a WebDAV COPY request.
     * 
     * <li>event source: the {@link org.apache.slide.webdav.method.CopyMethod}
     * <li>event sourceUri: <code>null</code>
     * <li>event destinationUri: the new resource URI
     */
    public void copyAfterDelete(DetailedWebdavEvent event) throws VetoException;

    /**
     * Called before a single resource is deleted while a WebDAV DELETE request.
     * 
     * <li>event source: the {@link org.apache.slide.webdav.method.DeleteMethod}
     * <li>event sourceUri: <code>null</code>
     * <li>event destinationUri: the new resource URI
     */
    public void deleteBeforeDelete(DetailedWebdavEvent event)
            throws VetoException;

    /**
     * Called after a single resource is deleted while a WebDAV DELETE request.
     * 
     * <li>event source: the {@link org.apache.slide.webdav.method.DeleteMethod}
     * <li>event sourceUri: <code>null</code>
     * <li>event destinationUri: the new resource URI
     */
    public void deleteAfterDelete(DetailedWebdavEvent event)
            throws VetoException;

    /**
     * Called before a resource is put under version control within a WebDAV 
     * request. This may happen while VERSION-CONTROL, PUT, COPY. 
     *
     * <li>event source: the corresponding method object.
     * <li>event sourceUri: <code>null</code> or the URI of an existing version
     * <li>event destinationUri: the new resource URI
     */
    public void versionControlBefore(DetailedWebdavEvent event)
            throws VetoException;

    /**
     * Called after a resource is put under version control within a WebDAV 
     * request. This can happen while e.g. VERSION-CONTROL, PUT, COPY. 
     *
     * <li>event source: the corresponding method object.
     * <li>event sourceUri: <code>null</code> or the URI of an existing version
     * <li>event destinationUri: the new resource URI
     */
    public void versionControlAfter(DetailedWebdavEvent event)
            throws VetoException;

}