/*
 *
 * ====================================================================
 *
 * Copyright 2004 The Apache Software Foundation 
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.apache.slide.projector.processor.core;

import java.util.Map;

import org.apache.commons.contract.Context;
import org.apache.commons.contract.Processor;
import org.apache.commons.contract.Result;
import org.apache.commons.contract.constraints.BooleanConstraints;
import org.apache.commons.contract.constraints.MapConstraints;
import org.apache.commons.contract.descriptor.ParameterDescriptor;
import org.apache.commons.contract.descriptor.ResultDescriptor;
import org.apache.commons.contract.i18n.ParameterMessage;
import org.apache.slide.projector.constraints.URIConstraints;
import org.apache.slide.projector.constraints.XMLConstraints;
import org.apache.slide.projector.context.ProjectorContext;
import org.apache.slide.projector.engine.Job;
import org.apache.slide.projector.engine.ProcessorManager;
import org.apache.slide.projector.engine.Scheduler;
import org.apache.slide.projector.expression.Expression;
import org.apache.slide.projector.expression.ExpressionFactory;
import org.apache.slide.projector.value.URI;
import org.apache.slide.projector.value.URIValue;
import org.apache.slide.projector.value.XMLValue;

/**
*/
public class Thread implements Processor {
	// parameters
	public final static String PROCESSOR = "processor";
	public final static String REPEAT = "repeat";
	public final static String CONDITION = "condition";
	public final static String REMAINING_CONDITION = "remainingCondition";
	public final static String PARAMETERS = "parameters";
	public final static String PERSISTENT = "persistent";
	
	private final static ParameterDescriptor[] parameterDescriptors = new ParameterDescriptor[] {
			new ParameterDescriptor(PROCESSOR, new ParameterMessage("job/parameter/processor"), new URIConstraints()),
			new ParameterDescriptor(REPEAT, new ParameterMessage("job/parameter/repeat"), new BooleanConstraints(), Boolean.FALSE),
			new ParameterDescriptor(PERSISTENT, new ParameterMessage("job/parameter/persistent"), new BooleanConstraints(), Boolean.TRUE),
			new ParameterDescriptor(CONDITION, new ParameterMessage("job/parameter/condition"), new XMLConstraints()),
			new ParameterDescriptor(REMAINING_CONDITION, new ParameterMessage("job/parameter/remainingCondition"), new XMLConstraints(), null),
			new ParameterDescriptor(PARAMETERS, new ParameterMessage("job/parameter/parameters"), MapConstraints.UNCONSTRAINED)
	};
	
    private final static ResultDescriptor[] resultDescriptors = new ResultDescriptor[] {
            ResultDescriptor.OK
    };
	
	public Result process(Map parameters, Context context) throws Exception {
			Map jobParameters = (Map)parameters.get(PARAMETERS);
			URI jobUri = (URIValue)parameters.get(PROCESSOR);
			XMLValue initialCondition = (XMLValue)parameters.get(CONDITION);
			boolean repeatJob = ((Boolean)parameters.get(REPEAT)).booleanValue();
			Object remainingConditionValue = parameters.get(REMAINING_CONDITION);
			XMLValue remainingCondition = null;
			if ( remainingConditionValue == null ) {
				remainingCondition = initialCondition;
			} else {
				remainingCondition = (XMLValue)remainingConditionValue;
			}
			boolean persistentJob = ((Boolean)parameters.get(PERSISTENT)).booleanValue();
			Expression remainingExpression = ExpressionFactory.create(remainingCondition.getRootElement());
			Expression initialExpression = ExpressionFactory.create(initialCondition.getRootElement());
			Processor jobProcessor = ProcessorManager.getInstance().getProcessor(jobUri);
			ProcessorManager.prepareValues(jobProcessor.getParameterDescriptors(), jobParameters, context);
			String processorId = ((ProjectorContext)context).getProcessId();
			Job job;
			if ( processorId == null ) {
				job = new Job(((ProjectorContext)context).getStep(), jobUri, initialExpression, remainingExpression, jobParameters, repeatJob, persistentJob); 
			} else {
				job = new Job(((ProjectorContext)context).getProcessId()+":"+((ProjectorContext)context).getStep(), jobUri, initialExpression, remainingExpression, jobParameters, repeatJob, persistentJob); 
			}
			Scheduler.getInstance().registerJob(job);
			Scheduler.getInstance().saveJobs();
			return Result.OK;
	}
	
	public ParameterDescriptor[] getParameterDescriptors() {
		return parameterDescriptors;
	}
	
	public ResultDescriptor[] getResultDescriptors() {
		return resultDescriptors;
	}
}