/*
 *
 * ====================================================================
 *
 * Copyright 2004 The Apache Software Foundation 
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.apache.slide.projector.store;

import org.apache.commons.contract.Context;
import org.apache.commons.contract.StoreException;
import org.apache.commons.httpclient.Credentials;
import org.apache.commons.i18n.LocalizedError;
import org.apache.commons.xmlio.out.XMLStringWriter;
import org.apache.slide.projector.Projector;
import org.apache.slide.projector.constraints.ConstraintsManager;
import org.apache.slide.projector.context.ProjectorContext;
import org.apache.slide.projector.repository.RepositoryException;
import org.apache.slide.projector.value.Streamable;
import org.apache.slide.projector.value.Text;
import org.apache.slide.projector.value.URIValue;

public class RepositoryStore extends AbstractStore {
    public final static String ID = "repository";

    private ProjectorContext context;

	public RepositoryStore(ProjectorContext context) {
		this.context = context;
	}
	
	public void put(String key, Object value, Context context) throws StoreException {
	    try {
	        if ( context instanceof ProjectorContext ) {
	            Credentials credentials = ((ProjectorContext)context).getCredentials();
	            if ( value instanceof Streamable) {
	                Projector.getRepository().setResource(new URIValue(key), (Streamable)value, credentials);
	            } else {
	                XMLStringWriter writer = XMLStringWriter.create();
	                writer.writeXMLDeclaration();
	                ConstraintsManager.getInstance().saveValue(value, writer);
	                Projector.getRepository().setResource(new URIValue(key), new Text(writer.toString()), credentials);
	            }
	        } else {
                throw new StoreException(new LocalizedError("credentialsRequired"));
            }
	    } catch (RepositoryException e) { 
            throw new StoreException(e.getLocalizedError());
        }
	}
	
	public Object get(String key, Context context) throws StoreException {
	    try {
	        if ( context instanceof ProjectorContext ) {
	            Credentials credentials = ((ProjectorContext)context).getCredentials();
	            return Projector.getRepository().getResource(new URIValue(key), credentials);
	        } else {
	            throw new StoreException(new LocalizedError("credentialsRequired"));
	        }
	    } catch (RepositoryException e) {
            throw new StoreException(e.getLocalizedError());
	    }
	}

	public void dispose(String key, Context context) throws StoreException {
	    try {
	        if ( context instanceof ProjectorContext ) {
	            Credentials credentials = ((ProjectorContext)context).getCredentials();
	            Projector.getRepository().removeResource(new URIValue(key), credentials);
	        } else {
	            throw new StoreException(new LocalizedError("credentialsRequired"));
	        }
	    } catch (RepositoryException e) {
	        throw new StoreException(e.getLocalizedError());
	    }
	}

    public String getId() {
        return ID;
    }
}