/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/external-tm/src/share/org/apache/slide/store/ExternalTransactionStore.java,v 1.1 2004/09/20 09:10:02 ozeigermann Exp $
 * $Revision: 1.1 $
 * $Date: 2004/09/20 09:10:02 $
 *
 * ====================================================================
 *
 * Copyright 1999-2003 The Apache Software Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.apache.slide.store;

import java.util.Hashtable;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.transaction.TransactionManager;

import org.apache.slide.common.Namespace;
import org.apache.slide.common.ServiceParameterErrorException;
import org.apache.slide.common.ServiceParameterMissingException;
import org.apache.slide.util.logger.Logger;

/**
 * Store that allows for transactional caching of data over JTA transactions
 * provided by an external transaction manager.
 *
 * @version $Revision: 1.1 $
 */
public class ExternalTransactionStore extends ExtendedStore {
  private TransactionManager txManager;

  /**
   * Parameter to specify the JNDI name of the transaction manager.
   */
  public static final String TX_MANAGER_NAME_PARAMETER = "tx-manager-name";

  /**
   * Default value for the transaction manager name parameter.
   */
  public static final String DEFAULT_TX_MANAGER_NAME = "java:/TransactionManager";

  private static final String LOG_CHANNEL = AppServerStore.class.getName();

  /**
   * Default constructor.
   */
  public ExternalTransactionStore() {
  }

  /* (non-Javadoc)
   * @see org.apache.slide.common.Service#setParameters(java.util.Hashtable)
   */
  public void setParameters(Hashtable parameters)
      throws ServiceParameterErrorException, ServiceParameterMissingException {
    super.setParameters(parameters);

    String txManagerName = (String) parameters.get(TX_MANAGER_NAME_PARAMETER);
    if (txManagerName == null) {
      txManagerName = DEFAULT_TX_MANAGER_NAME;
    }

    Logger logger = getLogger();
    logger.log("Setting transaction manager name for store " + getName() +
        " to " + txManagerName, LOG_CHANNEL, Logger.INFO);

    try {
      txManager = (TransactionManager) new InitialContext().lookup(txManagerName);
    }
    catch (NamingException e) {
      logger.log("Could not retrieve transaction manager named: " +
          txManagerName + "from JNDI context! Using default value", e,
          LOG_CHANNEL, Logger.WARNING);
    }
  }

  /* (non-Javadoc)
   * @see org.apache.slide.common.Service#setNamespace(
   *  org.apache.slide.common.Namespace)
   */
  public void setNamespace(Namespace namespace) {
    super.setNamespace(namespace);
    namespace.initTransactionManager(txManager);
  }
}
