/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/taglib/src/common/org/apache/slide/taglib/bean/RevisionBean.java,v 1.3 2004/07/30 06:51:47 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:51:47 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes software developed by the 
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. the names "The Jakarta Project", "Slide", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written 
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.slide.taglib.bean;

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.List;

import org.apache.slide.common.NamespaceAccessToken;
import org.apache.slide.common.SlideException;
import org.apache.slide.common.SlideToken;
import org.apache.slide.content.Content;
import org.apache.slide.content.NodeProperty;
import org.apache.slide.content.NodeRevisionDescriptor;
import org.apache.slide.content.NodeRevisionDescriptors;
import org.apache.slide.content.NodeRevisionNumber;

/**
 * Wraps around a NodeRevisionDescriptor object, providing a Java Bean style 
 * interface.
 *
 * <p>This class is intended to be used in the context of web presentation 
 * layer technologies such as JSP or Velocity.</p>
 * 
 * @author <a href="mailto:cmlenz@apache.org">Christopher Lenz</a>
 * @version $Revision: 1.3 $
 */
public class RevisionBean
    extends AbstractBean {
    
    
    // ----------------------------------------------------- Instance Variables
    
    
    /**
     * The wrapped revision object.
     */
    private NodeRevisionDescriptor revision;
    
    
    /**
     * The revision-descriptors object.
     */
    private NodeRevisionDescriptors revisions;
    
    
    // ----------------------------------------------------------- Construction
    
    
    /**
     * Default constructor.
     */
    public RevisionBean() {
        super();
        
    }
    
    
    /**
     * Constructor.
     *
     * @param nat the namespace access token
     * @param st the slide token
     * @param revision the revision to wrap
     */
    public RevisionBean(NamespaceAccessToken nat, SlideToken st,
                        NodeRevisionDescriptors revisions,
                        NodeRevisionDescriptor revision) {
        super(nat, st);
        
        setNodeRevisionDescriptors(revisions);
        setNodeRevisionDescriptor(revision);
    }
    
    
    // ------------------------------------------------------------- Properties
    
    
    /**
     * Returns the name of the branch the revision belongs to.
     *
     * @return a String containing the branch name
     */
    public String getBranch() {
        
        return revision.getBranchName();
    }
    
    
    /**
     * Returns the value of the <code>DAV:getcontentlanguage</code> property.
     *
     * @return a String containing the property value
     */
    public String getContentLanguage() {
        
        return revision.getContentLanguage();
    }
    
    
    /**
     * Returns the value of the <code>DAV:getcontentlength</code> property.
     *
     * @return the property value as long
     */
    public long getContentLength() {
        
        return revision.getContentLength();
    }
    
    
    /**
     * Returns the value of the <code>DAV:getcontenttype</code> property.
     *
     * @return a String containing the property value
     */
    public String getContentType() {
        
        return revision.getContentType();
    }
    
    
    /**
     * Returns the value of the <code>DAV:creationdate</code> property.
     *
     * @return a Date containing the property value
     */
    public Date getCreationDate() {
        
        return revision.getCreationDateAsDate();
    }
    
    
    /**
     * Returns the value of the <code>DAV:getetag</code> property.
     *
     * @return a String containing the property value
     */
    public String getETag() {
        
        return revision.getETag();
    }
    
    
    /**
     * Returns the value of the <code>DAV:getlastmodified</code> property.
     *
     * @return a Date containing the property value
     */
    public Date getLastModified() {
        
        return revision.getLastModifiedAsDate();
    }
    
    
    /**
     * Returns the value of the <code>DAV:owner</code> property.
     *
     * @return a String containing the property value
     */
    public String getOwner() {
        
        return revision.getOwner();
    }
    
    
    /**
     * Returns the value of the <code>DAV:resourcetype</code> property.
     *
     * @return a String containing the property value
     */
    public String getResourceType() {
        
        return revision.getResourceType();
    }
    
    
    /**
     * Returns the value of the <code>DAV:source</code> property.
     *
     * @return a String containing the property value
     */
    public String getSource() {
        
        return revision.getSource();
    }
    
    /**
     * Returns the list of properties of the revision.
     *
     * @return a List of PropertyBean objects, or an empty List if the
     *         revision doesn't have any properties
     */
    public List getProperties() {
        
        List properties = new ArrayList();
        Enumeration enum = revision.enumerateProperties();
        while (enum.hasMoreElements()) {
            NodeProperty np = (NodeProperty)enum.nextElement();
            properties.add(new PropertyBean(nat, st, np));
        }
        
        return properties;
    }
    
    
    /**
     * Returns the version number of the revision.
     *
     * @return a String containing the revision's version number
     */
    public String getNumber() {
        
        NodeRevisionNumber nrn = revision.getRevisionNumber();
        return (nrn != null) ? nrn.toString() : null;
    }
    
    
    /**
     * Returns the revision this bean wraps around.
     *
     * @return the wrapped NodeRevisionDescriptor
     */
    public NodeRevisionDescriptor getNodeRevisionDescriptor() {
        
        return revision;
    }
    
    
    /**
     * Sets the revision this bean wraps around.
     *
     * @param revision the NodeRevisionDescriptor to wrap
     */
    public void setNodeRevisionDescriptor(NodeRevisionDescriptor revision) {
        
        this.revision = revision;
    }
    
    
    /**
     * Returns the revision-descriptors this revision belongs to.
     *
     * @return the NodeRevisionDescriptors
     */
    public NodeRevisionDescriptors getNodeRevisionDescriptors() {
        
        return revisions;
    }
    
    
    /**
     * Sets the revision-descriptors this revision belongs to.
     *
     * @param revision the NodeRevisionDescriptors
     */
    public void setNodeRevisionDescriptors(NodeRevisionDescriptors revisions) {
        
        this.revisions = revisions;
    }
    
    
    // --------------------------------------------------------- Public Methods
    
    
    /**
     * Returns an InputStream of the content of the revision.
     * 
     * @return the revision content as InputStream
     * 
     * @throws SlideException Exception occurred when trying to access the
     *         resource through Slide.
     * @throws IOException IO exception occurred when trying to stream the 
     *         revision content
     */
    public InputStream getInputStream()
        throws SlideException, IOException {
        
        Content content = nat.getContentHelper();
        return content.retrieve(st, revisions, revision).streamContent();
    }
    
    
    /**
     * Returns a specific property of the node, identified by name and
     * namespace.
     *
     * @param name Name of the property
     * @param namespace Namespace of the property (specify <code>null</code> 
     *        for the default namespace
     * 
     * @return a PropertyBean representing the requested property, or 
     *         <code>null</code> if the property does not exist
     * 
     * @see #getProperties
     */
    public PropertyBean getProperty(String name, String namespace) {
        
        NodeProperty np;
        if ((namespace == null) || namespace.length() == 0) {
            np = revision.getProperty(name);
        }
        else {
            np = revision.getProperty(name, namespace);
        }
        
        if (np != null) {
            return new PropertyBean(nat, st, np);
        }
        
        return null;
    }
    
    
    /**
     * Returns a Reader for the content of the revision.
     * 
     * @return the revision content as Reader
     * 
     * @throws SlideException Exception occurred when trying to access the
     *         resource through Slide.
     * @throws IOException IO exception occurred when trying to read the 
     *         revision content
     */
    public Reader getReader()
        throws SlideException, IOException {
        
        Content content = nat.getContentHelper();
        return content.retrieve(st, revisions, revision).readContent();
    }
    
    
    /**
     * Returns a String representation of the revision, which contains the 
     * version number and the branch name.
     *
     * @return a String representation of the revision
     */
    public String toString() {
        
        return getNumber() + " (" + getBranch() + ")";
    }
    
    
}


