/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/taglib/src/common/org/apache/slide/taglib/util/Iterators.java,v 1.3 2004/07/30 06:51:49 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:51:49 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes software developed by the 
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Slide", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written 
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.slide.taglib.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.slide.taglib.bean.NodeBean;

/**
 * Utility class that serves as a factory for various specialized Iterators.
 *
 * @author <a href="mailto:cmlenz@apache.org">Christopher Lenz</a>
 * @version $Revision: 1.3 $
 */
public class Iterators {
    
    
    // ----------------------------------------------------------- Constructors
    
    
	/**
	 * Constructor hidden to guarantee non-instatiability.
	 */
	private Iterators() {
        
	}
    
    
    // -------------------------------------------------- Public Static Methods
    
    
    /**
     * Returns a special Iterator that iterates over the ancestors on a node,
     * starting at the root node.
     * 
     * @param node the node the ancestors of which should be iterated over
     * @return an Iterator over the ancestors on the node
     */
    public static Iterator ancestorsIterator(NodeBean node) {
        
        List ancestors = new ArrayList();
        while (node.getParent() != null) {
            node = node.getParent();
            ancestors.add(node);
        }
        Collections.reverse(ancestors);
        
        return ancestors.iterator();
    }
    
    
    /**
     * Returns a special Iterator that iterates over the locks on a node.
     * 
     * @param node the node the locks of which should be iterated over
     * @return an Iterator over the locks on the node
     */
    public static Iterator locksIterator(NodeBean node) {
        
        return new LocksIterator(node);
    }
    
    
    /**
     * Returns a special Iterator that iterates over the locks on a node and
     * indirect or direct members of that node, down to a specifiable depth.
     * 
     * @param node the node the locks of which should be iterated over
     * @param depth the recursion depth of the Iterator
     * @return an Iterator over the locks on the node and its members
     */
    public static Iterator locksIterator(NodeBean node, int depth) {
        
        return new LocksIterator(node, depth);
    }
    
    
    /**
     * Returns a special Iterator that iterates over the children of a node.
     * 
     * @param node the node the children of which should be iterated over
     * @return an Iterator over the children of the node
     */
    public static Iterator membersIterator(NodeBean node) {
        
        return new MembersIterator(node);
    }
    
    
    /**
     * Returns a special Iterator that recursively iterates over the direct and
     * indirect members of a node, with a specifiable depth.
     * 
     * @param node the node the children of which should be iterated over
     * @param depth the maximum depth of the recursion
     * @return an Iterator over the children of the node
     */
    public static Iterator membersIterator(NodeBean node, int depth) {
        
        return new MembersIterator(node, depth);
    }
    
    
    /**
     * Returns a special Iterator that recursively iterates over the direct and
     * indirect members of a node, with a specifiable depth, and with optional 
     * rules for excluding and/or including nodes of particular types and/or
     * roles.
     * 
     * @param node the node the children of which should be iterated over
     * @param depth the maximum depth of the recursion
     * @param excludeRoles the set of roles, each a string, to exclude from the
     *        iteration
     * @param includeRoles the set of roles, each a string, to include in the
     *        iteration
     * @param excludeTypes the set of types, each a string, to exclude from the
     *        iteration
     * @param includeTypes the set of types, each a string, to include in the
     *        iteration
     * @return an Iterator over the children of the node
     */
    public static Iterator membersIterator(NodeBean node, int depth,
                                           Set excludeRoles, Set includeRoles,
                                           Set excludeTypes, Set includeTypes) {
        
        return new MembersIterator(node, depth, excludeRoles, includeRoles,
                                   excludeTypes, includeTypes);
    }
    
    
}

