/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/taglib/src/common/org/apache/slide/taglib/util/LocksIterator.java,v 1.3 2004/07/30 06:51:49 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:51:49 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2002 The Apache Software Foundation.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes software developed by the 
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Slide", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written 
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */ 

package org.apache.slide.taglib.util;

import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Stack;

import org.apache.slide.taglib.bean.LockBean;
import org.apache.slide.taglib.bean.NodeBean;

/**
 * A special Iterator that can recursively iterate over the locks on a node and
 * its direct or indirect members.
 *
 * @author <a href="mailto:cmlenz@apache.org">Christopher Lenz</a>
 * @version $Revision: 1.3 $
 */
class LocksIterator
    implements Iterator {
    
    
    // ----------------------------------------------------- Instance Variables
    
    
    /**
     * Recursion depth of the iteration.
     */
    private int depth;
    
    
    /**
     * The node the children of which should be iterated over. 
     */
    private NodeBean node;
    
    
    /**
     * A stack containing iterators over the members. The iterator at the bottom
     * of the stack will contain the iterator over the children of the top node
     * who's members are being iterated over. The size of the stack indicates 
     * the depth of the recursion.
     */
    private Stack childIteratorStack = new Stack();
    
    
    /**
     * Iterator over the locks.
     */    
    private Iterator lockIterator;
    
    
    /**
     * The next lock. 
     */
    private LockBean nextLock;
    
    
    // ----------------------------------------------------------- Constructors
    
    
    /**
     * Constructor.
     */
    public LocksIterator(NodeBean node) {
        this(node, 0);
        
    }
    
    
    /**
     * Constructor.
     */
    public LocksIterator(NodeBean node, int depth) {
        
        if (node == null) {
            throw new NullPointerException();
        }
        childIteratorStack.push(new SingleObjectIterator(node));
        lockIterator = node.getLocks().iterator();
        
        this.depth = depth;
    }
    
    
    // --------------------------------------------------------- Public Methods
    
    
    /**
     * 
     */
    public boolean hasNext() {
        
        if (nextLock == null) {
            if (!childIteratorStack.isEmpty()) {
                nextLock = getNext((Iterator)childIteratorStack.peek());
            } else {
                return false;
            }
        }
        return (nextLock != null);
    }
    
    
    /**
     * 
     */
    public Object next() {
        
        if (!hasNext()) {
            throw new NoSuchElementException();
        }
        LockBean result = nextLock;
        nextLock = null;
        
        return result;
    }
    
    
    /**
     * Not supported.
     */
    public void remove() {
       
       throw new UnsupportedOperationException();
    }
    
    
    // -------------------------------------------------------- Private Methods
    
    
    /**
     * 
     */
    private LockBean getNext(Iterator children) {
       
        if (lockIterator.hasNext()) {
            return (LockBean)lockIterator.next();
        }
        
        while (children.hasNext()) {
            NodeBean child = (NodeBean)children.next();
            lockIterator = child.getLocks().iterator();
            if (childIteratorStack.size() < depth) {
                childIteratorStack.push(child.getChildren().iterator());
            }
            LockBean lock = getNext((Iterator)childIteratorStack.peek());
            if (lock != null) {
                return lock;
            }
        }
        
        if (!childIteratorStack.isEmpty()) {
            childIteratorStack.pop();
            if (!childIteratorStack.isEmpty()) {
                return getNext((Iterator)childIteratorStack.peek());
            }
        }
        
        return null;
    }
    
    
}

