/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/taglib/src/common/org/apache/slide/taglib/util/MembersIterator.java,v 1.3 2004/07/30 06:51:49 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:51:49 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2002 The Apache Software Foundation.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes software developed by the 
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Slide", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written 
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */ 

package org.apache.slide.taglib.util;

import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.Stack;

import org.apache.slide.taglib.bean.NodeBean;

/**
 * A special iterator that can recursively iterate over direct and indirect 
 * members of a node down to a specified depth. In addition, members can be 
 * included or excluded based on their role and/or type.
 *
 * @author <a href="mailto:cmlenz@apache.org">Christopher Lenz</a>
 * @version $Revision: 1.3 $
 */
class MembersIterator
    implements Iterator {
    
    
    // ----------------------------------------------------- Instance Variables
    
    
    /**
     * Recursion depth of the iteration.
     */
    private int depth;
    
    
    /**
     * The node the children of which should be iterated over. 
     */
    private NodeBean node;
    
    
    /**
     * The set of node roles, each of type String, to exclude.
     */
    private Set excludeRoles = Collections.EMPTY_SET;
    
    
    /**
     * The set of node roles, each of type String, to include.
     */
    private Set includeRoles = Collections.EMPTY_SET;
    
    
    /**
     * The set of node types, each of type String, to include.
     */
    private Set includeTypes = Collections.EMPTY_SET;
    
    
    /**
     * The set of node types, each of type String, to exclude.
     */
    private Set excludeTypes = Collections.EMPTY_SET;
    
    
    /**
     * A stack containing iterators over the members. The iterator at the bottom
     * of the stack will contain the iterator over the children of the top node
     * who's members are being iterated over. The size of the stack indicates 
     * the depth of the recursion.
     */
    private Stack childIteratorStack = new Stack();
    
    
    /**
     * The next member. 
     */
    private NodeBean nextNode;
    
    
    // ----------------------------------------------------------- Constructors
    
    
    /**
     * Constructor.
     */
    public MembersIterator(NodeBean node) {
        this(node, 0);
        
    }
    
    
    /**
     * Constructor.
     */
    public MembersIterator(NodeBean node, int depth) {
        this(node, depth, null, null, null, null);
        
    }
    
    
    /**
     * Constructor.
     */
    public MembersIterator(NodeBean node, int depth,
                           Set excludeRoles, Set includeRoles,
                           Set excludeTypes, Set includeTypes) {
        
        if (node == null) {
            throw new NullPointerException();
        }
        childIteratorStack.push(node.getChildren().iterator());
        
        if (excludeRoles != null) {
            this.excludeRoles = excludeRoles;
        }
        if (includeRoles != null) {
            this.includeRoles = includeRoles;
        }
        if (excludeTypes != null) {
            this.excludeTypes = excludeTypes;
        }
        if (includeTypes != null) {
           this.includeTypes = includeTypes;
        }
        this.depth = depth;
    }
    
    
    // --------------------------------------------------------- Public Methods
    
    
    /**
     * 
     */
    public boolean hasNext() {
        
        if (nextNode == null) {
            if (!childIteratorStack.isEmpty()) {
                nextNode = getNext((Iterator)childIteratorStack.peek());
            } else {
                return false;
            }
        }
        
        return (nextNode != null);
    }
    
    
    /**
     * 
     */
    public Object next() {
        
        if (!hasNext()) {
            throw new NoSuchElementException();
        }
        NodeBean result = nextNode;
        nextNode = null;
        
        return result;
    }
    
    
    /**
     * Not supported.
     */
    public void remove() {
       
       throw new UnsupportedOperationException();
    }
    
    
    // -------------------------------------------------------- Private Methods
    
    
    /**
     * 
     */
    private NodeBean getNext(Iterator children) {
        
        NodeBean node = null;
        while (children.hasNext()) {
            NodeBean child = (NodeBean)children.next();
            if (childIteratorStack.size() < depth) {
                childIteratorStack.push(child.getChildren().iterator());
            }
            if (!isExcluded(child) && isIncluded(child)) {
                node = child;
            } else if (childIteratorStack.size() < (depth + 1)) {
                node = getNext((Iterator)childIteratorStack.peek());
            }
            if (node != null) {
                return node;
            }
        }
        
        if (!childIteratorStack.isEmpty()) {
            childIteratorStack.pop();
            if (!childIteratorStack.isEmpty()) {
                return getNext((Iterator)childIteratorStack.peek());
            }
        }
        
        return null;
    }
    
    
    /**
     * 
     */
    private boolean isExcluded(NodeBean node) {
        
        // see if the node should be included or excluded based on its type
        if (excludeTypes.contains(node.getType())) {
            return true;
        }
        
        // see if the node should be included or excluded based on its
        // roles
        if (containsAny(excludeRoles, node.getRoles())) {
            return true;
        }
        
        return false;
    }
    
    
    /**
     * 
     */
    private boolean isIncluded(NodeBean node) {
        
        // see if the node should be included or excluded based on its type
        if (!includeTypes.isEmpty() &&
            !includeTypes.contains(node.getType())) {
            return false;
        }
        
        // see if the node should be included or excluded based on its
        // roles
        if (!includeRoles.isEmpty() &&
            !containsAny(includeRoles, node.getRoles())) {
            return false;
        }
        
        return true;
    }
    
    
    /**
     * 
     */
    private boolean containsAny(Set set, Collection of) {
        
        for (Iterator i = of.iterator(); i.hasNext(); ) {
            if (set.contains(i.next())) {
                return true;
            }
        }
        
        return false;
    }
    
    
}

