/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/taglib/src/jstl/org/apache/slide/taglib/tag/jstl/ForEachMemberTag.java,v 1.3 2004/07/30 06:51:49 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:51:49 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2002 The Apache Software Foundation.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes software developed by the 
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Slide", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written 
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */ 

package org.apache.slide.taglib.tag.jstl;

import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.StringTokenizer;

import javax.servlet.jsp.JspException;
import javax.servlet.jsp.JspTagException;
import javax.servlet.jsp.jstl.core.LoopTagStatus;
import javax.servlet.jsp.jstl.core.LoopTagSupport;

import org.apache.slide.taglib.bean.NodeBean;
import org.apache.slide.taglib.util.Iterators;

/**
 * Tag class that extends the JSTL loop tag to provide iteration over the 
 * members of a specific node up to a configurable depth, including and/or 
 * excluding specific types and/or roles of objects.
 * 
 * @author <a href="mailto:cmlenz@apache.org">Christopher Lenz</a>
 * @version $Revision: 1.3 $
 */
public class ForEachMemberTag
    extends LoopTagSupport {
    
    
    // ---------------------------------------------------------- Inner Classes
    
    
    /**
     * This class extends LoopTagStatus to provide an additional 'depth'
     * property, which returns the current depth of the iteration.
     */
    public static class Status
        implements LoopTagStatus {
        
        public LoopTagStatus status;
        
        public Integer getBegin() {
            return status.getBegin();
        }
        
        public int getCount() {
            return status.getCount();
        }
        
        public Object getCurrent() {
            return status.getCurrent();
        }
        
        public Integer getEnd() {
            return status.getEnd();
        }
        
        public int getIndex() {
            return status.getIndex();
        }
        
        public Integer getStep() {
            return status.getStep();
        }
        
        public boolean isFirst() {
            return status.isFirst();
        }
        
        public boolean isLast() {
            return status.isLast();
        }
        
        public Integer getDepth() {
            NodeBean current = (NodeBean)getCurrent();
            String uri = current.getUri();
            int index = uri.indexOf('/');
            int depth = 0;
            while (index >= 0) {
                uri = uri.substring(index + 1);
                index = uri.indexOf('/');
                depth++;
            }
            
            return new Integer(depth);
        }
        
    }
    
    
    // ----------------------------------------------------- Instance Variables
    
    
    /**
     * Value of the 'depth' attribute.
     */
    protected String depth;
    
    
    /**
     * Value of the 'includeRoles' attribute.
     */
    protected String includeRoles;
    
    
    /**
     * Value of the 'excludeRoles' attribute.
     */
    protected String excludeRoles;
    
    
    /**
     * Value of the 'includeTypes' attribute.
     */
    protected String includeTypes;
    
    
    /**
     * Value of the 'excludeTypes' attribute.
     */
    protected String excludeTypes;
    
    
    /**
     * Value of the 'node' attribute, which should be a JSP-EL expression 
     * referencing the NodeBean to use.
     */
    protected String node;
    
    
    /**
     * Iterator over the list of members.
     */
    protected Iterator iterator;
    
    
    /**
     * Instance of our custom LoopTagStatus.
     */
    private Status status;
    
    
    // ----------------------------------------------------------- Constructors
    
    
    /**
     * Default constructor.
     */
    public ForEachMemberTag() {
        
        init();
    }
    
    
    // ------------------------------------------ LoopTagSupport Implementation
    
    
    /**
     * Implemented to assemble the list of members and prepare the iterator.
     * 
     * @throws JspTagException if the tag is incorrectly nested or an attribute
     *         value is invalid
     */
    protected void prepare()
        throws JspTagException {
        
        NodeBean nodeBean = null;
        try {
            nodeBean = (NodeBean)
                JstlTagUtils.evaluateOptionalAttribute("forEachMember", "node",
                                                       node, NodeBean.class,
                                                       this, pageContext);
        } catch (JspException e) {
            // ignore
        }
        if (nodeBean == null) {
            nodeBean = JstlTagUtils.findNode(this, pageContext);
            if (nodeBean == null) {
                throw new JspTagException("The 'forEachMember' tag must be " +
                    "nested inside a 'node' tag or an iteration over nodes, " +
                    "or the 'node' attribute must be set.");
            }
        }
        
        int depthValue = 1;
        if (depth != null) {
            try {
                depthValue = Integer.parseInt(depth);
            } catch (NumberFormatException e) {
                if (depth.equalsIgnoreCase("infinity")) {
                    depthValue = Integer.MAX_VALUE;
                }
                else {
                    throw new JspTagException("The attribute 'depth' of the " +
                        "'forEachMember' tag must contain an integer value.");
                }
            }
        }
        
        this.iterator = Iterators.membersIterator(nodeBean, depthValue,
                                                  tokenizeString(excludeRoles),
                                                  tokenizeString(includeRoles),
                                                  tokenizeString(excludeTypes),
                                                  tokenizeString(includeTypes));
    }
    
    
    /**
     * Implemented to indicate whether there is another member in the list.
     * 
     * @return <tt>true</tt> if the members list has another element,
     *         <tt>false</tt> otherwise
     * @throws JspTagException (not thrown)
     */
    protected boolean hasNext()
        throws JspTagException {
        
        return iterator.hasNext();
    }
    
    
    /**
     * Implemented to return the next member in the list, if available.
     * 
     * @return the next member (of type NodeBean) in the list
     * @throws JspTagException (not thrown)
     */
    protected Object next()
        throws JspTagException {
        
        return iterator.next();
    }
    
    
    /**
     * Called by the JSP Engine when closing this tag.
     *
     * @throws JspException not thrown
     */
    public int doEndTag()
        throws JspException {
        
        init();
        
        return super.doEndTag();
    }
    
    
    /**
     * Releases any resources we may have (or inherit).
     */
    public void release() {
        super.release();
        
        init();
    }
    
    
    // ------------------------------------------------------------- Properties
    
    
    /**
     * Sets the 'depth' attribute.
     *
     * @param depth the attribute value
     */
    public void setDepth(String depth) {
        
        this.depth = depth;
    }
    
    
    /**
     * Sets the 'excludeTypes' attribute.
     *
     * @param excludeTypes the attribute value
     */
    public void setExcludeTypes(String excludeTypes) {
        
        this.excludeTypes = excludeTypes;
    }
    
    
    /**
     * Sets the 'includeTypes' attribute.
     *
     * @param includeTypes the attribute value
     */
    public void setIncludeTypes(String includeTypes) {
        
        this.includeTypes = includeTypes;
    }
    
    
    /**
     * Sets the 'excludeRoles' attribute.
     *
     * @param excludeRoles the attribute value
     */
    public void setExcludeRoles(String excludeRoles) {
        
        this.excludeRoles = excludeRoles;
    }
    
    
    /**
     * Sets the 'includeRoles' attribute.
     *
     * @param includeRoles the attribute value
     */
    public void setIncludeRoles(String includeRoles) {
        
        this.includeRoles = includeRoles;
    }
    
    
    /**
     * Sets the 'node' attribute.
     *
     * @param node the attribute value
     */
    public void setNode(String node) {
        
        this.node = node;
    }
    
    
    // -------------------------------------------------------- Private Methods
    
    
    /**
     * Initialize state for tag reuse.
     */
    private void init() {
        
        depth = null;
        includeRoles = null;
        excludeRoles = null;
        includeTypes = null;
        excludeTypes = null;
        node = null;
        iterator = null;
    }
    
    
    /**
     * Splits the given string into its tokens, separated by commata, and 
     * returns the tokens in a immutable Set.
     */
    private static Set tokenizeString(String values) {
        
        Set set = new HashSet();
        if (values != null) {
            StringTokenizer st = new StringTokenizer(values, ",");
            while (st.hasMoreElements()) {
                set.add(((String)st.nextElement()).trim());
            }
        }
        
        return Collections.unmodifiableSet(set);
    }
    
    
	/**
	 * Overridden to provide our custom, depth-enabled status.
     * 
     * @return an instance of ForEachMemberTag.Status
	 */
	public LoopTagStatus getLoopStatus() {
        
		if (status == null) {
            status = new Status();
        }
        status.status = super.getLoopStatus();
        return status;
	}
    
    
}

