/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/tamino/src/store/org/apache/slide/store/tamino/common/ICache.java,v 1.3 2004/07/30 06:51:52 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:51:52 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.slide.store.tamino.common;

import java.util.Collection;

/**
 * Cache for descriptors objects.
 *
 * @author    peter.nevermann@softwareag.com
 *
 * @version   $Revision: 1.3 $
 *
 * @see       IDescriptors
 *
 */
public interface ICache {

    /**
     * Get the descriptors objects identified by given URI.
     *
     * @pre        uuri != null
     * @post
     *
     * @param      uuri   the URI
     *
     * @return     the descriptors object identified by uri; null, if not found.
     */
    IDescriptors get( String uuri );

    /**
     * Get all descriptors objects contained in cache.
     *
     * @pre
     * @post
     *
     * @return     collection of all descriptors objects contained in cache;
     *             empty collection if none.
     */
    Collection getAll();

    /**
     * Check whether the descriptors objects identified by given URI is contained in cache.
     *
     * @pre        uuri != null
     * @post
     *
     * @param      uuri   the URI
     *
     * @return     true, if descriptors object identified by uri is contained;
     *             false, otherwise.
     */
    boolean contains( String uuri );

    /**
     * Add the given descriptors object to cache. If the cache previously contained
     * an entry for the URI in question, this will be displaced.
     *
     * @pre        d != null
     * @pre        d.getUri() != null
     * @post       d == get( d.getUri() )
     *
     * @param      d   the descriptors object
     */
    IDescriptors add( IDescriptors d );

    /**
     * Add all descriptors objects in the given collection to cache. If the cache previously
     * contained entries for some of the URIs in question, these will be displaced.
     *
     * @pre        pre of add for each d in dl
     * @post       post of add for each d in dl
     *
     * @param      dl   the collection of descriptors objects
     */
    void addAll( Collection dl );

    /**
     * Remove the descriptors objects identified by given URI.
     *
     * @pre        uuri != null
     * @post       get( uuri ) == null
     *
     * @param      uuri   the URI
     *
     * @return     the (removed) descriptors object identified by uri; null, if not found.
     */
    IDescriptors remove( String uuri );

    /**
     * Remove all descriptors objects contained in cache.
     *
     * @pre
     * @post       isEmpty() == true
     */
    void removeAll();

}
